package com.xebialabs.xlrelease.plugins.dashboard.service

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.plugins.dashboard.repository.SqlDashboardRepository
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.SecuredCis
import com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.security.{PermissionChecker, XLReleasePermissions}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Service
class DashboardSecurity @Autowired()(val permissionChecker: PermissionChecker,
                                     val roleService: RoleService,
                                     val securedCis: SecuredCis,
                                     val permissionEnforcer: PermissionEnforcer,
                                     val permissionEditor: PermissionEditor,
                                     val dashboardRepository: SqlDashboardRepository) {

  def checkCreate(parentId: String): Unit = Option(parentId).getOrElse(GLOBAL_SECURITY_ALIAS) match {
    case releaseId if isReleaseId(releaseId) => permissionChecker.checkEdit(releaseId)
    case folderId if isFolderId(folderId) => permissionChecker.check(EDIT_DASHBOARD, parentId)
    case _ => permissionChecker.check(CREATE_DASHBOARD)
  }

  def checkEdit(dashboardId: String): Unit = if (isReleaseId(getParentId(dashboardId))) {
    permissionChecker.checkEdit(getParentId(dashboardId))
  } else {
    checkDashboardPermission(EDIT_DASHBOARD, dashboardId)
  }

  def checkView(dashboardId: String): Unit = if (isReleaseId(getParentId(dashboardId))) {
    permissionChecker.checkView(getParentId(dashboardId))
  } else {
    checkDashboardPermission(VIEW_DASHBOARD, dashboardId)
  }

  def getPermissions(dashboard: Dashboard): mutable.Map[Role, mutable.Set[Permission]] = {
    getEffectiveSecurityUid(dashboard) match {
      case Some(securityUid) =>
        val roles = roleService.readRoleAssignments.asScala ++ roleService.readRoleAssignments(securityUid).asScala
        val permissionsByRole = permissionEditor.readPermissions(securityUid).asScala
        permissionsByRole.map { case (role, permissions) =>
          (roles.find(_.getId == role.getId).getOrElse(role), permissions.asScala) // <- :(
        }
      case _ =>
        mutable.Map[Role, mutable.Set[Permission]]()
    }
  }

  def savePermissions(dashboard: Dashboard): Unit = {
    // we save permissions only for global dashboards, folder ones have permissions defined on folders,
    // and release ones do not have any dedicated permissions
    if (dashboard.isGlobalDashboard) {
      dashboardRepository.getSecurityUid(dashboard.getId) match {
        case Some(securityUid) =>
          val roles = getAvailableRoles(dashboard)
          val permissions = permissionEditor.readPermissions(securityUid)
          setPermission(dashboard.getRoleViewers.asScala, XLReleasePermissions.VIEW_DASHBOARD, roles, permissions)
          setPermission(dashboard.getRoleEditors.asScala, XLReleasePermissions.EDIT_DASHBOARD, roles, permissions)
          permissionEditor.editPermissions(securityUid, permissions)
        case _ =>
      }
    }
  }

  def clearPermissions(dashboardId: String): Unit = {
    dashboardRepository.getSecurityUid(dashboardId) match {
      case Some(securityUid) =>
        permissionEditor.editPermissions(securityUid, new util.HashMap[Role, util.Set[Permission]]())
      case _ =>
    }
  }

  def validate(dashboard: Dashboard): Unit = {
    val allRoles = getAvailableRoles(dashboard)
    validateRoleNames(dashboard.getRoleViewers.asScala, allRoles)
    validateRoleNames(dashboard.getRoleEditors.asScala, allRoles)
  }

  private def validateRoleNames(roleNames: mutable.Buffer[String], allRoles: Seq[Role]): Unit = {
    roleNames
      .foreach(roleName => allRoles.find(_.getName == roleName).getOrElse(throw new NotFoundException(s"The role '$roleName' does not exist")))
  }

  private def getAvailableRoles(dashboard: Dashboard): Seq[Role] = {
    Option(dashboard.getParentId)
      .map(parentId => roleService.getRoles(securedCis.getEffectiveSecuredCi(parentId).getSecurityUid))
      .getOrElse(roleService.getRoles).asScala.toSeq
  }

  private def setPermission(roleNames: mutable.Buffer[String], permission: Permission, allRoles: Seq[Role],
                            permissions: util.Map[Role, util.Set[Permission]]): Unit = {

    permissions.asScala.foreach { case (_, rolePermissions) =>
      if (rolePermissions != null) {
        rolePermissions.remove(permission)
      }
    }

    roleNames
      .flatMap(roleName => allRoles.find(_.getName == roleName))
      .foreach { role =>
        val rolePermissions = permissions.getOrDefault(role, new util.HashSet[Permission]())
        rolePermissions.add(permission)
        permissions.put(role, rolePermissions)
      }
  }

  private def checkDashboardPermission(permission: Permission, dashboardId: String): Unit = {
    val dashboard = dashboardRepository.findDashboardById(dashboardId)
    if (dashboard.getOwner != getAuthenticatedUserName) {
      val securityUid = getEffectiveSecurityUid(dashboard)
      if (securityUid.isDefined && !permissionEnforcer.hasLoggedInUserPermission(permission, securityUid.get)) {
        throw PermissionDeniedException.forPermission(permission, dashboardId)
      }
    }
  }

  private def getEffectiveSecurityUid(dashboard: Dashboard): Option[String] = {
    if (dashboard.isFolderDashboard) {
      Option(securedCis.getEffectiveSecuredCi(dashboard.getParentId).getSecurityUid)
    } else if (dashboard.isGlobalDashboard) {
      dashboardRepository.getSecurityUid(dashboard.getId)
    } else {
      // release dashboards currently not saved in xlr_dashboards so no effective security to fetch
      None
    }
  }
}
