package com.xebialabs.xlrelease.status.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.ConfigurationApi
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.domain.environments.LiveDeploymentConfig
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.service.ConfigurationAutoconfigService
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.views.AutoconfigResponse
import com.xebialabs.xlrelease.webhooks.authentication.NoAuthentication
import com.xebialabs.xlrelease.webhooks.consumers.StatusWebhookEventSource
import com.xebialabs.xlrelease.webhooks.endpoint.WebhookEndpoint.PostWebhookEndpoint
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

trait EndpointSetupDeploymentStatusService {
  def setupApplicationStatusWebhook(folderId: String, serverId: String, configId: String): WebhookBasicSetup
}

@Service
class SetupApplicationStatusServiceImpl(
                                         configurationApi: ConfigurationApi,
                                         configurationAutoconfigService: ConfigurationAutoconfigService,
                                         configurationRepository: ConfigurationRepository,
                                       ) extends EndpointSetupDeploymentStatusService {

  @IsTransactional
  override def setupApplicationStatusWebhook(folderId: String, serverId: String, configId: String): WebhookBasicSetup = {
    val config = configurationRepository.read[LiveDeploymentConfig](configId)
    val server = configurationRepository.read[StatusHttpConnection](serverId)
    val webhookEndpoint: Configuration = createHttpEndpointForWebhook(folderId, server.getTitle)
    val statusWebhookSource: Configuration = createStatusWebhookSource(
      folderId,
      server,
      webhookEndpoint,
      config
    )

    val autoconfigResponse: AutoconfigResponse = configurationAutoconfigService.autoconfigure(statusWebhookSource)
    WebhookBasicSetup(
      autoconfigResponse.getSuccess,
      autoconfigResponse.getMessage,
      autoconfigResponse.getDetails,
      autoconfigResponse.getUrl,
      statusWebhookSource.getId
    )
  }

  private def createHttpEndpointForWebhook(folderId: String, title: String): Configuration = {

    val webhookEndpoint: PostWebhookEndpoint = Type.valueOf(classOf[PostWebhookEndpoint])
      .getDescriptor.newInstance("post-webhook-endpoint")

    webhookEndpoint.setTitle(title)
    webhookEndpoint.setFolderId(folderId)
    webhookEndpoint.setProperty("path", s"path-${UUID.randomUUID}")
    webhookEndpoint.setProperty("authentication", new NoAuthentication)

    configurationApi.addConfiguration(webhookEndpoint)
  }

  private def createStatusWebhookSource(folderId: String,
                                        server: StatusHttpConnection,
                                        eventSource: Configuration,
                                        config: LiveDeploymentConfig): Configuration = {
    val statusWebhookEventSourceType: String = server.getStatusWebhookEventSourceName()

    val statusWebhookEventSource: StatusWebhookEventSource = Type.valueOf(statusWebhookEventSourceType)
      .getDescriptor.newInstance("status-event-webhook")
    statusWebhookEventSource.setTitle(server.getTitle)
    statusWebhookEventSource.setFolderId(folderId)
    statusWebhookEventSource.setProperty("eventSource", eventSource)
    statusWebhookEventSource.setSourceServer(server)
    statusWebhookEventSource.setLiveDeploymentConfigs(Set(config).asJava)

    configurationApi.addConfiguration(statusWebhookEventSource)
  }
}

case class WebhookBasicSetup(success: Boolean,
                             message: String,
                             details: String,
                             url: String,
                             webhookSourceId: String)
