import React, { ChangeEvent } from 'react';
import { ReactWrapper } from 'enzyme';
import { CssGrid, DotTable, DotTablePagination } from '@digital-ai/dot-components';
import { mockIntersectionObserver, mockResizeObserver, mountComponentWithStore } from '@xlr-ui/tests/unit/testing-utils';
import { ExternalDeploymentsTable } from './external-deployments-table.component';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialPage, initialState } from '../ducks/external-deployments.reducer';
import {
    mockApplicationsMap,
    mockEnvironmentsMap,
    mockLiveDeploymentData,
    mockServersMap,
    mockStatusWebhookEventSource,
    pluginInfoDeploy,
} from '../__mocks__/external-deployments.mocks';
import { Search } from '@xlr-ui/app/react/components/search/search';
import { STATUS_HTTP_CONNECTION } from '../constants';
import { ExternalDeploymentCard } from './external-deployment-card.component';

const { getFilters, loadExternalDeployments, setCondition, storeFilters, setPage, subscribeToSseStream, unsubscribeFromSseStream, getAvailablePlugins } =
    folderExternalDeployments.actions;
const dispatch = jest.fn();
describe('ExternalDeploymentsTable component', () => {
    let wrapper: ReactWrapper;
    const onViewChange = jest.fn();
    const defaultProps = {
        folder: { id: 'folderId', title: 'Folder' },
        onViewChange,
    };

    const mountComponent = (state: FolderExternalDeploymentsState = initialState) => {
        wrapper = mountComponentWithStore(<ExternalDeploymentsTable {...defaultProps} />, dispatch, {
            folderExternalDeployments: state,
        });
        expect(dispatch).toBeCalledWith(getFilters('folderId'));
        expect(dispatch).toBeCalledWith(loadExternalDeployments());
        expect(dispatch).toBeCalledWith(subscribeToSseStream());
        expect(dispatch).toBeCalledWith(storeFilters());
        expect(dispatch).toBeCalledWith(getAvailablePlugins(STATUS_HTTP_CONNECTION));
    };
    beforeEach(() => {
        mockIntersectionObserver();
        mockResizeObserver();
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
        expect(dispatch).toBeCalledWith(unsubscribeFromSseStream());
    });
    const searchTable = (selector: string) => wrapper.find(`.external-deployments-table ${selector}`);

    it('should trigger filter search', () => {
        mountComponent({
            ...initialState,
            count: 1,
            statusWebhookEventSources: [mockStatusWebhookEventSource],
        });
        jest.useFakeTimers();
        jest.spyOn(global, 'clearTimeout');
        jest.spyOn(global, 'setTimeout');
        const e1 = { target: { value: 'filter' } };
        wrapper.find(Search).invoke('onFilterChange')?.(e1 as unknown as ChangeEvent<HTMLInputElement>);
        wrapper.update();
        expect(setTimeout).toHaveBeenCalledTimes(1);
        expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), 500);
        jest.runAllTimers();
        expect(dispatch).toHaveBeenCalledWith(setCondition('filter'));
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
        const e2 = { target: { value: 'filter 2' } };
        wrapper.find(Search).invoke('onFilterChange')?.(e2 as unknown as ChangeEvent<HTMLInputElement>);
        wrapper.update();
        expect(clearTimeout).toHaveBeenCalledTimes(1);
        jest.runAllTimers();
        expect(dispatch).toHaveBeenCalledWith(setCondition('filter 2'));
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
    });

    it('should not show one row and trigger page change', () => {
        mountComponent({
            ...initialState,
            applications: mockApplicationsMap,
            connectionServers: mockServersMap,
            count: 1,
            environments: mockEnvironmentsMap,
            isTableView: true,
            liveDeployments: [mockLiveDeploymentData],
            statusWebhookEventSources: [mockStatusWebhookEventSource],
        });
        expect(searchTable('.dot-typography').at(0).text()).toStrictEqual('Live deployments status (1)');
        expect(searchTable('.external-deployment-card').exists()).toBeFalsy();
        wrapper.find(DotTable).invoke('onUpdateData')?.('asc', 'column', 1, 10);
        expect(dispatch).toHaveBeenCalledWith(
            setPage({
                ...initialPage,
                folderId: defaultProps.folder.id,
                order: 'asc',
                orderBy: 'column',
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
    });

    it('should show one row and change page and results per page', () => {
        mountComponent({
            ...initialState,
            applications: mockApplicationsMap,
            connectionServers: mockServersMap,
            count: 1,
            environments: mockEnvironmentsMap,
            isTableView: true,
            liveDeployments: [mockLiveDeploymentData],
            validServerCards: [pluginInfoDeploy],
            statusWebhookEventSources: [mockStatusWebhookEventSource],
        });
        expect(searchTable('.dot-typography').at(0).text()).toStrictEqual('Live deployments status (1)');
        expect(searchTable('.external-deployment-card').exists()).toBeTruthy();
        wrapper.find(DotTablePagination).invoke('onPageChange')?.(2);
        expect(dispatch).toHaveBeenCalledWith(
            setPage({
                ...initialPage,
                folderId: defaultProps.folder.id,
                page: 2,
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
        wrapper.find(DotTablePagination).invoke('onRowsPerPageChange')?.({ target: { value: 10 } } as never);
        expect(dispatch).toHaveBeenCalledWith(
            setPage({
                ...initialPage,
                folderId: defaultProps.folder.id,
                resultsPerPage: 10,
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(loadExternalDeployments());
    });

    it('should show connection servers error', () => {
        mountComponent({
            ...initialState,
            count: 1,
            connectionErrors: ['Connection error 1', 'Connection error 2'],
            statusWebhookEventSources: [mockStatusWebhookEventSource],
        });
        expect(searchTable('.external-deployments-connection-errors').exists()).toBeTruthy();
        expect(searchTable('.external-deployments-connection-errors .dot-typography').at(0).text()).toStrictEqual('2 disconnected servers');
    });

    it('should show card view when table view is set to false', () => {
        mountComponent({
            ...initialState,
            applications: mockApplicationsMap,
            connectionServers: mockServersMap,
            count: 1,
            environments: mockEnvironmentsMap,
            liveDeployments: [mockLiveDeploymentData],
            validServerCards: [pluginInfoDeploy],
            statusWebhookEventSources: [mockStatusWebhookEventSource],
        });

        const externalDeploymentsCards = wrapper.find(`.external-deployments-cards`);
        expect(externalDeploymentsCards.exists()).toBeTruthy();
        const cssGrid = externalDeploymentsCards.find(CssGrid);
        expect(cssGrid).toExist();
        const cards = wrapper.find(ExternalDeploymentCard);
        expect(cards).toHaveLength(1);
    });
});
