package com.xebialabs.xlrelease.status.webhook.events

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.{DeploymentServerEvent, Event}
import com.xebialabs.xlplatform.webhooks.events.handlers.{EventProcessorHandler, EventPublisher}
import com.xebialabs.xlrelease.domain.environments.LiveDeployment
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.status.service.LiveDeploymentService
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.webhooks.consumers.StatusWebhookEventSource
import com.xebialabs.xlrelease.webhooks.service.EventSourceOperationService
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.util.{Failure, Success, Try}

@Component
class StatusWebhookConsumerHandler(val eventPublisher: EventPublisher,
                                   configurationRepository: ConfigurationRepository,
                                   externalDeploymentService: LiveDeploymentService,
                                   operationService: EventSourceOperationService)
  extends EventProcessorHandler[Event, Event, StatusWebhookEventSource]
    with Logging {


  override def consumerConfigType: Type = Type.valueOf(classOf[StatusWebhookEventSource])

  override def filter(config: StatusWebhookEventSource, event: Event): Boolean = operationService.executeFilterScript(config, event)

  override def map(config: StatusWebhookEventSource, event: Event): Event = {
    if (config.consumerEnabled && event.isInstanceOf[HttpRequestEvent]) {
      handleStatusEvent(config, event.asInstanceOf[HttpRequestEvent]).orNull
    } else {
      null
    }
  }

  private def handleStatusEvent(config: StatusWebhookEventSource, event: HttpRequestEvent): Option[Event] = {
    logger.trace(s"receivedEvent from ${event.sourceId} [${event.content}]")
    val sourceServerId = config.sourceServer.getId
    val connection = configurationRepository.read[StatusHttpConnection](sourceServerId)
    val externalDeploymentEvent = Try(operationService.executeMapScript(config, event)) match {
      case Success(event: DeploymentServerEvent) =>
        Some(event)
      case Success(errorObject) =>
        val errorType = Option(errorObject).map(_.getClass.getName).getOrElse("null")
        error(s"Illegal response in ${connection.getId} transformerScript response. " +
          s"Returned type was $errorType, expected state or package objects.")
        None
      case Failure(exception) =>
        error(s"Exception thrown from transformerScript for ${connection.getType.toString} endpoint.", exception)
        None
    }
    externalDeploymentEvent.foreach(processedEvent => {
      Try(externalDeploymentService.saveAndNotify(config.getFolderId, processedEvent)) match {
        case Failure(e) =>
          warn(s"Failed to process event, processing was skipped and didn't propagate to database or displayed on the Live Deployment screen: ${e.getMessage}")
        case Success(_) =>
      }
    })
    externalDeploymentEvent
  }
}

sealed class LiveDeploymentEvent(val deploymentId: String)

final case class CreateLiveDeploymentEvent() extends LiveDeploymentEvent(null)

final case class DeleteLiveDeploymentEvent(override val deploymentId: String) extends LiveDeploymentEvent(deploymentId)

sealed case class UpdateLiveDeploymentEvent(override val deploymentId: String, data: LiveDeployment) extends LiveDeploymentEvent(deploymentId)
