import { DeploymentServerDialogErrorAction, DeploymentServerSearch, folderDeploymentServers, initialState } from './deployment-server.reducer';
import { mockArgoConnectionServerMetadata, mockArgoDeploymentServer, mockLiveDeploymentConfigData } from '../__mocks__/deployment-server.mocks';
import { LiveDeploymentConfigPage } from '../deployment-server.types';
import { AUTH_ERROR_MESSAGE, OTHER_ERROR_MESSAGE } from '../constants';

describe('deployment server reducer', () => {
    const {
        closeDrawer,
        initDrawer,
        setDeploymentServers,
        filterDeploymentServers,
        setDeploymentServerSearch,
        setLiveDeploymentConfigs,
        setLiveDeploymentConfigsCount,
        runAddDeploymentServer,
        runUseDeploymentServer,
        openDeploymentServerDrawer,
        openLiveDeployments,
        openLiveDeploymentConfiguration,
        setConfigurationPage,
        setConfigurationCondition,
        setDialogError,
        setIsLoading,
        setConfigurationFolderId,
        loadLiveDeploymentsConfiguration,
        loadConnectionServerMetadata,
        setConnectionServerMetadata,
        runDeleteDeploymentServer,
        runEditDeploymentServer,
        runDeleteDeploymentServerApplication,
        reloadLiveDeploymentsConfiguration,
    } = folderDeploymentServers.actions;

    it('should have the initial state', () => {
        expect(folderDeploymentServers.name).toBe('folderDeploymentServers');
        expect(folderDeploymentServers.reducer(undefined, { type: 'default' })).toBe(initialState);
    });

    describe('reducers', () => {
        describe('closeDrawer', () => {
            it('should close the drawer and reset deployment servers', () => {
                const state = folderDeploymentServers.reducer(initialState, closeDrawer());
                expect(state.isDrawerOpen).toBe(false);
                expect(state.deploymentServers).toStrictEqual([]);
            });
        });

        describe('openDeploymentServerDrawer', () => {
            it('should open the drawer and set loading states', () => {
                const state = folderDeploymentServers.reducer(initialState, openDeploymentServerDrawer());
                expect(state.isDrawerOpen).toBe(true);
                expect(state.deploymentServerDialogError).toBeUndefined();
            });
        });

        describe('initDrawer', () => {
            it('should initialize the drawer and set loading states', () => {
                const state = folderDeploymentServers.reducer(initialState, initDrawer());
                expect(state.isLoadingDeploymentServers).toBe(true);
                expect(state.isLoadingConnectionServers).toBe(true);
                expect(state.deploymentServerDialogError).toBeUndefined();
            });
        });

        describe('setDeploymentServers', () => {
            it('should set deployment servers and update loading states', () => {
                const payload = [mockArgoDeploymentServer];
                const state = folderDeploymentServers.reducer(initialState, setDeploymentServers(payload));
                expect(state.deploymentServers).toStrictEqual(payload);
                expect(state.isLoadingDeploymentServers).toBe(false);
                expect(state.isLoading).toBe(false);
            });
        });

        describe('filterDeploymentServers', () => {
            it('should set loading state for deployment servers', () => {
                const payload: DeploymentServerSearch = { searchInput: 'test' };
                const state = folderDeploymentServers.reducer(initialState, filterDeploymentServers(payload));
                expect(state.isLoadingDeploymentServers).toBe(true);
            });
        });

        describe('setDeploymentServerSearch', () => {
            it('should set deployment server search', () => {
                const payload: DeploymentServerSearch = { searchInput: 'test' };
                const state = folderDeploymentServers.reducer(initialState, setDeploymentServerSearch(payload));
                expect(state.deploymentServerSearch).toStrictEqual(payload);
            });
        });

        describe('setLiveDeploymentConfigs', () => {
            it('should set live deployment configs', () => {
                const payload = [mockLiveDeploymentConfigData];
                const state = folderDeploymentServers.reducer(initialState, setLiveDeploymentConfigs(payload));
                expect(state.liveDeploymentConfigs).toStrictEqual(payload);
            });
        });

        describe('setLiveDeploymentConfigsCount', () => {
            it('should set live deployment configs count', () => {
                const payload = 5;
                const state = folderDeploymentServers.reducer(initialState, setLiveDeploymentConfigsCount(payload));
                expect(state.liveDeploymentConfigsCount).toBe(payload);
            });
        });

        describe('runAddDeploymentServer', () => {
            it('should not update state', () => {
                const state = folderDeploymentServers.reducer(initialState, runAddDeploymentServer('test'));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('runUseDeploymentServer', () => {
            it('should not update state', () => {
                const payload = {
                    connectionLabel: mockArgoDeploymentServer.connectionLabel,
                    folderId: 'folder-1',
                    eventSourceId: mockArgoDeploymentServer.eventSourceId,
                };
                const state = folderDeploymentServers.reducer(initialState, runUseDeploymentServer(payload));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('openLiveDeployments', () => {
            it('should not update state', () => {
                const state = folderDeploymentServers.reducer(initialState, openLiveDeployments());
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('openLiveDeploymentConfiguration', () => {
            it('should not update state', () => {
                const state = folderDeploymentServers.reducer(initialState, openLiveDeploymentConfiguration(false));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('setConfigurationPage', () => {
            it('should set configuration page', () => {
                const payload: LiveDeploymentConfigPage = { folderId: 'folder-1', order: 'asc', orderBy: 'title', page: 1, resultsPerPage: 10 };
                const state = folderDeploymentServers.reducer(initialState, setConfigurationPage(payload));
                expect(state.configurationPage).toStrictEqual(payload);
            });
        });

        describe('setConfigurationCondition', () => {
            it('should set configuration condition', () => {
                const payload = 'condition';
                const state = folderDeploymentServers.reducer(initialState, setConfigurationCondition(payload));
                expect(state.configurationCondition).toBe(payload);
            });
        });

        describe('setDialogError', () => {
            it('should set dialog error when payload is defined', () => {
                const payload: DeploymentServerDialogErrorAction = { error: 'error', status: 403 };
                const state = folderDeploymentServers.reducer(initialState, setDialogError(payload));
                expect(state.deploymentServerDialogError).toStrictEqual({
                    error: AUTH_ERROR_MESSAGE,
                    errorDetail: 'error',
                });
            });

            it('should set dialog other error message when payload is defined but status it other than 403', () => {
                const payload: DeploymentServerDialogErrorAction = { error: 'error', status: 404 };
                const state = folderDeploymentServers.reducer(initialState, setDialogError(payload));
                expect(state.deploymentServerDialogError).toStrictEqual({
                    error: OTHER_ERROR_MESSAGE,
                    errorDetail: 'error',
                });
            });

            it('should set dialog error to undefined when payload is NOT defined', () => {
                const state = folderDeploymentServers.reducer(initialState, setDialogError(undefined));
                expect(state.deploymentServerDialogError).toBeUndefined();
            });
        });

        describe('setIsLoading', () => {
            it('should set loading state', () => {
                const payload = true;
                const state = folderDeploymentServers.reducer(initialState, setIsLoading(payload));
                expect(state.isLoading).toBe(payload);
            });
        });

        describe('setConfigurationFolderId', () => {
            it('should set configuration folder id', () => {
                const payload = 'folder-1';
                const state = folderDeploymentServers.reducer(initialState, setConfigurationFolderId(payload));
                expect(state.configurationPage.folderId).toBe(payload);
            });
        });

        describe('loadLiveDeploymentsConfiguration', () => {
            it('should not update state', () => {
                const state = folderDeploymentServers.reducer(initialState, loadLiveDeploymentsConfiguration());
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('loadConnectionServerMetadata', () => {
            it('should not update state', () => {
                const state = folderDeploymentServers.reducer(initialState, loadConnectionServerMetadata());
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('setConnectionServerMetadata', () => {
            it('should set connection server metadata', () => {
                const payload = [mockArgoConnectionServerMetadata];
                const state = folderDeploymentServers.reducer(initialState, setConnectionServerMetadata(payload));
                expect(state.connectionServerMetadata).toStrictEqual(payload);
                expect(state.isLoadingConnectionServers).toBe(false);
            });
        });

        describe('runDeleteDeploymentServer', () => {
            it('should not update state', () => {
                const payload = {
                    connectionLabel: mockArgoDeploymentServer.connectionLabel,
                    folderId: 'folder-1',
                    eventSourceId: mockArgoDeploymentServer.eventSourceId,
                };
                const state = folderDeploymentServers.reducer(initialState, runDeleteDeploymentServer(payload));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('runEditDeploymentServer', () => {
            it('should not update state', () => {
                const payload = {
                    connectionLabel: mockArgoDeploymentServer.connectionLabel,
                    folderId: 'folder-1',
                    eventSourceId: mockArgoDeploymentServer.eventSourceId,
                };
                const state = folderDeploymentServers.reducer(initialState, runEditDeploymentServer(payload));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('runConfigureDeploymentServerApplication', () => {
            it('should not update state', () => {
                const payload = {
                    configurationId: mockArgoDeploymentServer.connectionId,
                    connectionLabel: mockArgoDeploymentServer.connectionLabel,
                    folderId: 'folder-1',
                    eventSourceId: mockArgoDeploymentServer.eventSourceId,
                };
                const state = folderDeploymentServers.reducer(initialState, runDeleteDeploymentServerApplication(payload));
                expect(state).toStrictEqual(initialState);
            });
        });

        describe('reloadLiveDeploymentsConfiguration', () => {
            it('should not update state', () => {
                const payload = {
                    id: 'configuration-1',
                    isUpdate: true,
                };
                const state = folderDeploymentServers.reducer(initialState, reloadLiveDeploymentsConfiguration(payload));
                expect(state).toStrictEqual(initialState);
            });
        });
    });
});
