import React, { memo, useEffect, useState } from 'react';
import { Folder } from '@xlr-ui/app/types';
import { DotButton, DotColumnHeader, DotIcon, DotLink, DotTable, DotTableActions, DotThemeProvider, DotTypography, Order } from '@digital-ai/dot-components';
import { Search } from '@xlr-ui/app/react/components/search/search';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { AvatarWithTextCell } from '@digital-ai/dot-components/src/lib/components/table/TableCell';
import capitalize from 'lodash/capitalize';
import startCase from 'lodash/startCase';
import { folderDeploymentServers, initialConfigurationPage } from '../../ducks/deployment-server.reducer';
import {
    getConfigurationCondition,
    getConfigurationPage,
    getConnectionServerMetadata,
    getIsLoadingSelector,
    getLiveDeploymentConfigsCountSelector,
    getLiveDeploymentConfigsSelector,
} from '../../ducks/deployment-server.selectors';
import { CONNECTION_COLUMN_ID, FILTERS_COLUMN_ID, NAME_COLUMN_ID } from '../../constants';
import { ConnectionServerMetadata, LiveDeploymentConfigPage } from '../../deployment-server.types';
import { LiveDeploymentConfigComponentPortal } from './live-deployment-config-component.portal';

import './live-deployment-configuration-table.component.less';
import { SetupDeploymentPageComponent } from './setup-deployment-page.component';

const {
    setConfigurationCondition,
    setConfigurationPage,
    loadLiveDeploymentsConfiguration,
    loadConnectionServerMetadata,
    openLiveDeployments,
    runDeleteDeploymentServer,
    runEditDeploymentServer,
    setConfigurationFolderId,
} = folderDeploymentServers.actions;

export interface LiveDeploymentConfigurationTableProps {
    folder: Folder;
    forceSetup: boolean;
}

const LiveDeploymentConfigurationTableComponent = ({ folder, forceSetup }: LiveDeploymentConfigurationTableProps) => {
    const condition = useAppSelector(getConfigurationCondition);
    const configPage = useAppSelector(getConfigurationPage);
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const count = useAppSelector(getLiveDeploymentConfigsCountSelector);
    const liveDeploymentConfigs = useAppSelector(getLiveDeploymentConfigsSelector);
    const connectionServerMetadata: ConnectionServerMetadata[] = useAppSelector(getConnectionServerMetadata);

    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(setConfigurationFolderId(folder.id));
        dispatch(loadLiveDeploymentsConfiguration());
        dispatch(loadConnectionServerMetadata());
        return () => {
            dispatch(setConfigurationPage(initialConfigurationPage));
        };
    }, []);

    const [showSetupDeploymentServer, openSetupDeploymentServer] = useState(forceSetup);
    const [filterTimeout, setFilterTimeout] = useState<ReturnType<typeof setTimeout> | undefined>(undefined);

    const onDisplayFilterChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const value = event.target.value;
        if (filterTimeout) {
            clearTimeout(filterTimeout);
            setFilterTimeout(undefined);
        }
        setFilterTimeout(
            setTimeout(() => {
                dispatch(setConfigurationCondition(value));
            }, 500),
        );
    };

    const onPageChange = (newPage: LiveDeploymentConfigPage) => {
        dispatch(setConfigurationPage(newPage));
    };

    return (
        <>
            {(count > 0 || condition) && !showSetupDeploymentServer && (
                <>
                    <LiveDeploymentConfigComponentPortal folder={folder} onAddClick={openSetupDeploymentServer} />
                    <DotThemeProvider>
                        <div className="live-deployment-configuration-table">
                            <div className="live-deployment-configuration-table-title">
                                <DotButton onClick={() => dispatch(openLiveDeployments())} type="text">
                                    <DotIcon iconId="back" />
                                </DotButton>
                                <DotTypography variant="h1">Live deployments configuration</DotTypography>
                                <Search
                                    defaultValue={condition}
                                    inputPlaceholder="Filter by values"
                                    key={condition}
                                    onFilterChange={onDisplayFilterChange}
                                    searchClassNames={['mtm', 'mbm']}
                                />
                            </div>
                            <DotTable
                                columns={headerDefinitions}
                                count={count}
                                data={liveDeploymentConfigs.map((config, index) => {
                                    const metadata = connectionServerMetadata.find((m) => config.connectionType === m.type);
                                    const { connectionTitle, connectionLabel, connectionUrl, eventSourceId, filterProperties, id, title } = config;
                                    return {
                                        id: `${id}-${index}`,
                                        rowData: {
                                            name: eventSourceId
                                                ? ({
                                                      avatarImageSrc: metadata?.iconLocation,
                                                      mainText: title,
                                                  } as AvatarWithTextCell)
                                                : ({
                                                      avatarIcon: 'integrations',
                                                      mainText: title,
                                                  } as AvatarWithTextCell),
                                            connection: (
                                                <>
                                                    <DotTypography variant="body1">{connectionTitle ? connectionTitle : '-'}</DotTypography>
                                                    <DotLink href={connectionUrl} target="_blank" tooltip="Open in new tab">
                                                        {connectionUrl}
                                                    </DotLink>
                                                </>
                                            ),
                                            filters: (
                                                <>
                                                    {Object.entries(filterProperties).map(([key, value]) => (
                                                        <DotTypography key={`${id}-${key}`} variant="body1">
                                                            {capitalize(startCase(key))}: {value}
                                                        </DotTypography>
                                                    ))}
                                                </>
                                            ),
                                            actions: (
                                                <>
                                                    <DotTableActions
                                                        actions={[
                                                            {
                                                                disabled: !eventSourceId,
                                                                iconId: 'edit',
                                                                id: 'edit',
                                                                label: 'Edit',
                                                                onClick: () =>
                                                                    dispatch(
                                                                        runEditDeploymentServer({
                                                                            connectionLabel,
                                                                            eventSourceId,
                                                                            configurationId: id,
                                                                            folderId: folder.id,
                                                                        }),
                                                                    ),
                                                            },
                                                            {
                                                                iconId: 'delete',
                                                                id: 'delete',
                                                                label: 'Delete',
                                                                onClick: () =>
                                                                    dispatch(
                                                                        runDeleteDeploymentServer({
                                                                            connectionLabel,
                                                                            eventSourceId,
                                                                            configurationId: id,
                                                                            folderId: folder.id,
                                                                        }),
                                                                    ),
                                                            },
                                                        ]}
                                                        id="superman"
                                                        key={`${eventSourceId}-${id}-${index}`}
                                                    />
                                                </>
                                            ),
                                        },
                                    };
                                })}
                                loading={isLoading}
                                /* eslint-disable-next-line max-params */
                                onUpdateData={(order: Order, orderBy: string, page: number, rowsPerPage) =>
                                    onPageChange({
                                        ...configPage,
                                        page,
                                        resultsPerPage: rowsPerPage as 10 | 25 | 50 | 100,
                                        order,
                                        orderBy,
                                        folderId: folder.id,
                                    })
                                }
                                order={configPage.order}
                                orderBy={configPage.orderBy}
                                page={configPage.page}
                                rowsPerPage={configPage.resultsPerPage as 10 | 25 | 50 | 100}
                                rowsPerPageOptions={[10, 25, 50, 100]}
                            />
                        </div>
                    </DotThemeProvider>
                </>
            )}
            {((!isLoading && !condition && count === 0) || showSetupDeploymentServer) && (
                <DotThemeProvider>
                    <SetupDeploymentPageComponent folderId={folder.id} />
                </DotThemeProvider>
            )}
        </>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: 'left',
        id: NAME_COLUMN_ID,
        label: 'Name',
        sortable: true,
        truncate: true,
        width: '20%',
    },
    {
        id: CONNECTION_COLUMN_ID,
        label: 'Connection',
        truncate: true,
        width: '35%',
    },
    {
        id: FILTERS_COLUMN_ID,
        label: 'Filter properties',
        sortable: false,
        width: '35%',
    },
    {
        id: 'actions',
        width: '5%',
    },
];

export const LiveDeploymentConfigurationTable = memo(LiveDeploymentConfigurationTableComponent);
