package com.xebialabs.xlrelease.status.service

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.internal.filter.LiveDeploymentFilters
import com.xebialabs.xlrelease.api.internal.views.{LiveDeploymentOrderDirection, LiveDeploymentOrderMode}
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments._
import com.xebialabs.xlrelease.environments.repository.sql.persistence.EnvironmentPersistence
import com.xebialabs.xlrelease.environments.repository.sql.{mapApplicationContent, mapEnvironmentContent}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentLabelRepository, EnvironmentRepository, EnvironmentStageRepository}
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.status.repository.persistence.LiveDeploymentPersistence
import com.xebialabs.xlrelease.status.repository.persistence.data.LiveDeploymentRow
import com.xebialabs.xlrelease.status.sse.service.ServerSentEventsService
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, PageRequest, ReleasePageImpl}
import org.springframework.stereotype.Service

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait LiveDeploymentService {
  def batchSaveOrUpdate(folderId: CiId, states: Vector[DeploymentServerEvent]): List[String]

  def findOnFolder(folderId: String,
                   limit: Int, page: Int,
                   orderBy: LiveDeploymentOrderMode,
                   direction: LiveDeploymentOrderDirection,
                   condition: String = ""): Page[LiveDeployment]

  def exists(folderId: CiId, maxAge: Long): Boolean

  def saveAndNotify(folderId: String, processedEvent: DeploymentServerEvent): Unit

  def count(folderId: CiId, condition: String): Integer

  def removeStaleDeployments(eventSourceId: String, exclude: List[String] = List()): Unit

  def search(liveDeploymentFilters: LiveDeploymentFilters): List[LiveDeployment]

  def findByDeploymentId(deploymentId: String): LiveDeployment

  def searchEnvironments(liveDeploymentFilters: LiveDeploymentFilters): List[Environment]

  def searchApplications(liveDeploymentFilters: LiveDeploymentFilters): List[Application]
}

@Service
@IsTransactional
class LiveDeploymentServiceImpl(deploymentPersistence: LiveDeploymentPersistence,
                                 applicationRepository: ApplicationRepository,
                                 environmentRepository: EnvironmentRepository,
                                 environmentStageRepository: EnvironmentStageRepository,
                                 serverSentEventsService: ServerSentEventsService,
                                 repositoryAdapter: SqlRepositoryAdapter,
                               ) extends LiveDeploymentService with Logging {
  private val DefaultUnknown: String = "Unknown"

  override def batchSaveOrUpdate(folderId: CiId, states: Vector[DeploymentServerEvent]): List[String] = {
    implicit val envs: mutable.Map[CiId, Environment] = mutable.Map.empty
    implicit val apps: mutable.Map[CiId, Application] = mutable.Map.empty
    states.flatMap { state =>
      saveOrUpdateStatusEvent(folderId, state, fetchDefaultStage).map(_.getId)
    }.toList
  }

  private def saveOrUpdateStatusEvent(folderId: CiId, event: DeploymentServerEvent, defaultStage: EnvironmentStage)
                                     (implicit envs: mutable.Map[String, Environment] = mutable.Map.empty[String, Environment],
                                      apps: mutable.Map[String, Application] = mutable.Map.empty[String, Application]): Option[LiveDeployment] = {
    val env = envs.getOrElseUpdate(event.environmentCuid, {
      val envMaybe = environmentRepository.findInFolderByCorrelationId(folderId, event.environmentCuid)
      envMaybe.map(existingEnv => {
        refreshEnvironment(existingEnv, event)
      }).getOrElse(environmentRepository.createEnvironment(constructEnvironment(folderId, event, defaultStage)))
    })
    val app = apps.getOrElseUpdate(event.applicationCuid, {
      val appMaybe = applicationRepository.findInFolderByCorrelationId(folderId, event.applicationCuid)
      appMaybe.map(existingApp => {
        refreshApplication(existingApp, event, env)
      }).getOrElse(applicationRepository.createApplication(constructApplication(folderId, event, env)))
    })

    Try(deploymentPersistence.save(toLiveDeployment(folderId, event.configId, app, env, event.deploymentState))) match {
      case Failure(exception) =>
        warn(s"Couldn't save fetched external deployment for Endpoint [${folderId}] into database, skipping record: ${exception.getMessage}", exception)
        None
      case Success(row) => Some(toLiveDeployment(row))
    }
  }

  private def refreshEnvironment(original: Environment, update: DeploymentServerEvent): Environment = {
    var dirty = false
    if (original == null || update == null) {
      original
    } else {
      if (original.getTitle != update.environmentTitle) {
        original.setTitle(update.environmentTitle)
        dirty = true
      }
      val originalTarget = original.getDeploymentTarget
      val updateTarget = update.deploymentTarget

      if (originalTarget != null && updateTarget == null) {
        original.setDeploymentTarget(null)
        dirty = true
      }

      // always update Target if not null
      if (updateTarget != null) {
        original.setDeploymentTarget(updateTarget)
        dirty = true
      }
      if (dirty) environmentRepository.updateEnvironment(original) else original
    }
  }

  def refreshApplication(original: Application, update: DeploymentServerEvent, withEnv: Environment): Application = {
    var dirty = false
    if (original == null || update == null) {
      original
    } else {
      if (original.getTitle != update.applicationTitle) {
        original.setTitle(update.applicationTitle)
        dirty = true
      }
      val containedEnvCorrelationIds = original.getEnvironments.asScala.map(e => e.getCorrelationUid)
      if (!containedEnvCorrelationIds.contains(withEnv.getCorrelationUid)) {
        original.getEnvironments.add(withEnv)
        dirty = true
      }

      val originalSource = original.getApplicationSource
      val updateSource = update.applicationSource

      if (originalSource != null && updateSource == null) {
        original.setApplicationSource(null)
        dirty = true
      }
      // always update Source if not null
      if (updateSource != null) {
        original.setApplicationSource(updateSource)
        dirty = true
      }
      if (dirty) applicationRepository.updateApplication(original) else original
    }
  }

  //noinspection ScalaStyle
  override def saveAndNotify(folderId: String,
                             externalDeploymentEvent: DeploymentServerEvent): Unit = {
    val event: Option[LiveDeploymentEvent] = externalDeploymentEvent match {
      case removeEvent: DeploymentServerEvent if removeEvent.operation == StateOperation.RemoveState =>
        val env = environmentRepository.findInFolderByCorrelationId(folderId, externalDeploymentEvent.environmentCuid)
          .getOrElse(throw new NotFoundException(s"Couldn't find environment with correlation '${externalDeploymentEvent.environmentCuid}'"))
        val app = applicationRepository.findInFolderByCorrelationId(folderId, externalDeploymentEvent.applicationCuid)
          .getOrElse(throw new NotFoundException(s"Couldn't find application with correlation '${externalDeploymentEvent.applicationCuid}'"))
        val liveDeploymentId = deploymentPersistence.fetchLiveDeploymentId(folderId, app.getId, env.getId)
        liveDeploymentId.map{ deploymentId =>
          deploymentPersistence.delete(deploymentId)
          DeleteLiveDeploymentEvent(deploymentId)
        }
      case createEvent: DeploymentServerEvent
        if createEvent.operation == StateOperation.CreateState && Option(createEvent.deploymentState).exists(s => Option(s.getStatus).isDefined) =>
        debug(s"Received CreateStatusEvent " +
          s"[${externalDeploymentEvent.applicationTitle} on " +
          s"${externalDeploymentEvent.environmentTitle}] - refreshing live deployments.")
        Some(CreateLiveDeploymentEvent())
      case updateEvent: DeploymentServerEvent
        if Option(updateEvent.deploymentState).exists(s => Option(s.getStatus).isDefined) =>
        val deployment = saveOrUpdateStatusEvent(folderId, updateEvent, fetchDefaultStage)
        deployment.map(d => UpdateLiveDeploymentEvent(d.getId, d))
      case event: DeploymentServerEvent
        if Option(event.environmentCuid).isEmpty =>
        info(s"Received `${event.getOperation}` event for creation of correlated application ${Option(event.applicationCuid).getOrElse("-")}" +
          s" - skipping status update.")
        None
      case event: DeploymentServerEvent =>
        info(s"Received `${event.getOperation}` event for correlated application ${Option(event.applicationCuid).getOrElse("-")}" +
          s" on correlated environment ${Option(event.environmentCuid).getOrElse("-")} with no status - skipping update.")
        None
      case event => // all the ignored types for now
        warn(s"Couldn't execute ExternalDeploymentEvent[${event.getClass.getName}] persistence - skipping save to database.")
        None
    }
    event.foreach(serverSentEventsService.send)
  }

  override def findOnFolder(folderId: CiId,
                            limit: Int, page: Int,
                            orderBy: LiveDeploymentOrderMode,
                            direction: LiveDeploymentOrderDirection,
                            condition: String): Page[LiveDeployment] = {
    implicit val envCache: mutable.Map[CiId, Environment] = mutable.Map.empty
    implicit val appCache: mutable.Map[CiId, Application] = mutable.Map.empty
    val results = deploymentPersistence.findOnFolder(folderId, limit, limit * page, orderBy, direction, condition)
      .map(toLiveDeployment)
    val totalCount = deploymentPersistence.count(folderId, condition)
    val pageable = PageRequest.of(page, limit, LiveDeploymentOrderDirection.toDirection(direction), orderBy.toString)
    new ReleasePageImpl[LiveDeployment](results.asJava, pageable, totalCount)
  }

  override def exists(folderId: CiId, maxAge: Long): Boolean =
    deploymentPersistence.exists(folderId, maxAge)

  private def toLiveDeployment(row: LiveDeploymentRow)
                              (implicit envCache: mutable.Map[String, Environment],
                               appCache: mutable.Map[String, Application]
                              ): LiveDeployment = {
    val ci = CiSerializerHelper.deserialize(row.json, repositoryAdapter).asInstanceOf[LiveDeployment]
    ci.setFolderId(row.folderId)
    val env = envCache.getOrElseUpdate(
      ci.getEnvironment.getId,
      environmentRepository.findEnvironmentById(ci.getEnvironment.getId)
    )
    val app = appCache.getOrElseUpdate(
      ci.getApplication.getId,
      applicationRepository.findApplicationById(ci.getApplication.getId)
    )
    ci.setEnvironment(env)
    ci.setApplication(app)
    ci
  }

  private def toLiveDeployment(folderId: CiId, eventSourceId: CiId,
                               application: Application, environment: Environment,
                               state: DeploymentState): LiveDeployment = {

    LiveDeployment.create(
      folderId, eventSourceId,
      application, environment,
      state
    )
  }

  private def constructApplication(folderId: String,
                                   event: DeploymentServerEvent,
                                   environment: Environment
                                  ): Application = {
    val application = new Application
    application.setId(null)
    application.setTitle(event.applicationTitle)
    application.setApplicationSource(event.applicationSource)
    application.setCorrelationUid(event.applicationCuid)
    application.setFolderId(folderId)
    application.setEnvironments(List.apply(environment).asJava)
    application
  }

  private def constructEnvironment(folderId: CiId, event: DeploymentServerEvent, stage: EnvironmentStage): Environment = {
    val environment = new Environment
    environment.setId(null)
    environment.setTitle(event.environmentTitle)
    environment.setCorrelationUid(event.environmentCuid)
    environment.setDeploymentTarget(event.deploymentTarget)
    environment.setFolderId(folderId)
    environment.setStage(stage)
    Option(event.deploymentTarget).foreach { target =>
      Option(target.getTargetUrl).filter(!_.isBlank).foreach { url => environment.setDescription(s"Target: $url") }
    }
    environment
  }

  private def fetchDefaultStage = {
    Try(environmentStageRepository.findByTitle(DefaultUnknown)) match {
      case Success(stage) => stage
      case Failure(_: NotFoundException) =>
        environmentStageRepository.create(EnvironmentStage.create(DefaultUnknown))
      case Failure(exception) => throw exception
    }
  }

  override def count(folderId: CiId, condition: String): Integer = {
    deploymentPersistence.count(folderId, condition)
  }

  override def removeStaleDeployments(eventSourceId: String, exclude: List[String]): Unit = {
    deploymentPersistence.deleteByEventSourceIdAndDeploymentIdNotIn(eventSourceId, exclude)
  }

  override def search(liveDeploymentFilters: LiveDeploymentFilters): List[LiveDeployment] = {
    implicit val envs: mutable.Map[CiId, Environment] = mutable.Map.empty
    implicit val apps: mutable.Map[CiId, Application] = mutable.Map.empty
    deploymentPersistence.search(liveDeploymentFilters)
      .map(toLiveDeployment)
      .toList
  }

  override def findByDeploymentId(deploymentId: String): LiveDeployment = {
    implicit val envs: mutable.Map[CiId, Environment] = mutable.Map.empty
    implicit val apps: mutable.Map[CiId, Application] = mutable.Map.empty
    deploymentPersistence.findByDeploymentId(deploymentId).map(toLiveDeployment).getOrElse(
      throw new NotFoundException(s"Couldn't find LiveDeployment with id $deploymentId")
    )
  }

  override def searchEnvironments(liveDeploymentFilters: LiveDeploymentFilters): List[Environment] = {
    implicit val cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty
    implicit val cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty
    implicit val environmentLabelRepository: EnvironmentLabelRepository = null
    implicit val environmentStageRepository: EnvironmentStageRepository = this.environmentStageRepository
    implicit val repositoryAdapter: SqlRepositoryAdapter = this.repositoryAdapter

    deploymentPersistence.searchEnvironments(liveDeploymentFilters)
      .map(mapEnvironmentContent)
      .toList
  }

  override def searchApplications(liveDeploymentFilters: LiveDeploymentFilters): List[Application] = {
    implicit val cachedStages: mutable.Map[CiId, EnvironmentStage] = mutable.Map.empty
    implicit val cachedLabels: mutable.Map[CiId, EnvironmentLabel] = mutable.Map.empty
    implicit val environmentLabelRepository: EnvironmentLabelRepository = null
    implicit val environmentStageRepository: EnvironmentStageRepository = this.environmentStageRepository
    implicit val repositoryAdapter: SqlRepositoryAdapter = this.repositoryAdapter
    implicit val environmentPersistence: EnvironmentPersistence = this.deploymentPersistence.environmentPersistence

    deploymentPersistence.searchApplications(liveDeploymentFilters)
      .map(mapApplicationContent)
      .toList
  }
}

object StateOperation {
  final val CreateState = "create"
  final val RemoveState = "remove"
}
