package com.xebialabs.xlrelease.status.webhook.events

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.platform.script.jython.ScriptSource
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlplatform.webhooks.events.handlers.{EventProcessorHandler, EventPublisher}
import com.xebialabs.xlrelease.domain.environments.LiveDeployment
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.status.service.LiveDeploymentService
import com.xebialabs.xlrelease.status.service.script.{FilterScriptService, MapScriptService}
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.webhooks.utils.JythonScriptEventProcessingUtils
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component
import org.springframework.util.ResourceUtils

import java.io.FileNotFoundException
import scala.util.{Failure, Success, Try}

@Component
class StatusWebhookConsumerHandler(val eventPublisher: EventPublisher,
                                   configurationRepository: ConfigurationRepository,
                                   externalDeploymentService: LiveDeploymentService,
                                   jythonScriptUtils: JythonScriptEventProcessingUtils,
                                   filterScriptService: FilterScriptService,
                                   mapScriptService: MapScriptService)
  extends EventProcessorHandler[Event, Event, StatusWebhookEventSource]
    with Logging {

  import jythonScriptUtils._

  override def consumerConfigType: Type = Type.valueOf(classOf[StatusWebhookEventSource])

  override def filter(config: StatusWebhookEventSource, event: Event): Boolean = {
    if (config.isInstanceOf[RemoteStatusWebhookEventSource]) {
      filterScriptService.executeScript(config, event)
    } else {
      nonEmptyScriptLocation(config.filterScript).forall { path =>
        Try(ResourceUtils.getFile(path)).map { _ =>
          executeFilter(config, event, sandbox = false)
            .apply(logger)
            .apply(ScriptSource.byResource(path))
        }.recover {
          case e: FileNotFoundException =>
            logger.error("Filter caused exception, assuming false", e)
            false
        }.get
      }
    }
  }

  override def map(config: StatusWebhookEventSource, event: Event): Event = {
    if (config.consumerEnabled && event.isInstanceOf[HttpRequestEvent]) {
      handleStatusEvent(config, event.asInstanceOf[HttpRequestEvent]).orNull
    } else {
      null
    }
  }

  private def handleStatusEvent(config: StatusWebhookEventSource, event: HttpRequestEvent): Option[Event] = {
    logger.trace(s"receivedEvent from ${event.sourceId} [${event.content}]")
    val sourceServerId = config.sourceServer.getId
    val connection = configurationRepository.read[StatusHttpConnection](sourceServerId)
    val externalDeploymentEvent = Try(toExternalDeploymentEvent(config, event)) match {
      case Success(event: DeploymentServerEvent) =>
        Some(event)
      case Success(errorObject) =>
        val errorType = Option(errorObject).map(_.getClass.getName).getOrElse("null")
        error(s"Illegal response in ${connection.getId} transformerScript response. " +
          s"Returned type was $errorType, expected state or package objects.")
        None
      case Failure(exception) =>
        error(s"Exception thrown from transformerScript for ${connection.getType.toString} endpoint.", exception)
        None
    }
    externalDeploymentEvent.foreach(processedEvent => {
      Try(externalDeploymentService.saveAndNotify(config.getFolderId, processedEvent)) match {
        case Failure(e) =>
          warn(s"Failed to process event, processing was skipped and didn't propagate to database or displayed on the Live Deployment screen: ${e.getMessage}")
        case Success(_) =>
      }
    })
    externalDeploymentEvent
  }

  private def toExternalDeploymentEvent(config: StatusWebhookEventSource, event: Event): Event = {
    if (config.isInstanceOf[RemoteStatusWebhookEventSource]) {
      mapScriptService.executeScript(config, event)
    } else {
      executeMap(config, event, sandbox = false)
        .apply(logger)
        .apply(ScriptSource.byResource(scriptLocationOrDefault(config.getType, config.mapScript)))
    }
  }
}

sealed class LiveDeploymentEvent(val deploymentId: String)

final case class CreateLiveDeploymentEvent() extends LiveDeploymentEvent(null)

final case class DeleteLiveDeploymentEvent(override val deploymentId: String) extends LiveDeploymentEvent(deploymentId)

sealed case class UpdateLiveDeploymentEvent(override val deploymentId: String, data: LiveDeployment) extends LiveDeploymentEvent(deploymentId)
