import React from 'react';
import { ReactWrapper } from 'enzyme';
import { DotButton, DotDrawer, DotIconButton } from '@digital-ai/dot-components';
import { DeploymentServerDrawer } from './deployment-server-drawer.component';
import { folderDeploymentServers } from '../../ducks/deployment-server.reducer';
import { mockResizeObserver, mountWithStoreAndTheme } from '@xlr-ui/tests/unit/testing-utils';
import { DeploymentServerCatalog } from './deployment-server-catalog.component';
import { Folder } from '@xlr-ui/app/types';
import {
    mockArgoConnectionServerMetadata,
    mockArgoDeploymentServer,
    mockDeployConnectionServerMetadata,
    mockDeployDeploymentServer,
} from '../../__mocks__/deployment-server.mocks';

describe('DeploymentServerDrawer', () => {
    const { filterDeploymentServers, runUseDeploymentServer, setDialogError } = folderDeploymentServers.actions;
    const dispatch = jest.fn();
    const folder: Folder = { id: '1', title: 'Test Folder' };

    const defaultState = {
        connectionServerMetadata: [mockArgoConnectionServerMetadata, mockDeployConnectionServerMetadata],
        isLoadingDeploymentServers: false,
        isLoadingConnectionServers: false,
        deploymentServerSearch: {},
        deploymentServers: [mockArgoDeploymentServer, mockDeployDeploymentServer],
        deploymentServerDialogError: undefined,
        isDrawerOpen: true,
    };

    const mountComponent = (state = defaultState) =>
        mountWithStoreAndTheme(<DeploymentServerDrawer folderId={folder.id} />, dispatch, { folderDeploymentServers: state });

    const getDrawer = (wrapper: ReactWrapper) => wrapper.findWhere((node) => node.is(DotDrawer) && node.props().className === 'deployment-server-drawer');
    const getAddButton = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotButton) && node.props()['data-testid'] === 'deployment-server-drawer-add-btn');
    const getCloseButton = (wrapper: ReactWrapper) =>
        wrapper.findWhere((node) => node.is(DotIconButton) && node.props()['data-testid'] === 'deployment-server-drawer-close-btn');
    const getCatalog = (wrapper: ReactWrapper) => wrapper.find(DeploymentServerCatalog);

    beforeEach(() => {
        jest.clearAllMocks();
        mockResizeObserver();
    });

    it('should render properly', () => {
        const wrapper = mountComponent();

        const drawerProps = getDrawer(wrapper).props();
        expect(drawerProps.PaperProps).toStrictEqual({
            style: {
                top: '48px',
                padding: 0,
            },
        });
        expect(drawerProps.anchor).toBe('bottom');
        expect(drawerProps.className).toBe('deployment-server-drawer');
        expect(drawerProps.open).toBe(true);

        expect(getCloseButton(wrapper)).toExist();
        expect(getCatalog(wrapper)).toExist();
    });

    it('should not render content if isDrawerOpen is false', () => {
        const wrapper = mountComponent({ ...defaultState, isDrawerOpen: false });
        expect(getDrawer(wrapper).props().open).toBe(false);
        expect(getAddButton(wrapper)).not.toExist();
        expect(getCloseButton(wrapper)).not.toExist();
        expect(getCatalog(wrapper)).not.toExist();
    });

    it('should dispatch correct action when onConnectionChange is executed', () => {
        const currentSearch = defaultState.deploymentServerSearch;
        const wrapper = mountComponent();
        const catalog = getCatalog(wrapper);
        catalog.invoke('onConnectionChange')(currentSearch);
        expect(dispatch).toHaveBeenCalledWith(setDialogError(undefined));
        expect(dispatch).toHaveBeenCalledWith(filterDeploymentServers(currentSearch));
    });

    it('should dispatch correct action when onUseDeploymentServer is executed', () => {
        const currentServer = defaultState.deploymentServers[0];
        const wrapper = mountComponent();
        const catalog = getCatalog(wrapper);
        catalog.invoke('onUseDeploymentServer')(currentServer);
        expect(dispatch).toHaveBeenCalledWith(setDialogError(undefined));
        expect(dispatch).toHaveBeenCalledWith(
            runUseDeploymentServer({
                connectionLabel: currentServer.connectionLabel,
                folderId: folder.id,
                eventSourceId: currentServer.eventSourceId,
            }),
        );
    });

    it('should dispatch correct action when onFilter is executed', () => {
        const currentSearch = defaultState.deploymentServerSearch;
        const wrapper = mountComponent();
        const catalog = getCatalog(wrapper);
        catalog.invoke('onFilter')(currentSearch);
        expect(dispatch).toHaveBeenCalledWith(setDialogError(undefined));
        expect(dispatch).toHaveBeenCalledWith(filterDeploymentServers(currentSearch));
    });
});
