package com.xebialabs.xlrelease.status.webhook.events

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.platform.script.jython.ScriptSource
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlplatform.webhooks.events.handlers.EventProcessorHandler
import com.xebialabs.xlrelease.domain.environments.LiveDeployment
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.status.service.LiveDeploymentService
import com.xebialabs.xlrelease.status.service.script.{FilterScriptService, MapScriptService}
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.webhooks.consumers.BaseConsumerHandler
import com.xebialabs.xlrelease.webhooks.jms.JmsEventPublisherHandler
import com.xebialabs.xlrelease.webhooks.utils.JythonScriptEventProcessingUtils
import grizzled.slf4j.Logging
import org.springframework.jms.core.JmsTemplate
import org.springframework.stereotype.Component
import org.springframework.util.ResourceUtils

import java.io.FileNotFoundException
import java.util.Date
import scala.util.{Failure, Success, Try}

@Component
class StatusWebhookConsumerHandler(override val jmsTemplate: JmsTemplate,
                                   configurationRepository: ConfigurationRepository,
                                   externalDeploymentService: LiveDeploymentService,
                                   jythonScriptUtils: JythonScriptEventProcessingUtils,
                                   filterScriptService: FilterScriptService,
                                   mapScriptService: MapScriptService)
  extends EventProcessorHandler[Event, Event, StatusWebhookEventSource]
    with JmsEventPublisherHandler[Event, StatusWebhookEventSource]
    with BaseConsumerHandler[StatusWebhookEventSource]
    with Logging {

  import jythonScriptUtils._

  override def consumerConfigType: Type = Type.valueOf(classOf[StatusWebhookEventSource])

  override def consumeEvent(config: StatusWebhookEventSource, event: Event): Boolean = {
    event match {
      case httpEvent: HttpRequestEvent if config.consumerEnabled =>
        try {
          handleStatusEvent(config, httpEvent)
          true
        } catch {
          case e: Throwable =>
            throw e
        }
      case _ => true
    }
  }

  override def filter(config: StatusWebhookEventSource, event: Event): Boolean = {
    if (config.isInstanceOf[RemoteStatusWebhookEventSource]) {
      filterScriptService.executeScript(config, event)
    } else {
      nonEmptyScriptLocation(config.filterScript).forall { path =>
        Try(ResourceUtils.getFile(path)).map { _ =>
          executeFilter(config, event, sandbox = false)
            .apply(logger)
            .apply(ScriptSource.byResource(path))
        }.recover {
          case e: FileNotFoundException =>
            logger.error("Filter caused exception, assuming false", e)
            false
        }.get
      }
    }
  }

  def map(config: StatusWebhookEventSource, event: Event): Event = {
    if (config.isInstanceOf[RemoteStatusWebhookEventSource]) {
      mapScriptService.executeScript(config, event)
    } else {
      executeMap(config, event, sandbox = false)
        .apply(logger)
        .apply(ScriptSource.byResource(scriptLocationOrDefault(config.getType, config.mapScript)))
    }
  }

  private def handleStatusEvent(config: StatusWebhookEventSource, event: HttpRequestEvent): Unit = {
    logger.trace(s"receivedEvent from ${event.sourceId} [${event.content}]")
    val sourceServerId = config.sourceServer.getId
    val connection = configurationRepository.read[StatusHttpConnection](sourceServerId)
    val externalDeploymentEvent = Try(map(config, event)) match {
      case Success(event: DeploymentServerEvent) =>
        Some(event)
      case Success(errorObject) =>
        val errorType = Option(errorObject).map(_.getClass.getName).getOrElse("null")
        error(s"Illegal response in ${connection.getId} transformerScript response. " +
          s"Returned type was $errorType, expected state or package objects.")
        None
      case Failure(exception) =>
        error(s"Exception thrown from transformerScript for ${connection.getType.toString} endpoint.", exception)
        None
    }
    externalDeploymentEvent.foreach(processedEvent => {
      Try(externalDeploymentService.saveAndNotify(
        config.getFolderId,
        processedEvent
      )) match {
        case Failure(e) =>
          warn(s"Failed to process event, processing was skipped and didn't propagate to database or displayed on the Live Deployment screen: ${e.getMessage}")
        case Success(_) =>
      }

      // publish deployment event to JMS to consume it later for event based triggers
      processedEvent.sourceId = config.getId
      processedEvent.received = new Date()

      Try(publish(config, processedEvent)) match {
        case Failure(e) =>
          warn(s"Failed to publish event, processing was skipped: ${e.getMessage}")
        case Success(_) =>
          trace(s"Published event for ${processedEvent.getType} with sourceId ${processedEvent.sourceId}")
      }
    })
  }
}

sealed class LiveDeploymentEvent(val deploymentId: String)

final case class CreateLiveDeploymentEvent() extends LiveDeploymentEvent(null)

final case class DeleteLiveDeploymentEvent(override val deploymentId: String) extends LiveDeploymentEvent(deploymentId)

sealed case class UpdateLiveDeploymentEvent(override val deploymentId: String, data: LiveDeployment) extends LiveDeploymentEvent(deploymentId)
