import React from 'react';
import { DotDrawer, DotIconButton, DotTypography } from '@digital-ai/dot-components';
import { useAppDispatch, useAppSelector } from '@xlr-ui/app/js/hooks';
import { DeploymentServerSearch, folderDeploymentServers, getDeploymentServerState } from '../../ducks/deployment-server.reducer';

import './deployment-server-drawer.component.less';
import { DeploymentServerCatalog } from './deployment-server-catalog.component';
import { DeploymentServer } from '../../deployment-server.types';
import { workflow } from '@xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { CREATE_DEPLOYMENT_PROVIDER_WORKFLOWS_TITLE, DEFAULT_ADD_DEPLOYMENT_PROVIDER_SEARCH_CATEGORY } from '../../constants';

export interface DeploymentServerDrawerProps {
    folderId: string;
}

const { closeDrawer, filterDeploymentServers, runUseDeploymentServer, setDialogError } = folderDeploymentServers.actions;
const { initCondenseViewDrawer } = workflow.actions;

export const DeploymentServerDrawer = ({ folderId }: DeploymentServerDrawerProps) => {
    const dispatch = useAppDispatch();

    const {
        connectionServerMetadata,
        isLoadingDeploymentServers,
        isLoadingConnectionServers,
        deploymentServerSearch,
        deploymentServers,
        deploymentServerDialogError,
        isDrawerOpen,
    } = useAppSelector(getDeploymentServerState);

    const handleCloseDrawer = () => {
        dispatch(closeDrawer());
    };

    const handleDialogErrorClear = () => {
        dispatch(setDialogError(undefined));
    };

    const handleConnectionServerChange = (currentSearch: DeploymentServerSearch) => {
        handleDialogErrorClear();
        dispatch(filterDeploymentServers(currentSearch));
    };

    const handleAddDeploymentServer = () => {
        handleDialogErrorClear();
        dispatch(
            initCondenseViewDrawer({
                catalogTitle: CREATE_DEPLOYMENT_PROVIDER_WORKFLOWS_TITLE,
                workflowSearch: { categories: [DEFAULT_ADD_DEPLOYMENT_PROVIDER_SEARCH_CATEGORY] },
            }),
        );
    };

    const handleUseDeploymentServer = (deploymentServer: DeploymentServer) => {
        handleDialogErrorClear();
        const { connectionLabel, eventSourceId } = deploymentServer;
        dispatch(runUseDeploymentServer({ connectionLabel, eventSourceId, folderId }));
    };

    const handleDeploymentServerFilter = (currentSearch: DeploymentServerSearch) => {
        handleDialogErrorClear();
        dispatch(filterDeploymentServers(currentSearch));
    };

    return (
        <DotDrawer
            PaperProps={{
                style: {
                    top: '48px',
                    padding: 0,
                },
            }}
            anchor="bottom"
            className="deployment-server-drawer"
            open={isDrawerOpen}
        >
            {isDrawerOpen && (
                <>
                    <div className="deployment-server-drawer-header">
                        <div className="deployment-server-drawer-header-inner">
                            <div>
                                <DotTypography variant="h1">Choose deployment server</DotTypography>
                            </div>
                            <div>
                                <DotIconButton data-testid="deployment-server-drawer-close-btn" iconId="close" onClick={handleCloseDrawer} tooltip="Close" />
                            </div>
                        </div>
                    </div>
                    <div className="deployment-server-drawer-content">
                        <DeploymentServerCatalog
                            addDeploymentServerHandler={handleAddDeploymentServer}
                            connectionServers={connectionServerMetadata}
                            deploymentServerDialogError={deploymentServerDialogError}
                            deploymentServers={deploymentServers}
                            isLoadingConnectionServers={isLoadingConnectionServers}
                            isLoadingDeploymentServers={isLoadingDeploymentServers}
                            onConnectionChange={handleConnectionServerChange}
                            onFilter={handleDeploymentServerFilter}
                            onUseDeploymentServer={handleUseDeploymentServer}
                            search={deploymentServerSearch}
                        />
                    </div>
                </>
            )}
        </DotDrawer>
    );
};
