import React, { ChangeEvent } from 'react';
import { ReactWrapper } from 'enzyme';
import { DotIconButton, DotTable } from '@digital-ai/dot-components';
import { mountComponentWithStore } from '@xlr-ui/tests/unit/testing-utils';
import { LiveDeploymentConfigurationTable } from './live-deployment-configuration-table.component';
import { Search } from '@xlr-ui/app/react/components/search/search';
import { mockLiveDeploymentConfigData } from '../../__mocks__/deployment-server.mocks';
import { DeploymentServerState, folderDeploymentServers, initialConfigurationPage, initialState } from '../../ducks/deployment-server.reducer';
import { SetupDeploymentPageComponent } from './setup-deployment-page.component';
import { LiveDeploymentConfigComponentPortal } from './live-deployment-config-component.portal';

const { setConfigurationFolderId, setConfigurationPage, setConfigurationCondition, loadLiveDeploymentsConfiguration, runEditDeploymentServer } =
    folderDeploymentServers.actions;
const dispatch = jest.fn();

describe('LiveDeploymentConfigTable component', () => {
    let wrapper: ReactWrapper;
    const defaultProps = {
        folder: { id: 'folderId', title: 'Folder' },
        forceSetup: false,
    };

    const fakeEvent = {
        stopPropagation: jest.fn(),
    };

    const mountComponent = (state: DeploymentServerState = initialState, props = defaultProps) => {
        wrapper = mountComponentWithStore(<LiveDeploymentConfigurationTable {...props} />, dispatch, {
            folderDeploymentServers: state,
        });
        expect(dispatch).toBeCalledWith(setConfigurationFolderId('folderId'));
        expect(dispatch).toBeCalledWith(loadLiveDeploymentsConfiguration());
    };

    afterEach(() => {
        wrapper.unmount();
    });

    const searchTable = (selector: string) => wrapper.find(`.live-deployment-configuration-table ${selector}`);

    it('should show choose or onboard empty screen when no configs', () => {
        mountComponent();
        expect(wrapper.find('.choose-onboard-connect')).toExist();
    });

    it('should trigger filter search', () => {
        mountComponent({
            ...initialState,
            liveDeploymentConfigsCount: 1,
            liveDeploymentConfigs: [mockLiveDeploymentConfigData],
        });
        jest.useFakeTimers();
        jest.spyOn(global, 'clearTimeout');
        jest.spyOn(global, 'setTimeout');
        const e1 = { target: { value: 'filter' } };
        wrapper.find(Search).invoke('onFilterChange')?.(e1 as unknown as ChangeEvent<HTMLInputElement>);
        wrapper.update();
        expect(setTimeout).toHaveBeenCalledTimes(1);
        expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), 500);
        jest.runAllTimers();
        expect(dispatch).toHaveBeenCalledWith(setConfigurationCondition('filter'));
        expect(dispatch).toHaveBeenCalledWith(loadLiveDeploymentsConfiguration());
        const e2 = { target: { value: 'filter 2' } };
        wrapper.find(Search).invoke('onFilterChange')?.(e2 as unknown as ChangeEvent<HTMLInputElement>);
        wrapper.update();
        expect(clearTimeout).toHaveBeenCalledTimes(1);
        jest.runAllTimers();
        expect(dispatch).toHaveBeenCalledWith(setConfigurationCondition('filter 2'));
        expect(dispatch).toHaveBeenCalledWith(loadLiveDeploymentsConfiguration());
    });

    it('should not show one row and trigger page change', () => {
        mountComponent({
            ...initialState,
            liveDeploymentConfigsCount: 1,
            liveDeploymentConfigs: [mockLiveDeploymentConfigData],
        });
        expect(searchTable('.dot-tbody .empty-row').exists()).toBeFalsy();
        wrapper.find(DotTable).invoke('onUpdateData')?.('asc', 'column', 1, 10);
        expect(dispatch).toHaveBeenCalledWith(
            setConfigurationPage({
                ...initialConfigurationPage,
                folderId: defaultProps.folder.id,
                order: 'asc',
                orderBy: 'column',
            }),
        );
        expect(dispatch).toHaveBeenCalledWith(loadLiveDeploymentsConfiguration());
    });

    it('should show rows when data available', () => {
        mountComponent({
            ...initialState,
            liveDeploymentConfigsCount: 1,
            liveDeploymentConfigs: [mockLiveDeploymentConfigData],
        });
        expect(searchTable('.dot-tbody .empty-row').exists()).toBeFalsy();
        expect(searchTable('.dot-table').exists()).toBeTruthy();
    });

    it('should dispatch correct action when Edit is clicked', () => {
        mountComponent({
            ...initialState,
            liveDeploymentConfigsCount: 1,
            liveDeploymentConfigs: [mockLiveDeploymentConfigData],
        });
        const editButton = wrapper.findWhere((node) => node.is(DotIconButton) && node.props().tooltip === 'Edit').at(0);
        editButton.invoke('onClick')(fakeEvent);
        expect(dispatch).toHaveBeenCalledWith(
            runEditDeploymentServer({
                connectionLabel: mockLiveDeploymentConfigData.connectionLabel,
                eventSourceId: mockLiveDeploymentConfigData.eventSourceId,
                configurationId: mockLiveDeploymentConfigData.id,
                folderId: 'folderId',
            }),
        );
    });

    it('should handle empty search input correctly', () => {
        mountComponent({
            ...initialState,
            liveDeploymentConfigsCount: 1,
            liveDeploymentConfigs: [mockLiveDeploymentConfigData],
        });
        jest.useFakeTimers();
        const e = { target: { value: '' } };
        wrapper.find(Search).invoke('onFilterChange')?.(e as unknown as ChangeEvent<HTMLInputElement>);
        wrapper.update();
        jest.runAllTimers();
        expect(dispatch).toHaveBeenCalledWith(setConfigurationCondition(''));
        expect(dispatch).toHaveBeenCalledWith(loadLiveDeploymentsConfiguration());
    });

    it('should show setup deployment server portal when forceSetup is true', () => {
        mountComponent(initialState, { ...defaultProps, forceSetup: true });
        expect(wrapper.find(SetupDeploymentPageComponent)).toExist();
    });

    it('should not show setup deployment server portal when forceSetup is false', () => {
        mountComponent(initialState, { ...defaultProps, forceSetup: false });
        expect(wrapper.find(LiveDeploymentConfigComponentPortal)).not.toExist();
    });
});
