import React from 'react';
import { mockResizeObserver, mountWithTheme, ReactWrapper } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { DeploymentServerCatalog, DeploymentServerCatalogProps } from './deployment-server-catalog.component';
import { CssGrid, DotAlertBanner, DotInputText } from '@digital-ai/dot-components';
import { DeploymentServerCard } from './deployment-server-card.component';
import { DeploymentServerSkeletons } from './skeletons/deployment-server-skeletons.component';
import { DeploymentServerConnectionFilters } from './deployment-server-connection-filters.component';
import { mockArgoDeploymentServer, mockDeployDeploymentServer } from '../../__mocks__/deployment-server.mocks';

describe('DeploymentServerCatalog', () => {
    const onConnectionChangeMock = jest.fn();
    const onFilterMock = jest.fn();
    const onUseDeploymentServerMock = jest.fn();
    const addDeploymentServerHandlerMock = jest.fn();

    const defaultProps: DeploymentServerCatalogProps = {
        addDeploymentServerHandler: addDeploymentServerHandlerMock,
        connectionServers: [],
        deploymentServers: [],
        isLoadingConnectionServers: false,
        isLoadingDeploymentServers: false,
        onConnectionChange: onConnectionChangeMock,
        onFilter: onFilterMock,
        onUseDeploymentServer: onUseDeploymentServerMock,
    };

    let wrapper: ReactWrapper;

    const mountComponent = (props = defaultProps) => {
        wrapper = mountWithTheme(<DeploymentServerCatalog {...props} />);
    };

    beforeEach(() => {
        mockResizeObserver();
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    it('should render main CSS Grid and CSS Cells', () => {
        mountComponent();
        const cssGrid = wrapper.find(CssGrid);
        expect(cssGrid).toExist();
        const leftWrapperCell = wrapper.find('.deployment-server-drawer-content-left-cell');
        expect(leftWrapperCell).toExist();
        const rightWrapperCell = wrapper.find('.tab-content-cell');
        expect(rightWrapperCell).toExist();
    });

    it('should render and filter by text', () => {
        mountComponent();
        const searchInput = wrapper.findWhere((node) => node.is(DotInputText) && node.props().id === 'search');
        expect(searchInput).toExist();
        searchInput.props().onChange({ target: { value: 'test' } });
        expect(onFilterMock).toHaveBeenCalledWith({ searchInput: 'test' });
    });

    it('should call addDeploymentServerHandler when add-deployment-server card is clicked', () => {
        const customProps: DeploymentServerCatalogProps = {
            ...defaultProps,
        };
        mountComponent(customProps);
        const addDeploymentServerCell = wrapper.findWhere((node) => node.key() === 'add-deployment-server');
        expect(addDeploymentServerCell).toExist();
        addDeploymentServerCell.find('.add-deployment-server-button').simulate('click');
        expect(addDeploymentServerHandlerMock).toHaveBeenCalled();
    });

    it('should render deployment server cards', () => {
        const customProps: DeploymentServerCatalogProps = {
            ...defaultProps,
            deploymentServers: [mockArgoDeploymentServer, mockDeployDeploymentServer],
        };
        mountComponent(customProps);
        const cards = wrapper.find(DeploymentServerCard);
        expect(cards).toHaveLength(2);
    });

    it('should render skeletons when loading', () => {
        const customProps = {
            ...defaultProps,
            isLoadingDeploymentServers: true,
        };
        mountComponent(customProps);
        const skeletons = wrapper.find(DeploymentServerSkeletons);
        expect(skeletons).toExist();
    });

    it('should render error message when there is an error', () => {
        const customProps = {
            ...defaultProps,
            deploymentServerDialogError: { error: 'Error message', errorDetail: 'Error details' },
        };
        mountComponent(customProps);
        const alertBanner = wrapper.find(DotAlertBanner);
        expect(alertBanner).toExist();
        expect(alertBanner.text()).toContain('Error message');
    });

    it('should handle connection change', () => {
        mountComponent();
        const filters = wrapper.find(DeploymentServerConnectionFilters);
        expect(filters).toExist();
        filters.props().onConnectionServersFilterChange([{ value: 'server1' }]);
        expect(onConnectionChangeMock).toHaveBeenCalledWith({ connectionServers: ['server1'] });
    });
});
