import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import noop from 'lodash/noop';
import { RootState } from '../../../../../../../../../core/xlr-ui/app/js/store.types';
import { ConnectionServerMetadata, DeploymentServer, LiveDeploymentConfigData, LiveDeploymentConfigPage } from '../deployment-server.types';
import {
    AUTH_ERROR_MESSAGE,
    DEFAULT_CONDITION,
    DEFAULT_CONFIGURATION_ORDER_BY,
    DEFAULT_ORDER,
    DEFAULT_PAGE,
    DEFAULT_RESULTS_PER_PAGE,
    OTHER_ERROR_MESSAGE,
} from '../constants';
import { ConfigurationCallbackPayload } from '../../../../../../../../../core/xlr-ui/app/features/configuration/types';

export interface DeploymentServerSearch {
    connectionServers?: string[];
    folderId?: string;
    searchInput?: string;
}

export interface DeploymentServerDialogError {
    error: string;
    errorDetail: string;
}

export interface DeploymentServerDialogErrorAction {
    error: string;
    status?: number;
}

export interface DeploymentServerState {
    configurationCondition: string;
    configurationPage: LiveDeploymentConfigPage;
    connectionServerMetadata: Array<ConnectionServerMetadata>;
    deploymentServerDialogError?: DeploymentServerDialogError;
    deploymentServerSearch?: DeploymentServerSearch;
    deploymentServers: Array<DeploymentServer>;
    folderId?: string;
    isDrawerOpen: boolean;
    isLoading: boolean;
    isLoadingConnectionServers: boolean;
    isLoadingDeploymentServers: boolean;
    liveDeploymentConfigs: Array<LiveDeploymentConfigData>;
    liveDeploymentConfigsCount: number;
    liveDeploymentConfigurationOpen: boolean;
}

export const initialConfigurationPage: LiveDeploymentConfigPage = {
    folderId: undefined,
    order: DEFAULT_ORDER,
    orderBy: DEFAULT_CONFIGURATION_ORDER_BY,
    page: DEFAULT_PAGE,
    resultsPerPage: DEFAULT_RESULTS_PER_PAGE,
};

export const initialState: DeploymentServerState = {
    configurationCondition: DEFAULT_CONDITION,
    configurationPage: initialConfigurationPage,
    folderId: undefined,
    isDrawerOpen: false,
    isLoading: false,
    isLoadingConnectionServers: false,
    isLoadingDeploymentServers: false,
    liveDeploymentConfigs: [],
    liveDeploymentConfigsCount: 0,
    liveDeploymentConfigurationOpen: false,
    deploymentServers: [],
    connectionServerMetadata: [],
};

export interface ConfigureDeploymentServer {
    configurationId?: string;
    connectionLabel: string;
    eventSourceId: string;
    folderId: string;
}

export interface ConfigureDeploymentServerApplication {
    folderId: string;
    liveDeploymentId: string;
    workflowTags: string[];
}

export const folderDeploymentServers = createSlice({
    name: 'folderDeploymentServers',
    initialState,
    reducers: {
        closeDrawer: (state) => {
            state.isDrawerOpen = false;
            state.deploymentServers = [];
        },
        openDeploymentServerDrawer: (state) => {
            state.isDrawerOpen = true;
        },
        initDrawer: (state, _action: PayloadAction<Partial<DeploymentServerSearch> | undefined>) => {
            state.isLoadingDeploymentServers = true;
            state.isLoadingConnectionServers = true;
            state.deploymentServerDialogError = undefined;
        },
        setDeploymentServers: (state, action: PayloadAction<Array<DeploymentServer>>) => {
            state.deploymentServers = action.payload;
            state.isLoadingDeploymentServers = false;
            state.isLoading = false;
        },
        filterDeploymentServers: (state, _action: PayloadAction<DeploymentServerSearch>) => {
            state.isLoadingDeploymentServers = true;
        },
        setDeploymentServerSearch: (state, action: PayloadAction<DeploymentServerSearch | undefined>) => {
            state.deploymentServerSearch = action.payload;
        },
        setLiveDeploymentConfigs: (state, action: PayloadAction<Array<LiveDeploymentConfigData>>) => {
            state.liveDeploymentConfigs = action.payload;
        },
        setLiveDeploymentConfigsCount: (state, action: PayloadAction<number>) => {
            state.liveDeploymentConfigsCount = action.payload;
        },
        runAddDeploymentServer: (_state, _action: PayloadAction<string>) => noop(),
        runDeleteDeploymentServer: (_state, _action: PayloadAction<ConfigureDeploymentServer>) => noop(),
        runEditDeploymentServer: (_state, _action: PayloadAction<ConfigureDeploymentServer>) => noop(),
        runUseDeploymentServer: (_state, _action: PayloadAction<ConfigureDeploymentServer>) => noop(),
        runDeleteDeploymentServerApplication: (_state, _action: PayloadAction<ConfigureDeploymentServer>) => noop(),
        openLiveDeployments: (_state) => noop(),
        openLiveDeploymentConfiguration: (_state, _action: PayloadAction<boolean>) => noop(),
        reloadLiveDeploymentsConfiguration: (_state, _action: PayloadAction<ConfigurationCallbackPayload>) => noop(),
        setConfigurationPage: (state, action: PayloadAction<LiveDeploymentConfigPage>) => {
            state.configurationPage = action.payload;
        },
        setConfigurationCondition: (state, action: PayloadAction<string>) => {
            state.configurationCondition = action.payload;
        },
        setDialogError: (state, action: PayloadAction<DeploymentServerDialogErrorAction | undefined>) => {
            if (action.payload) {
                state.deploymentServerDialogError = {
                    error: action.payload.status === 403 ? AUTH_ERROR_MESSAGE : OTHER_ERROR_MESSAGE,
                    errorDetail: action.payload.error,
                };
            } else {
                state.deploymentServerDialogError = undefined;
            }
        },
        setIsLoading: (state, action: PayloadAction<boolean>) => {
            state.isLoading = action.payload;
        },
        setConfigurationFolderId: (state, action: PayloadAction<string>) => {
            state.configurationPage.folderId = action.payload;
        },
        loadLiveDeploymentsConfiguration: (_state) => noop(),
        loadConnectionServerMetadata: (_state) => noop(),
        setConnectionServerMetadata: (state, action: PayloadAction<Array<ConnectionServerMetadata>>) => {
            state.connectionServerMetadata = action.payload;
            state.isLoadingConnectionServers = false;
        },
    },
});

export const getDeploymentServerState = (state: RootState): DeploymentServerState => state.folderDeploymentServers;
