import React from 'react';
import moment from 'moment';
import { ReactWrapper } from 'enzyme';
import { mountWithStoreAndTheme } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ExternalDeployment } from './external-deployment.component';
import {
    argoServerMock,
    deployServerMock,
    mockApplication,
    mockEnvironment,
    mockLiveDeploymentData,
    mockStatusWebhookEventSource,
} from '../__mocks__/external-deployments.mocks';
import { IMG_SRC } from '../constants';
import { pluginInfoArgo, pluginInfoDeploy } from '../__mocks__/external-deployments.mocks';

describe('ExternalDeployment component', () => {
    let wrapper: ReactWrapper;
    const openDeepLink = () => wrapper.find(`.link-column MenuItem[iconId="open-new-tab"]`);

    const openMenu = () => wrapper.find(`button[data-testid="options-button"]`);

    beforeEach(() => {
        wrapper = mountWithStoreAndTheme(
            <ExternalDeployment
                application={mockApplication}
                connectionServer={deployServerMock}
                environment={mockEnvironment}
                liveDeployment={mockLiveDeploymentData}
                pluginInfo={pluginInfoDeploy}
                statusWebhookEventSource={mockStatusWebhookEventSource}
            />,
        );
    });
    const searchDotCard = (selector: string) => wrapper.find(`.external-deployment-card ${selector}`);

    it('should show the server icon', () => {
        expect(searchDotCard('.icon-column .server-icon').props().src).toStrictEqual('icon');
    });

    it('should show the application name', () => {
        expect(searchDotCard('.application-name-column').text()).toStrictEqual(mockApplication.title);
        expect(searchDotCard('.application-name-column').props().title).toStrictEqual(mockApplication.title);
    });

    it('should show the environment icon and name deploy', () => {
        expect(searchDotCard('.environment-column .server-icon').props().src).toStrictEqual(IMG_SRC.environment);
        expect(searchDotCard('.environment-column > span').text()).toStrictEqual(mockEnvironment.title);
        expect(searchDotCard('.environment-column > span').props().title).toStrictEqual(mockEnvironment.title);
    });

    it('should show the version', () => {
        expect(searchDotCard('.version-column').text()).toStrictEqual(mockLiveDeploymentData.deploymentState.versionTag);
        expect(searchDotCard('.version-column .dot-link').at(0).prop('href')).toStrictEqual(mockLiveDeploymentData.deploymentState.versionLink);
    });

    it('should show the time passed from last update', () => {
        expect(searchDotCard('.last-updated-column').at(0).text()).toStrictEqual(moment(mockLiveDeploymentData.deploymentState.lastChangeTime).fromNow());
    });

    it('should show the menu icon', () => {
        expect(searchDotCard('.link-column .dot-i')).toBeTruthy();
    });

    it('should show menu icon items', () => {
        const menu = searchDotCard('#options-menu');
        expect(menu).toBeTruthy();
        openMenu().simulate('click');
        expect(searchDotCard('.link-column li').length).toBe(3);
        expect(searchDotCard('.link-column li').at(0).text()).toStrictEqual('View in deployment system');
        expect(searchDotCard('.link-column li').at(1).text()).toStrictEqual('View configuration details');
        expect(searchDotCard('.link-column li').at(2).text()).toStrictEqual('Delete application');
    });

    it('should handle on click of deep link', () => {
        window.open = jest.fn().mockReturnValue({});
        openMenu().simulate('click');
        openDeepLink().simulate('click');
        expect(window.open).toHaveBeenCalledWith(mockLiveDeploymentData.deploymentState.deploymentLink, '_blank');
    });

    it('should show the environment icon and name argo', () => {
        wrapper = mountWithStoreAndTheme(
            <ExternalDeployment
                application={mockApplication}
                connectionServer={argoServerMock}
                environment={mockEnvironment}
                liveDeployment={mockLiveDeploymentData}
                pluginInfo={pluginInfoArgo}
                statusWebhookEventSource={mockStatusWebhookEventSource}
            />,
        );
        expect(searchDotCard('.environment-column .server-icon').props().src).toStrictEqual(IMG_SRC.environment);
        expect(searchDotCard('.environment-column > span').text()).toStrictEqual(mockEnvironment.title);
        expect(searchDotCard('.environment-column > span').props().title).toStrictEqual(mockEnvironment.title);
    });
});
