import React, { KeyboardEvent, memo, useState } from 'react';
import Moment from 'react-moment';
import upperFirst from 'lodash/upperFirst';
import startCase from 'lodash/startCase';
import camelCase from 'lodash/camelCase';
import deburr from 'lodash/deburr';
import { DotCard, DotCardContent, DotCardHeader, DotDivider, DotIcon, DotIconButton, DotMenu, DotPill, DotTypography } from '@digital-ai/dot-components';
import { DeploymentState, PillStatus } from '../external-deployment.types';
import './external-deployment-card.component.less';
import {
    configureDeploymentServerApplicationMenuItems,
    ExternalDeploymentProps,
    onConfigureDeploymentServerApplication,
} from './external-deployment.component';
import { getPillIcon } from '../helper/utils';
import { useAppDispatch } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import { TextAreaMarkdown } from '../../../../../../../../../core/xlr-ui/app/react/components/markdown/textarea-markdown.component';
import { ConnectionDialog } from '../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/common/connection-dialog/connection-dialog.component';
import { folderDeploymentServers } from '../../deployment-server/ducks/deployment-server.reducer';
import { ExternalDeploymentDeleteApplicationDialog } from './external-deployment-delete-application-dialog.component';

const { reloadLiveDeploymentsConfiguration, runDeleteDeploymentServerApplication } = folderDeploymentServers.actions;

const ExternalDeploymentCardComponent = ({
    application,
    connectionServer: { type, title },
    environment,
    liveDeployment: { deploymentState, folderId, id },
    pluginInfo: pluginInfo,
    statusWebhookEventSource: { id: eventSourceId, type: eventSourceType },
}: ExternalDeploymentProps) => {
    const [open, setOpen] = useState(false);
    const [showDialog, setShowDialog] = useState(false);
    const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
    const [anchorEl, setAnchorEl] = useState<undefined | HTMLElement>(undefined);
    const dispatch = useAppDispatch();

    const renderConnectionDialog = () => {
        setShowDialog(true);
    };

    const handleCloseEditModal = () => {
        setShowDialog(false);
    };

    const handleDelete = () => {
        setIsDeleteDialogOpen(true);
    };

    const handleToggle = (event: React.MouseEvent<HTMLButtonElement, MouseEvent>) => {
        setAnchorEl(event.currentTarget);
        setOpen(!open);
    };

    const onItemSelect =
        (state: DeploymentState) => (_event: React.MouseEvent<Element, MouseEvent> | KeyboardEvent<Element>, _menuId: string, itemKey: string) => {
            setOpen(false);
            onConfigureDeploymentServerApplication(state, eventSourceId, itemKey, eventSourceType, renderConnectionDialog, handleDelete);
        };

    return (
        <DotCard className={`external-deployments-card ${deploymentState.statusGroup}`}>
            <DotCardHeader
                action={
                    <>
                        <div className="actions">
                            <DotIconButton color="default" iconId="options" onClick={handleToggle} />
                        </div>
                        <DotMenu
                            anchorEl={anchorEl}
                            disablePortal={true}
                            id="actions-menu"
                            menuItems={configureDeploymentServerApplicationMenuItems}
                            onLeave={() => setOpen(false)}
                            onSelect={onItemSelect(deploymentState)}
                            open={open}
                        />
                    </>
                }
                className="card-header"
                title={application.title}
                titleSize="large"
            />
            <DotCardContent>
                <div className="card-status-container">
                    <div className="card-status-item">
                        <DotPill
                            className="card-status-item-pill version-pill"
                            label={deploymentState.versionTag}
                            onClick={() => window.open(deploymentState.versionLink, '_blank')}
                            variant="outlined"
                        />
                    </div>
                    <div className="card-status-item">
                        <DotPill
                            className="card-status-item-pill"
                            icon={<DotIcon iconId={getPillIcon(deploymentState.statusGroup)} />}
                            label={upperFirst(camelCase(deburr(deploymentState.status)))}
                            status={deploymentState.statusGroup.replace('_', '-') as PillStatus}
                            variant="outlined"
                        />
                    </div>
                    <div className="card-status-item">
                        <div className="card-status-date">
                            <Moment fromNow interval={30000}>
                                {deploymentState.lastChangeTime}
                            </Moment>
                        </div>
                    </div>
                </div>
                <div className="card-server">
                    <img alt={type} className="server-icon" src={pluginInfo.iconLocation} />
                    <DotTypography variant="h3">{title}</DotTypography>
                </div>
                <div className="card-properties">
                    <DotTypography className="card-properties-title" variant="body1">
                        SOURCE
                    </DotTypography>
                    <DotDivider />
                    {Object.keys(application.applicationSource)
                        .filter((key) => !['id', 'type', 'serverUrl'].includes(key) && application.applicationSource[key])
                        .map((key) => {
                            return (
                                <div className="card-properties-item" key={key}>
                                    <DotTypography variant="body2">{startCase(camelCase(deburr(key)))}</DotTypography>
                                    <DotTypography variant="body1">
                                        <TextAreaMarkdown inputId={key} isReadOnly={true} name={key} rows={1} text={application.applicationSource[key]} />
                                    </DotTypography>
                                </div>
                            );
                        })}
                </div>
                <div className="card-properties">
                    <DotTypography className="card-properties-title" variant="body1">
                        TARGET
                    </DotTypography>
                    <DotDivider />
                    {Object.keys(environment.deploymentTarget)
                        .filter((key) => !['id', 'type', 'targetUrl'].includes(key) && environment.deploymentTarget[key])
                        .map((key) => {
                            return (
                                <div className="card-properties-item" key={key}>
                                    <DotTypography variant="body2">{startCase(camelCase(deburr(key)))}</DotTypography>
                                    <DotTypography variant="body1">
                                        <TextAreaMarkdown inputId={key} isReadOnly={true} name={key} rows={1} text={environment.deploymentTarget[key]} />
                                    </DotTypography>
                                </div>
                            );
                        })}
                </div>
            </DotCardContent>
            {showDialog && (
                <ConnectionDialog
                    configurationSaveAction={reloadLiveDeploymentsConfiguration}
                    connectionId={eventSourceId}
                    folderId={folderId}
                    isNewInstance={false}
                    keyOrName={eventSourceId}
                    onCancel={handleCloseEditModal}
                    onSave={handleCloseEditModal}
                    referencedType={eventSourceType}
                    showTestButton={false}
                />
            )}
            {isDeleteDialogOpen && (
                <ExternalDeploymentDeleteApplicationDialog
                    applicationName={application.title}
                    connectionLabel={title}
                    deploymentLink={deploymentState.deploymentLink}
                    isOpen={isDeleteDialogOpen}
                    onCancel={() => setIsDeleteDialogOpen(false)}
                    onSubmit={() => {
                        dispatch(
                            runDeleteDeploymentServerApplication({
                                configurationId: id,
                                connectionLabel: pluginInfo.title,
                                eventSourceId,
                                folderId,
                            }),
                        );
                        setIsDeleteDialogOpen(false);
                    }}
                />
            )}
        </DotCard>
    );
};

export const ExternalDeploymentCard = memo(ExternalDeploymentCardComponent);
