import React, { memo, useEffect, useRef, useState, KeyboardEvent } from 'react';
import Moment from 'react-moment';
import upperFirst from 'lodash/upperFirst';
import camelCase from 'lodash/camelCase';
import deburr from 'lodash/deburr';
import { DotCard, DotCardContent, DotIconButton, DotLink, DotMenu, DotPill, MenuItemProps, DotIcon } from '@digital-ai/dot-components';
import { ConnectionDialog } from '../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/common/connection-dialog/connection-dialog.component';
import {
    Server,
    LiveDeploymentData,
    Application,
    Environment,
    PluginInfo,
    StatusWebhookEventSource,
    DeploymentState,
    PillStatus,
} from '../external-deployment.types';
import { IMG_SRC } from '../constants';
import { getPillIcon } from '../helper/utils';
import './external-deployment.component.less';
import { MenuItem } from '../../../../../../../../../core/xlr-ui/app/react/components/menu-item/menu-item.component';
import { useAppDispatch } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import { AppDispatch } from '../../../../../../../../../core/xlr-ui/app/js/store.types';
import { folderDeploymentServers } from '../../deployment-server/ducks/deployment-server.reducer';

const { runDeleteDeploymentServerApplication, reloadLiveDeploymentsConfiguration } = folderDeploymentServers.actions;

export interface ExternalDeploymentProps {
    application: Application;
    connectionServer: Server;
    environment: Environment;
    liveDeployment: LiveDeploymentData;
    pluginInfo: PluginInfo;
    statusWebhookEventSource: StatusWebhookEventSource;
}

export const configureDeploymentServerApplicationMenuItems: MenuItemProps[] = [
    { children: <MenuItem iconId="open-new-tab" label="View in deployment system" />, key: 'view' },
    { children: <MenuItem iconId="configuration-preferences" label="View configuration details" />, key: 'details' },
    { children: <MenuItem iconId="delete" label="Delete application" />, key: 'delete' },
];

export const onConfigureDeploymentServerApplication = (
    state: DeploymentState,
    folderId: string,
    configurationId: string,
    eventSourceId: string,
    connectionLabel: string,
    dispatch: AppDispatch,
    itemKey: string,
    eventSourceType: string,
    renderConnectionDialog: (eventSourceId: string, eventSourceType: string) => void,
    // eslint-disable-next-line max-params
) => {
    switch (itemKey) {
        case 'view':
            window.open(state.deploymentLink, '_blank');
            break;
        case 'delete':
            dispatch(
                runDeleteDeploymentServerApplication({
                    connectionLabel,
                    eventSourceId,
                    configurationId,
                    folderId,
                }),
            );
            break;
        case 'details':
            renderConnectionDialog(eventSourceId, eventSourceType);
            break;
        default:
            break;
    }
};

const ExternalDeploymentComponent = ({
    application,
    connectionServer: { type },
    environment,
    liveDeployment: { deploymentState, folderId, id },
    pluginInfo: pluginInfo,
    statusWebhookEventSource: { id: eventSourceId, type: eventSourceType },
}: ExternalDeploymentProps) => {
    const [open, setOpen] = useState(false);
    const [showDialog, setShowDialog] = useState(false);
    const [anchorEl, setAnchorEl] = useState<undefined | HTMLElement>(undefined);
    const dispatch = useAppDispatch();
    const handleToggle = (event: React.MouseEvent<HTMLButtonElement, MouseEvent>) => {
        setAnchorEl(event.currentTarget);
        setOpen(!open);
    };

    const renderConnectionDialog = () => {
        setShowDialog(true);
    };

    const handleCloseEditModal = () => {
        setShowDialog(false);
    };

    // return focus to the button when we transitioned from !open -> open
    const prevOpen = useRef(open);
    useEffect(() => {
        if (anchorEl && prevOpen.current && !open) {
            anchorEl.focus();
        }
        prevOpen.current = open;
    }, [open]);

    const onItemSelect =
        (state: DeploymentState) => (_event: React.MouseEvent<Element, MouseEvent> | KeyboardEvent<Element>, _menuId: string, itemKey: string) => {
            setOpen(false);
            onConfigureDeploymentServerApplication(
                state,
                folderId,
                id,
                eventSourceId,
                pluginInfo.title,
                dispatch,
                itemKey,
                eventSourceType,
                renderConnectionDialog,
            );
        };

    return (
        <DotCard className="external-deployment-card" key={application.correlationUid + environment.correlationUid}>
            <div data-testid={'app' + application.title}>
                <DotCardContent>
                    <div className="icon-column">
                        <img alt={type} className="server-icon" src={pluginInfo.iconLocation} />
                    </div>
                    <div className="application-name-column" title={application.title}>
                        {application.title}
                    </div>
                    <div className="version-column">
                        <DotLink
                            data-testid={`versionLabel-${application.correlationUid}-${environment.title}`}
                            href={deploymentState.versionLink}
                            target="_blank"
                            tooltip={deploymentState.versionTooltip}
                        >
                            {deploymentState.versionTag}
                        </DotLink>
                    </div>
                    <div className="status-column">
                        <DotPill
                            className="card-status-item-pill"
                            icon={<DotIcon iconId={getPillIcon(deploymentState.statusGroup)} />}
                            label={upperFirst(camelCase(deburr(deploymentState.status)))}
                            size="small"
                            status={deploymentState.statusGroup.replace('_', '-') as PillStatus}
                            variant="outlined"
                        />
                    </div>
                    <div className="environment-column">
                        <img alt="environment" className="server-icon" src={IMG_SRC.environment} />
                        <span title={environment.title}>{environment.title}</span>
                    </div>
                    <div className="last-updated-column">
                        <Moment fromNow interval={30000}>
                            {deploymentState.lastChangeTime}
                        </Moment>
                    </div>
                    <div className="link-column">
                        <DotIconButton
                            data-testid={'options-button'}
                            iconId="options"
                            onClick={(e) => {
                                handleToggle(e);
                            }}
                        />
                        <DotMenu
                            anchorEl={anchorEl}
                            id="options-menu"
                            menuItems={configureDeploymentServerApplicationMenuItems}
                            onLeave={() => setOpen(false)}
                            onSelect={onItemSelect(deploymentState)}
                            open={open}
                        />
                    </div>
                </DotCardContent>
            </div>
            {showDialog && (
                <ConnectionDialog
                    configurationSaveAction={reloadLiveDeploymentsConfiguration}
                    connectionId={eventSourceId}
                    folderId={folderId}
                    isNewInstance={false}
                    keyOrName={eventSourceId}
                    onCancel={handleCloseEditModal}
                    onSave={handleCloseEditModal}
                    referencedType={eventSourceType}
                    showTestButton={false}
                />
            )}
        </DotCard>
    );
};

export const ExternalDeployment = memo(ExternalDeploymentComponent);
