import React, { useEffect, useMemo, useRef, useState } from 'react';
import {
    CheckboxProps,
    CssCell,
    CssGrid,
    DotAlertBanner,
    DotCard,
    DotCardContent,
    DotIcon,
    DotIconButton,
    DotInputText,
    DotTypography,
} from '@digital-ai/dot-components';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore we don't have types for xl-react-components
import { DeploymentServerCard } from './deployment-server-card.component';
import { ConnectionServerMetadata, DeploymentServer } from '../../deployment-server.types';
import { calculateCellProps } from '../../../../../../../../../../core/xlr-ui/app/features/workflow/helper';
import { NUMBER_OF_LOADING_CARD_SKELETONS } from '../../constants';
import { DeploymentServerSkeletons } from './skeletons/deployment-server-skeletons.component';
import { DeploymentServerDialogError, DeploymentServerSearch } from '../../ducks/deployment-server.reducer';
import { DeploymentServerCardSkeleton } from './skeletons/deployment-server-card-skeleton.component';
import './deployment-server-catalog.component.less';
import { DeploymentServerConnectionFilters } from './deployment-server-connection-filters.component';
import { IMG_SRC } from '../../../external-deployments/constants';

export interface DeploymentServerCatalogProps {
    addDeploymentServerHandler: () => void;
    connectionServers: ConnectionServerMetadata[];
    deploymentServerDialogError?: DeploymentServerDialogError;
    deploymentServers: DeploymentServer[];
    isLoadingConnectionServers: boolean;
    isLoadingDeploymentServers: boolean;
    onConnectionChange: (deploymentServerSearch: DeploymentServerSearch) => void;
    onFilter: (deploymentServerSearch: DeploymentServerSearch) => void;
    onUseDeploymentServer: (deploymentServer: DeploymentServer) => void;
    search?: DeploymentServerSearch;
}

export const DeploymentServerCatalog = ({
    addDeploymentServerHandler,
    connectionServers,
    deploymentServers,
    deploymentServerDialogError,
    isLoadingConnectionServers,
    isLoadingDeploymentServers,
    onConnectionChange,
    onFilter,
    onUseDeploymentServer,
    search,
}: DeploymentServerCatalogProps) => {
    const [currentDeploymentServers, setCurrentDeploymentServers] = useState<DeploymentServer[]>([]);
    const isLoading = isLoadingDeploymentServers || isLoadingConnectionServers;
    const predefinedSearchInput = search?.searchInput;
    const deploymentServerCardRef = useRef<HTMLDivElement | null>(null);
    const isInitialLoad = !currentDeploymentServers.length && isLoading;
    const [showErrorDetails, setShowErrorDetails] = useState(false);

    useEffect(() => {
        setCurrentDeploymentServers(deploymentServers);
    }, [deploymentServers]);

    const getDeploymentServerSearch = (currentDeploymentServerSearch: DeploymentServerSearch) =>
        search ? { ...search, ...currentDeploymentServerSearch } : currentDeploymentServerSearch;

    const memoizedDeploymentServerSkeletons = useMemo(() => (isLoading ? <DeploymentServerSkeletons /> : undefined), [isLoading]);

    const handleShowDetails = () => {
        setShowErrorDetails(!showErrorDetails);
    };

    const handleConnectionChange = (connectionsSelected: Array<CheckboxProps>) => {
        onConnectionChange(getDeploymentServerSearch({ connectionServers: connectionsSelected.map((conn) => conn.value as string) }));
    };

    const renderError = (message: string) => {
        return (
            <DotAlertBanner action={<DotIconButton iconId={showErrorDetails ? 'arrow-up' : 'arrow-down'} onClick={handleShowDetails} />} severity="error">
                <DotTypography>{message}</DotTypography>
                {showErrorDetails && (
                    <DotTypography>
                        <br />
                        {deploymentServerDialogError?.errorDetail}
                    </DotTypography>
                )}
            </DotAlertBanner>
        );
    };

    const renderDeploymentServers = () => {
        if (isInitialLoad) {
            return memoizedDeploymentServerSkeletons;
        } else {
            const zeroProps = calculateCellProps(0);
            return (
                <CssGrid>
                    <CssCell {...zeroProps} key="add-deployment-server">
                        <div className="add-deployment-server-button" onClick={addDeploymentServerHandler}>
                            <DotCard className="add-deployment-server-card">
                                <DotCardContent className="add-deployment-server-content">
                                    {/* use DotIllustration instead of img once version 2.0 is out */}
                                    {/*<DotIllustration className="add-deployment-server-illustration" illustrationId="digitalai-add-new" />*/}
                                    <img alt="Add new" className="add-deployment-server-illustration" src={IMG_SRC.addNewDeploymentServer} />
                                    <DotTypography variant="subtitle2">Add server</DotTypography>
                                </DotCardContent>
                            </DotCard>
                        </div>
                    </CssCell>
                    {currentDeploymentServers.map((currentDeploymentServer, index) => {
                        const props = calculateCellProps(index + 1);
                        return (
                            <CssCell {...props} key={currentDeploymentServer.connectionId}>
                                <DeploymentServerCard
                                    deploymentServer={currentDeploymentServer}
                                    onClick={() => onUseDeploymentServer(currentDeploymentServer)}
                                />
                            </CssCell>
                        );
                    })}
                    {isLoading &&
                        [...Array(NUMBER_OF_LOADING_CARD_SKELETONS)].map((_value, skeletonIndex) => {
                            const props = calculateCellProps(currentDeploymentServers.length + skeletonIndex);
                            return (
                                <CssCell {...props} key={currentDeploymentServers.length + skeletonIndex}>
                                    <DeploymentServerCardSkeleton />
                                </CssCell>
                            );
                        })}
                </CssGrid>
            );
        }
    };

    const deploymentServerCatalogFilters = (
        <DeploymentServerConnectionFilters
            connectionServers={connectionServers}
            deploymentServerSearch={search}
            isLoadingConnectionServers={isLoadingConnectionServers}
            onConnectionServersFilterChange={(options) => handleConnectionChange(options)}
        />
    );

    return (
        <CssGrid className="deployment-server-catalog">
            <CssCell
                center={false}
                className="deployment-server-drawer-content-left-cell"
                lg={{ start: 1, span: 3 }}
                md={{ start: 1, span: 3 }}
                sm={{ start: 1, span: 3 }}
                xl={{ start: 1, span: 3 }}
            >
                <div className="deployment-server-drawer-content-left">{deploymentServerCatalogFilters}</div>
            </CssCell>
            <CssCell
                center={false}
                className="tab-content-cell"
                lg={{ start: 4, span: 9 }}
                md={{ start: 4, span: 9 }}
                sm={{ start: 1, span: 12 }}
                xl={{ start: 4, span: 9 }}
                xs={{ start: 1, span: 12 }}
            >
                <div className="deployment-server-drawer-header-search">
                    {deploymentServerDialogError && renderError(deploymentServerDialogError?.error)}
                    <div className="search-row">
                        <DotInputText
                            hasDebounce={true}
                            id="search"
                            label="Search deployment server"
                            name="search"
                            onChange={(e) => onFilter(getDeploymentServerSearch({ searchInput: e.target.value }))}
                            persistentLabel={true}
                            placeholder="Start typing to filter deployment servers"
                            startIcon={<DotIcon data-testid="name-filter-search-icon" fontSize="small" iconId="search" />}
                            value={predefinedSearchInput}
                        />
                    </div>
                </div>
                <div className="deployment-server-cards" ref={deploymentServerCardRef}>
                    {renderDeploymentServers()}
                </div>
            </CssCell>
        </CssGrid>
    );
};
