import { all, call, CallEffect, put, select, takeEvery } from 'redux-saga/effects';
import { IHttpResponse } from 'angular';
import { AxiosError } from 'axios';
import { PayloadAction } from '@reduxjs/toolkit';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialPage } from './external-deployments.reducer';
import { ClientSettings, CustomConfiguration, FilterQueryParams, Folder } from '../../../../../../../../../core/xlr-ui/app/types';
import {
    ExternalDeploymentFilters,
    SaveWebhookSourceFiltersRequest,
    Server,
    SetupLiveUpdateRequest,
    WebhookEndpoint,
    WebhookSource,
    WebhookSourceAutoConfigDetails,
    WebhookSourceFilter,
    LiveDeploymentsPreconditionFilter,
    LiveDeploymentData,
    Environment,
    Application,
    StatusWebhookEventSource,
    ExternalDeploymentEvent,
    PluginInfo,
} from '../external-deployment.types';
import { mapOrderByValue } from '../helper/utils';
import { DEFAULT_MAX_AGE, EVENT_TYPE, STATUS_WEBHOOK_EVENT_SOURCE_TYPE, MAX_AGE_NO_CACHE, SSE_REGISTRATION_URL } from '../constants';
import { getFolderExternalDeployments, getWebhookSourcesSelector } from './external-deployments.selectors';
import ToastrFactory from '../../../../../../../../../core/xlr-ui/app/js/util/toastrFactory';
import { httpDELETE, httpGET, httpPOST, httpPUT } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { withFlagChangingState } from '../../../../../../../../../core/xlr-ui/app/react/utils/saga-utils';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { BreadcrumbService } from '../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import { CustomSSEEvent, sse } from '../../../../../../../../../core/xlr-ui/app/features/sse/sse.reducer';
import { CiTypeDescriptor, ConfigurationCallbackPayload, ConfigurationInstance } from '../../../../../../../../../core/xlr-ui/app/features/configuration/types';
import { IStateParamsService } from 'angular-ui-router';
import { Page } from '../../../../../../../../../core/xlr-ui/app/js/http/backend';
import { folderDeploymentServers } from '../../deployment-server/ducks/deployment-server.reducer';

const { broadcastSseEvent } = sse.actions;

const {
    createApplicationStatus,
    updateServerList,
    addValidPluginToArray,
    getAvailablePlugins,
    createServer,
    deleteApplicationStatus,
    loadExternalDeployments,
    loadWebhookSources,
    loadWebhookSourceFilters,
    loadServers,
    refreshExternalDeployments,
    setIsLoading,
    setConnectionServers,
    setApplications,
    setConfigDetails,
    setEnvironments,
    setLiveDeployments,
    setServers,
    setLiveDeploymentsCount,
    setMaxAge,
    setLiveUpdate,
    setWebhookSources,
    setWebhookSourceFilters,
    addServerToServers,
    subscribeToSseStream,
    unsubscribeFromSseStream,
    updateApplicationStatus,
    updateBreadcrumbs,
    updateWebhookSourceFilters,
    deleteWebhookSource,
    removeWebhookSource,
    setFilterEventSource,
    loadFilterEventSource,
    saveEventSource,
    storeFilters,
    getFilters,
    setPage,
    setCondition,
    setConnectionErrors,
    setDisconnectedServers,
    patchExternalDeployments,
    setPreSelectedServer,
    setLiveDeploymentsPreconditionFilter,
    setIsTableView,
    setStatusWebhookEventSources,
} = folderExternalDeployments.actions;

const { openLiveDeploymentConfiguration } = folderDeploymentServers.actions;
export const toaster = ToastrFactory();
const url = 'api/v1';

export function* executeLoadExternalDeploymentsWithRefreshAction() {
    yield put(setMaxAge(MAX_AGE_NO_CACHE));
    yield call(executeLoadExternalDeploymentsAction);
    yield put(setMaxAge(DEFAULT_MAX_AGE));
}

export function* executeDeleteApplicationPackageStatusAction(action: PayloadAction<ExternalDeploymentEvent>) {
    const { liveDeployments, page, condition, count }: FolderExternalDeploymentsState = yield select(getFolderExternalDeployments);
    const deleteEvent: ExternalDeploymentEvent = action.payload;

    const exists = liveDeployments.find(({ id }) => {
        return id === deleteEvent.deploymentId;
    });
    const updatedDeployments = exists
        ? liveDeployments.filter(({ id }) => {
              return id !== deleteEvent.deploymentId;
          })
        : [...liveDeployments];

    // when deleting we need to re-fetch if we have more deployments than fits the page, or we lose table row in UI
    if (exists && count > page.resultsPerPage && liveDeployments.length >= page.resultsPerPage) {
        yield call(executeLoadExternalDeploymentsWithRefreshAction);
    } else {
        // re-count live deployments
        if (page.folderId) {
            const { data: newCount }: IHttpResponse<number> = yield call(httpGET, `live-deployment/count?folderId=${page.folderId}&condition=${condition}`);
            yield put(setLiveDeploymentsCount(newCount));
        }

        yield put(setLiveDeployments(updatedDeployments));
    }
}

export function* executeCreateApplicationPackageStatusAction(action: PayloadAction<ExternalDeploymentEvent>) {
    const { liveDeployments, page, condition, applications, environments, connectionServers }: FolderExternalDeploymentsState = yield select(
        getFolderExternalDeployments,
    );
    const event: ExternalDeploymentEvent = action.payload;
    if (event.data) {
        const updatedDeployments = [...liveDeployments];
        const { data: application }: IHttpResponse<Application> = yield call(httpGET, `application-source/${event.data.application}`);
        const { data: environment }: IHttpResponse<Environment> = yield call(httpGET, `deployment-target/${event.data.environment}`);
        const { data: config }: IHttpResponse<StatusWebhookEventSource> = yield call(httpGET, `${url}/config/${event.data.eventSourceId}`);
        const { data: server }: IHttpResponse<Server> = yield call(httpGET, `${url}/config/${config.sourceServer}`);

        yield put(setApplications(new Map(applications).set(application.id, application)));
        yield put(setEnvironments(new Map(environments).set(environment.id, environment)));
        yield put(setConnectionServers(new Map(connectionServers).set(config.id, server)));

        // only push to current view if length < max per page
        if (updatedDeployments.length < page.resultsPerPage) updatedDeployments.push(event.data);

        // re-count live deployments
        if (page.folderId) {
            const { data: newCount }: IHttpResponse<number> = yield call(httpGET, `live-deployment/count?folderId=${page.folderId}&condition=${condition}`);
            yield put(setLiveDeploymentsCount(newCount));
        }
        yield put(setLiveDeployments(updatedDeployments));
    }
}

export function* executeUpdateApplicationPackageStatusAction(action: PayloadAction<ExternalDeploymentEvent>) {
    const { liveDeployments }: FolderExternalDeploymentsState = yield select(getFolderExternalDeployments);
    const event: ExternalDeploymentEvent = action.payload;

    if (event.data) {
        const updatedDeployments = [...liveDeployments];
        const index = updatedDeployments.findIndex(({ id }) => id === event.deploymentId);
        if (index !== -1) {
            // update existing deployment status
            updatedDeployments[index] = { ...updatedDeployments[index], ...event.data };
            yield put(setLiveDeployments(updatedDeployments));
        } else {
            // add new deployment status
            yield call(executeCreateApplicationPackageStatusAction, action);
        }
    }
}

export function* withLoadingState<R>(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        const result: R = yield effect;
        return result;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* executeLoadExternalDeploymentsAction() {
    yield call(withFlagChangingState, call(executeLoadExternalDeployments), setIsLoading);
}

export function* executeLoadExternalDeployments() {
    const { page, maxAge, condition }: FolderExternalDeploymentsState = yield select(getFolderExternalDeployments);
    if (!page || !page.folderId) {
        return;
    }
    try {
        const { data: connectionServerErrors }: IHttpResponse<object> = yield call(
            httpGET,
            `live-deployment?max-age=${maxAge}&folderId=${page.folderId}`,
            true,
        );
        const connectionServerErrorMap = new Map(Object.entries(connectionServerErrors).map(([key, value]) => [key, value]));
        const { data: liveDeployments }: IHttpResponse<Page<LiveDeploymentData>> = yield call(
            httpGET,
            `live-deployment/page?max-age=${maxAge}&folderId=${page.folderId}&page=${page.page}&resultsPerPage=${page.resultsPerPage}&orderBy=${mapOrderByValue(
                page.orderBy,
            )}&order=${page.order.toUpperCase()}&condition=${encodeURIComponent(condition)}`,
            true,
        );

        if (!condition && liveDeployments.totalElements === 0) {
            yield put(openLiveDeploymentConfiguration(true));
            return;
        }

        const { data: applications }: IHttpResponse<Array<Application>> = yield call(httpGET, `application-source/folder/${page.folderId}`);
        const { data: environments }: IHttpResponse<Array<Environment>> = yield call(httpGET, `deployment-target/folder/${page.folderId}`);
        const { data: configs }: IHttpResponse<Array<StatusWebhookEventSource>> = yield call(
            httpGET,
            `${url}/config/byTypeAndTitle?configurationType=${STATUS_WEBHOOK_EVENT_SOURCE_TYPE}&folderId=${page.folderId}&folderOnly=true`,
        );
        const { data: servers }: IHttpResponse<Array<Server>> = yield call(httpPOST, `${url}/config/byIds`, Array.from(configs.map((c) => c.sourceServer)));

        yield put(setStatusWebhookEventSources(configs));
        yield put(
            setConnectionServers(
                configs.reduce((map, config) => {
                    map.set(config.id as string, servers.find((server) => server.id === config.sourceServer) as Server);
                    return map;
                }, new Map<string, Server>()),
            ),
        );

        if (connectionServerErrorMap.size > 0) {
            yield put(setConnectionErrors(Array.from(connectionServerErrorMap.values()).filter((value) => value !== null) as string[]));
            yield put(setDisconnectedServers([...connectionServerErrorMap.keys()]));
        }
        yield put(
            setApplications(
                applications.reduce((map, app) => {
                    map.set(app.id, app);
                    return map;
                }, new Map<string, Application>()),
            ),
        );
        yield put(
            setEnvironments(
                environments.reduce((map, env) => {
                    map.set(env.id, env);
                    return map;
                }, new Map<string, Environment>()),
            ),
        );
        yield put(setLiveDeployments(liveDeployments.content));
        yield put(setLiveDeploymentsCount(liveDeployments.totalElements));
    } catch (e: unknown) {
        const err = e as AxiosError<CustomConfiguration, unknown>;
        const errServerData = err.response?.data;
        const errorMessage =
            'Error fetching status data. Check connection to ' + errServerData?.title + ' HTTP Connection or check application logs for more details.';
        yield call(toaster.error, errorMessage);
    }
}

export function* executeSseStreamSubscribeAction() {
    yield call(httpPOST, SSE_REGISTRATION_URL, {}, true);
}

export function* handleApplicationSseEventAction(action: PayloadAction<CustomSSEEvent>) {
    const { eventName, payload } = action.payload;
    switch (eventName) {
        case EVENT_TYPE.applicationCreated:
            yield put(createApplicationStatus(payload));
            break;
        case EVENT_TYPE.applicationDeleted:
            yield put(deleteApplicationStatus(payload));
            break;
        case EVENT_TYPE.applicationChanged:
            yield put(updateApplicationStatus(payload));
            break;
        default:
        //do nothing
    }
}

export function* executeSseStreamUnsubscribeAction() {
    yield call(httpDELETE, SSE_REGISTRATION_URL);
}

export function* executeCreateServerAction(action: PayloadAction<Server>) {
    const server = action.payload;
    const { data }: IHttpResponse<Server> = yield call(withLoadingState, call(httpPOST, `${url}/config`, server));
    yield call(toaster.success, `Server saved successfully`);
    yield put(addServerToServers(data));
    yield put(setPreSelectedServer(data));
}

export function* updateServerListAction(action: PayloadAction<ConfigurationCallbackPayload>) {
    const id = action.payload.id;
    const { data: server }: IHttpResponse<ConfigurationInstance> = yield call(httpGET, `api/v1/config/${id}`);
    const finalServerObj: Server = server as unknown as Server;
    yield call(toaster.success, `Server saved successfully`);
    yield put(addServerToServers(finalServerObj));
    yield put(setPreSelectedServer(finalServerObj));
}

export function* getAvailablePluginsAction(action: PayloadAction<string>) {
    const superType: string = action.payload;
    const { data: temp }: IHttpResponse<Array<WebhookSource>> = yield call(httpGET, `metadata/type`);

    const filteredObjects = (temp as unknown as CiTypeDescriptor[]).filter((object) => object.superTypes.includes(superType));
    for (const filteredObj of filteredObjects) {
        const iconLocation = (filteredObj.properties.find((obj) => obj.name === 'iconLocation')?.default as string) || '';
        const isAutoconfigurable = (filteredObj.properties.find((obj) => obj.name === 'isAutoconfigurable')?.default as boolean) || false;
        const subheader = (filteredObj.properties.find((obj) => obj.name === 'serverCardSubheader')?.default as string) || '';
        const title = (filteredObj.properties.find((obj) => obj.name === 'serverCardTitle')?.default as string) || '';

        if (subheader.length === 0 || title.length === 0) {
            continue;
        }

        const pluginInfo: PluginInfo = {
            iconLocation,
            isAutoconfigurable,
            subheader,
            title,
            type: filteredObj.type,
        };
        yield put(addValidPluginToArray(pluginInfo));
    }
}

export function* executeSetupLiveUpdateAction(action: PayloadAction<SetupLiveUpdateRequest>) {
    const folderId = action.payload.folderId;
    const serverId = action.payload.serverId;
    const { data }: IHttpResponse<WebhookSourceAutoConfigDetails> = yield call(
        withLoadingState,
        call(httpPOST, `live-deployment/setup/${serverId}?folderId=${folderId}`),
    );
    yield put(setConfigDetails(data));
    yield call(getWebhookSources, folderId);
}

export function* getWebhookSources(folderId: string) {
    const { data }: IHttpResponse<Array<WebhookSource>> = yield call(
        withLoadingState,
        call(httpGET, `${url}/config/byTypeAndTitle?configurationType=events.StatusWebhookEventSource&folderId=${folderId}&folderOnly=true`),
    );
    yield put(setWebhookSources(data));
}

export function* executeLoadWebhookSourceFiltersAction(action: PayloadAction<string>) {
    const webhookSourceId = action.payload;
    try {
        const { data: webhookSourceFilters }: IHttpResponse<WebhookSourceFilter> = yield call(
            withLoadingState,
            call(httpGET, `live-deployment/filters/${webhookSourceId}`),
        );
        yield put(setWebhookSourceFilters(webhookSourceFilters));
    } catch (e) {
        const err = e as AxiosError;
        yield call(toaster.error, err.message);
    }
}

export function* executeUpdateWebhookSourceFiltersAction(action: PayloadAction<SaveWebhookSourceFiltersRequest>) {
    const saveWebhookSourceFiltersRequest = action.payload;
    const { data: updatedSource }: IHttpResponse<WebhookSource> = yield call(
        withLoadingState,
        call(httpPUT, `live-deployment/filters/${saveWebhookSourceFiltersRequest.webhookSourceId}`, saveWebhookSourceFiltersRequest.filteredFolders),
    );
    const webhookSources: Array<WebhookSource> = yield select(getWebhookSourcesSelector);
    const updatedWebhookSources: Array<WebhookSource> = webhookSources.map((webhookSource): WebhookSource => {
        if (webhookSource.id === updatedSource.id) return updatedSource;
        return webhookSource;
    });
    yield put(setWebhookSources(updatedWebhookSources));
}

export function* executeLoadServersAction(action: PayloadAction<string>) {
    const folderId: string = action.payload;
    const { data: servers }: IHttpResponse<Array<Server>> = yield call(
        withLoadingState,
        call(httpGET, `${url}/config/byTypeAndTitle?configurationType=configuration.StatusHttpConnection&folderId=${folderId}`),
    );
    yield put(setServers(servers));
}

export function* executeLoadWebhookSourcesAction(action: PayloadAction<string>) {
    const folderId = action.payload;
    yield call(getWebhookSources, folderId);
}

export function* executeDeleteWebhookSourceAction(action: PayloadAction<string>) {
    const webhookSourceId = action.payload;
    yield call(withLoadingState, call(httpDELETE, `live-deployment/delete/${webhookSourceId}`));
    yield put(removeWebhookSource(webhookSourceId));
}

export function* executeLoadFilterEventSourceAction(action: PayloadAction<string>) {
    const eventSourceId = action.payload;
    const { data: eventSource }: IHttpResponse<WebhookEndpoint> = yield call(withLoadingState, call(httpGET, `${url}/config/${eventSourceId}`));
    yield put(setFilterEventSource(eventSource));
}

export function* executeSaveEventSourceAction(action: PayloadAction<WebhookEndpoint>) {
    const webhookEventSource = action.payload;
    yield call(withLoadingState, call(httpPUT, `${url}/config/${webhookEventSource.id}`, webhookEventSource));
    yield call(toaster.success, `Webhook endpoint updated successfully`);
}

export function* storeFiltersAction() {
    const { page, condition, isTableView } = yield select(getFolderExternalDeployments);
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    clientSettings.setExternalDeploymentsFilters({ ...page, condition, isTableView });
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    FiltersQueryParams.update({ ...page, condition, isTableView });
}

export function* getFiltersAction(action: PayloadAction<string>) {
    const folderId: string = action.payload;
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    const filters: ExternalDeploymentFilters = clientSettings.getExternalDeploymentsFilters();

    const stateParams: IStateParamsService = yield call(getAngularService, '$stateParams');
    const condition = stateParams.condition;
    if (condition) filters.condition = condition;

    if (filters) {
        yield put(setCondition(filters.condition));
        yield put(
            setPage({
                folderId,
                page: filters.page,
                order: filters.order,
                orderBy: filters.orderBy,
                resultsPerPage: filters.resultsPerPage,
            }),
        );
        yield put(setIsTableView(filters.isTableView));
    }
}

export function* updateBreadcrumbsAction(action: PayloadAction<Folder>) {
    const folder = action.payload;

    const breadcrumbs = [
        {
            pathSuffix: 'folders/' + folder.id,
            label: folder.title,
            folder,
        },
        {
            pathSuffix: 'folders/' + folder.id + '/external-deployments',
            label: 'Application deployments',
            folder,
        },
        {
            label: 'Application Sources',
        },
    ];

    const breadcrumbService: BreadcrumbService = yield call(getAngularService, 'BreadcrumbsService');
    breadcrumbService.updateBreadcrumbs(breadcrumbs, false);
}

export function* executePatchExternalDeploymentsAction(action: PayloadAction<string>) {
    const webhookSourceId = action.payload;
    yield call(withLoadingState, call(httpPOST, `live-deployment/patch/${webhookSourceId}`));
    yield call(toaster.success, `External deployments patched successfully`);
}

export function* setLiveDeploymentsPreconditionFilterAction(action: PayloadAction<LiveDeploymentsPreconditionFilter>) {
    const folderId = action.payload.folderId;
    const condition = action.payload.applicationName;

    yield put(setPage({ ...initialPage, folderId }));
    yield put(setCondition(condition));

    yield call(storeFiltersAction);
}

export default function* externalDeploymentsRootSaga() {
    yield all([
        takeEvery(createApplicationStatus, executeCreateApplicationPackageStatusAction),
        takeEvery(createServer, executeCreateServerAction),
        takeEvery(updateServerList, updateServerListAction),
        takeEvery(getAvailablePlugins, getAvailablePluginsAction),
        takeEvery(deleteApplicationStatus, executeDeleteApplicationPackageStatusAction),
        takeEvery(deleteWebhookSource, executeDeleteWebhookSourceAction),
        takeEvery(getFilters, getFiltersAction),
        takeEvery(loadExternalDeployments, executeLoadExternalDeploymentsAction),
        takeEvery(loadFilterEventSource, executeLoadFilterEventSourceAction),
        takeEvery(loadServers, executeLoadServersAction),
        takeEvery(loadWebhookSourceFilters, executeLoadWebhookSourceFiltersAction),
        takeEvery(loadWebhookSources, executeLoadWebhookSourcesAction),
        takeEvery(patchExternalDeployments, executePatchExternalDeploymentsAction),
        takeEvery(refreshExternalDeployments, executeLoadExternalDeploymentsWithRefreshAction),
        takeEvery(saveEventSource, executeSaveEventSourceAction),
        takeEvery(setLiveUpdate, executeSetupLiveUpdateAction),
        takeEvery(storeFilters, storeFiltersAction),
        takeEvery(subscribeToSseStream, executeSseStreamSubscribeAction),
        takeEvery(unsubscribeFromSseStream, executeSseStreamUnsubscribeAction),
        takeEvery(updateApplicationStatus, executeUpdateApplicationPackageStatusAction),
        takeEvery(updateBreadcrumbs, updateBreadcrumbsAction),
        takeEvery(updateWebhookSourceFilters, executeUpdateWebhookSourceFiltersAction),
        takeEvery(broadcastSseEvent, handleApplicationSseEventAction),
        takeEvery(setLiveDeploymentsPreconditionFilter, setLiveDeploymentsPreconditionFilterAction),
    ]);
}
