import { SagaIterator } from 'redux-saga';
import { all, call, put, race, select, take } from 'redux-saga/effects';
import { PayloadAction } from '@reduxjs/toolkit';
import { IStateService } from 'angular-ui-router';
import { httpDELETE, httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { withFlagChangingState } from '../../../../../../../../../core/xlr-ui/app/react/utils/saga-utils';
import { Page } from '../../../../../../../../../core/xlr-ui/app/js/http/backend';
import { DeploymentServer, LiveDeploymentConfigData } from '../deployment-server.types';
import { mapConfigOrderByValue } from '../helper/utils';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import {
    deploymentServerConfigurationSaga,
    executeLoadLiveDeploymentConfigurationsAction,
    executeLoadLiveDeploymentsConfiguration,
    fetchConnectionServerMetadata,
    filterDeploymentServersSaga,
    findConnectionServerMetadata,
    getFilteredDeploymentServers,
    goBackToLiveDeployments,
    goToLiveDeploymentConfiguration,
    handleError,
    initSaga,
    runAddDeploymentServerSaga,
    runDeleteDeploymentServerSaga,
    runEditDeploymentServerSaga,
    runUseDeploymentServerSaga,
    searchDeploymentServers,
    setConfigurationConditionAction,
    setConfigurationPageAction,
    withLoadingState,
} from './deployment-server.saga';
import {
    ConfigureDeploymentServer,
    DeploymentServerSearch,
    DeploymentServerState,
    folderDeploymentServers,
    getDeploymentServerState,
} from './deployment-server.reducer';
import { STATUS_HTTP_CONNECTION, STATUS_WEBHOOK_EVENT_SOURCE_TYPE } from '../../external-deployments/constants';
import { getConnectionServerMetadata } from './deployment-server.selectors';
import { RunWorkflowAction, workflow } from '../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import {
    argoServerMock,
    mockArgoConnectionServerMetadata,
    mockArgoDeploymentServer,
    mockArgoWebhookSource,
    mockDeployConnectionServerMetadata,
    mockDeployDeploymentServer,
    mockLiveDeploymentConfigData,
} from '../__mocks__/deployment-server.mocks';
import { StatusWebhookEventSource } from '../../external-deployments/external-deployment.types';
import {
    createOrUpdateVariable,
    getReleaseVariables,
    updateReleaseVariable,
} from '../../../../../../../../../core/xlr-ui/app/features/tasks/ducks/variables-service.saga';
import ToastrFactory from '../../../../../../../../../core/xlr-ui/app/js/util/toastrFactory';
import { Variable } from '../../../../../../../../../core/xlr-ui/app/types';
import { AUTH_ERROR_MESSAGE, CREATE_DEPLOYMENT_PROVIDER_WORKFLOWS_TITLE, DEFAULT_ADD_DEPLOYMENT_PROVIDER_SEARCH_CATEGORY } from '../constants';
import angular from 'angular';
import IdsFactory from '../../../../../../../../../core/xlr-ui/app/js/util/ids';
import { searchWorkflowIdByTags } from '../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.saga';

const {
    openDeploymentServerDrawer,
    setConnectionServerMetadata,
    setDeploymentServers,
    setDeploymentServerSearch,
    setDialogError,
    setIsLoading,
    setLiveDeploymentConfigs,
    setLiveDeploymentConfigsCount,
} = folderDeploymentServers.actions;

const { createWorkflowError, createWorkflowSuccess, initCondenseViewDrawer, runWorkflow } = workflow.actions;

const toaster = ToastrFactory();
const Ids = IdsFactory();

describe('initSaga', () => {
    it('should initialize saga and open deployment server drawer', () => {
        const action = { payload: {} } as PayloadAction<Partial<DeploymentServerSearch>>;
        const gen: SagaIterator = initSaga(action);
        const connectionServerTypes = [mockArgoConnectionServerMetadata, mockDeployConnectionServerMetadata];
        const deploymentServers: DeploymentServer[] = [mockDeployDeploymentServer];

        expect(gen.next().value).toStrictEqual(call(findConnectionServerMetadata));
        expect(gen.next(connectionServerTypes).value).toStrictEqual(put(setConnectionServerMetadata(connectionServerTypes)));
        expect(gen.next().value).toStrictEqual(call(searchDeploymentServers, action.payload));
        expect(gen.next().value).toStrictEqual(select(getDeploymentServerState));
        expect(gen.next({ deploymentServers }).value).toStrictEqual(put(openDeploymentServerDrawer()));
        expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
        expect(gen.next().done).toBeTruthy();
    });
    it('should initialize saga and open workflow drawer if no deploymentServers are fetched', () => {
        const action = { payload: {} } as PayloadAction<Partial<DeploymentServerSearch>>;
        const gen: SagaIterator = initSaga(action);
        const connectionServerTypes = [mockArgoConnectionServerMetadata, mockDeployConnectionServerMetadata];
        const deploymentServers: DeploymentServer[] = [];

        expect(gen.next().value).toStrictEqual(call(findConnectionServerMetadata));
        expect(gen.next(connectionServerTypes).value).toStrictEqual(put(setConnectionServerMetadata(connectionServerTypes)));
        expect(gen.next().value).toStrictEqual(call(searchDeploymentServers, action.payload));
        expect(gen.next().value).toStrictEqual(select(getDeploymentServerState));
        expect(gen.next({ deploymentServers }).value).toStrictEqual(
            put(
                initCondenseViewDrawer({
                    catalogTitle: CREATE_DEPLOYMENT_PROVIDER_WORKFLOWS_TITLE,
                    workflowSearch: { categories: [DEFAULT_ADD_DEPLOYMENT_PROVIDER_SEARCH_CATEGORY] },
                }),
            ),
        );
        expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('getFilteredDeploymentServers', () => {
    it('should filter deployment servers based on search input and connection servers', () => {
        const servers = [argoServerMock];
        const connectionServerTypes = [mockArgoConnectionServerMetadata, mockDeployConnectionServerMetadata];
        const sourceToServerMap = { [mockArgoWebhookSource.sourceServer]: mockArgoWebhookSource };
        const deploymentServerSearch = { searchInput: 'argocd', connectionServers: ['argocd.ArgoCD'] } as DeploymentServerSearch;

        const gen = getFilteredDeploymentServers(sourceToServerMap, deploymentServerSearch);

        expect(gen.next().value).toStrictEqual(call(httpPOST, `api/v1/config/byIds`, Object.keys(sourceToServerMap)));
        expect(gen.next({ data: servers } as never).value).toStrictEqual(select(getConnectionServerMetadata));
        expect(gen.next(connectionServerTypes as never).value).toStrictEqual(
            servers
                .map((server) => {
                    const source = sourceToServerMap[server.id as string];
                    const metadata = connectionServerTypes.find((type) => type.type === server.type);
                    return {
                        connectionIconLocation: metadata?.iconLocation,
                        connectionId: server.id as string,
                        connectionLabel: metadata?.title,
                        connectionTitle: server?.title,
                        connectionType: metadata?.type,
                        connectionUrl: server.url,
                        eventSourceId: source.id,
                        eventSourceTitle: source.title,
                    };
                })
                .filter((provider) => {
                    const searchInput = deploymentServerSearch.searchInput?.toLowerCase();
                    const connectionServers = deploymentServerSearch.connectionServers;
                    const connectionTitle = provider.connectionTitle?.toLowerCase();
                    const connectionLabel = provider.connectionLabel?.toLowerCase();
                    const connectionUrl = provider.connectionUrl.toLowerCase();
                    const connectionType = provider.connectionType?.toLowerCase();
                    const eventSourceTitle = provider.eventSourceTitle.toLowerCase();
                    return (
                        (!searchInput ||
                            connectionTitle?.includes(searchInput) ||
                            connectionLabel?.includes(searchInput) ||
                            connectionUrl.includes(searchInput) ||
                            connectionType?.includes(searchInput) ||
                            eventSourceTitle.includes(searchInput)) &&
                        (!connectionServers || connectionServers.length === 0 || connectionServers.includes(provider.connectionType as string))
                    );
                }),
        );
        expect(gen.next().done).toBeTruthy();
    });
});
describe('searchDeploymentServers', () => {
    it('should search deployment servers by type', () => {
        const deploymentServerSearch = {
            folderId: 'Applications/FolderSamplesAndTutorials',
            connectionServers: ['argocd.ArgoCD'],
        } as DeploymentServerSearch;
        const gen: SagaIterator = searchDeploymentServers(deploymentServerSearch);
        const configs = [mockArgoWebhookSource];
        const sourceToServerMap: Record<string, StatusWebhookEventSource> = { [mockArgoWebhookSource.sourceServer]: mockArgoWebhookSource };

        expect(gen.next().value).toStrictEqual(put(setDeploymentServerSearch(deploymentServerSearch)));
        expect(gen.next().value).toStrictEqual(
            call(
                httpGET,
                `api/v1/config/byTypeAndTitle?configurationType=${STATUS_WEBHOOK_EVENT_SOURCE_TYPE}&folderId=${deploymentServerSearch.folderId}&folderOnly=true`,
            ),
        );
        expect(gen.next({ data: configs }).value).toStrictEqual(call(getFilteredDeploymentServers, sourceToServerMap, deploymentServerSearch));
        expect(gen.next([mockArgoDeploymentServer]).value).toStrictEqual(put(setDeploymentServers([mockArgoDeploymentServer])));
        expect(gen.next().done).toBeTruthy();
    });

    it('should search deployment servers by search condition', () => {
        const deploymentServerSearch = { folderId: 'Applications/FolderSamplesAndTutorials', searchInput: 'argocd' } as DeploymentServerSearch;
        const gen: SagaIterator = searchDeploymentServers(deploymentServerSearch);
        const configs = [mockArgoWebhookSource];
        const sourceToServerMap: Record<string, StatusWebhookEventSource> = { [mockArgoWebhookSource.sourceServer]: mockArgoWebhookSource };

        expect(gen.next().value).toStrictEqual(put(setDeploymentServerSearch(deploymentServerSearch)));
        expect(gen.next().value).toStrictEqual(
            call(
                httpGET,
                `api/v1/config/byTypeAndTitle?configurationType=${STATUS_WEBHOOK_EVENT_SOURCE_TYPE}&folderId=${deploymentServerSearch.folderId}&folderOnly=true`,
            ),
        );
        expect(gen.next({ data: configs }).value).toStrictEqual(call(getFilteredDeploymentServers, sourceToServerMap, deploymentServerSearch));
        expect(gen.next([mockArgoDeploymentServer]).value).toStrictEqual(put(setDeploymentServers([mockArgoDeploymentServer])));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('withLoadingState', () => {
    it('should handle loading state', () => {
        const effect = call(() => 'result');
        const gen: SagaIterator = withLoadingState(effect);

        expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
        expect(gen.next().value).toStrictEqual(effect);
        expect(gen.next('result').value).toStrictEqual(put(setIsLoading(false)));
        expect(gen.next('result').value).toStrictEqual('result');
        expect(gen.next().done).toBeTruthy();
    });
});

describe('executeLoadLiveDeploymentConfigurationsAction', () => {
    it('should execute load live deployment configurations action', () => {
        const gen: SagaIterator = executeLoadLiveDeploymentConfigurationsAction();

        expect(gen.next().value).toStrictEqual(call(withFlagChangingState, call(executeLoadLiveDeploymentsConfiguration), setIsLoading));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('executeLoadLiveDeploymentsConfiguration', () => {
    it('should execute load live deployments configuration', () => {
        const state = {
            configurationPage: { folderId: 'folderId', page: 0, resultsPerPage: 10, orderBy: 'orderBy', order: 'asc' },
            configurationCondition: 'condition',
        } as DeploymentServerState;
        const configs = { content: [mockLiveDeploymentConfigData], totalElements: 1 } as Page<LiveDeploymentConfigData>;
        const gen: SagaIterator = executeLoadLiveDeploymentsConfiguration();

        expect(gen.next().value).toStrictEqual(select(getDeploymentServerState));
        expect(gen.next(state).value).toStrictEqual(
            call(
                httpGET,
                `live-deployment/configs?folderId=${state.configurationPage.folderId}&page=${state.configurationPage.page}&resultsPerPage=${
                    state.configurationPage.resultsPerPage
                }&orderBy=${mapConfigOrderByValue(
                    state.configurationPage.orderBy,
                )}&order=${state.configurationPage.order.toUpperCase()}&condition=${encodeURIComponent(state.configurationCondition)}`,
                true,
            ),
        );
        expect(gen.next({ data: configs }).value).toStrictEqual(put(setLiveDeploymentConfigs(configs.content)));
        expect(gen.next().value).toStrictEqual(put(setLiveDeploymentConfigsCount(configs.totalElements)));
        expect(gen.next().done).toBeTruthy();
    });

    it('should handle axios error and show toaster error', () => {
        const state = {
            configurationPage: { folderId: 'folderId', page: 0, resultsPerPage: 10, orderBy: 'orderBy', order: 'asc' },
            configurationCondition: 'condition',
        } as DeploymentServerState;
        const error = {
            isAxiosError: true,
            response: {
                data: { title: 'Server' },
                status: 403,
            },
        };
        const gen: SagaIterator = executeLoadLiveDeploymentsConfiguration();

        expect(gen.next().value).toStrictEqual(select(getDeploymentServerState));
        expect(gen.next(state).value).toStrictEqual(
            call(
                httpGET,
                `live-deployment/configs?folderId=${state.configurationPage.folderId}&page=${state.configurationPage.page}&resultsPerPage=${
                    state.configurationPage.resultsPerPage
                }&orderBy=${mapConfigOrderByValue(
                    state.configurationPage.orderBy,
                )}&order=${state.configurationPage.order.toUpperCase()}&condition=${encodeURIComponent(state.configurationCondition)}`,
                true,
            ),
        );
        // using stringify because some strange thing about jest and serialization https://github.com/jestjs/jest/issues/8475
        // eslint-disable-next-line angular/json-functions
        expect(angular.toJson(gen.throw?.(error).value)).toStrictEqual(
            // eslint-disable-next-line angular/json-functions
            angular.toJson(
                call(toaster.error, 'Error fetching status data. Check connection to Server HTTP Connection or check application logs for more details.'),
            ),
        );
        expect(gen.next().done).toBeTruthy();
    });
});

describe('findConnectionServerMetadata', () => {
    it('should find connection server metadata', () => {
        const temp = [
            {
                superTypes: [STATUS_HTTP_CONNECTION],
                properties: [
                    { name: 'iconLocation', default: 'icon-location' },
                    { name: 'serverCardSubheader', default: 'subheader' },
                    { name: 'serverCardTitle', default: 'title' },
                ],
                type: 'type',
            },
        ];
        const connectionServerMetadata = [{ iconLocation: 'icon-location', subheader: 'subheader', title: 'title', type: 'type' }];
        const gen: SagaIterator = findConnectionServerMetadata();

        expect(gen.next().value).toStrictEqual(call(withLoadingState, call(httpGET, `metadata/type`)));
        expect(gen.next({ data: temp }).value).toStrictEqual(
            temp
                .filter((object) => object.superTypes.includes(STATUS_HTTP_CONNECTION))
                .map((filteredObj) => {
                    const iconLocation = (filteredObj.properties.find((obj) => obj.name === 'iconLocation')?.default as string) || '';
                    const subheader = (filteredObj.properties.find((obj) => obj.name === 'serverCardSubheader')?.default as string) || '';
                    const title = (filteredObj.properties.find((obj) => obj.name === 'serverCardTitle')?.default as string) || '';

                    return {
                        subheader,
                        title,
                        iconLocation,
                        type: filteredObj.type,
                    };
                }),
        );
        expect(gen.next(connectionServerMetadata).done).toBeTruthy();
    });
});

describe('startUseDeploymentServerSaga', () => {
    it('should start use deployment servers saga', () => {
        const action = {
            payload: {
                connectionLabel: mockArgoDeploymentServer.connectionLabel,
                folderId: 'folderId',
                eventSourceId: mockArgoDeploymentServer.eventSourceId,
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen: SagaIterator = runUseDeploymentServerSaga(action);
        expect(gen.next().value).toStrictEqual(call(deploymentServerConfigurationSaga, action));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('runEditDeploymentServerSaga', () => {
    it('should start use deployment servers saga', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: mockArgoDeploymentServer.connectionLabel,
                folderId: 'folderId',
                eventSourceId: mockArgoDeploymentServer.eventSourceId,
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen: SagaIterator = runEditDeploymentServerSaga(action);
        expect(gen.next().value).toStrictEqual(call(deploymentServerConfigurationSaga, action));
        expect(gen.next().done).toBeTruthy();
    });
    it('should handle successful execution', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: 'connectionLabel',
                folderId: 'folderId',
                eventSourceId: 'eventSourceId',
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen = runEditDeploymentServerSaga(action);

        expect(gen.next().value).toStrictEqual(call(deploymentServerConfigurationSaga, action));
        expect(gen.next().done).toBeTruthy();
    });

    it('should handle generic error', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: 'connectionLabel',
                folderId: 'folderId',
                eventSourceId: 'eventSourceId',
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen = runEditDeploymentServerSaga(action);
        const error = new Error('generic error');

        gen.next();
        // using stringify because some strange thing about jest and serialization https://github.com/jestjs/jest/issues/8475
        // eslint-disable-next-line angular/json-functions
        expect(angular.toJson(gen.throw?.(error).value)).toStrictEqual(angular.toJson(call(toaster.error, 'generic error')));
        expect(gen.next().done).toBeTruthy();
    });

    it('should handle Axios error with 403 status', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: 'connectionLabel',
                folderId: 'folderId',
                eventSourceId: 'eventSourceId',
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen = runEditDeploymentServerSaga(action);
        const error = {
            isAxiosError: true,
            response: {
                status: 403,
            },
        };

        gen.next();
        expect(angular.toJson(gen.throw(error).value)).toStrictEqual(angular.toJson(call(toaster.error, AUTH_ERROR_MESSAGE)));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('runDeleteDeploymentServerSaga', () => {
    it('should handle successful execution', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: mockArgoDeploymentServer.connectionLabel,
                folderId: 'folderId',
                eventSourceId: mockArgoDeploymentServer.eventSourceId,
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const workflowId = 'workflowId';
        const variables: Array<Variable> = [
            { id: 'a-var-1', key: 'statusWebhookEventSourceId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
        ];
        const updatedWorkflowVariables: Array<Variable> = [
            { id: 'a-var-1', key: 'statusWebhookEventSourceId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
            { id: 'a-var-2', key: 'liveDeploymentConfigurationId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
        ];

        const gen = runDeleteDeploymentServerSaga(action);

        expect(gen.next().value).toStrictEqual(
            call(searchWorkflowIdByTags, [mockArgoDeploymentServer.connectionLabel.toLowerCase(), 'deployment server', 'delete', 'internal']),
        );
        expect(gen.next(workflowId as never).value).toStrictEqual(call(getReleaseVariables, workflowId, true));
        expect(gen.next(variables as never).value).toStrictEqual(
            call(
                createOrUpdateVariable,
                workflowId,
                variables,
                'statusWebhookEventSourceId',
                mockArgoDeploymentServer.eventSourceId,
                'xlrelease.StringVariable',
            ),
        );
        expect(gen.next(updatedWorkflowVariables[0] as never).value).toStrictEqual(
            call(createOrUpdateVariable, workflowId, variables, 'liveDeploymentConfigurationId', 'configId', 'xlrelease.StringVariable'),
        );
        expect(gen.next(updatedWorkflowVariables[1] as never).value).toStrictEqual(
            put(runWorkflow({ folderId: Ids.toDomainId(action.payload.folderId), workflow: { id: workflowId } } as RunWorkflowAction)),
        );
        expect(gen.next().value).toStrictEqual(
            race({
                error: take(createWorkflowError.type),
                success: take(createWorkflowSuccess.type),
            }),
        );
        expect(gen.next().value).toStrictEqual(all(updatedWorkflowVariables.map((v) => call(updateReleaseVariable, { ...v, value: '' }))));
        expect(gen.next().done).toBeTruthy();
    });

    it('should handle configs without source', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: mockArgoDeploymentServer.connectionLabel,
                folderId: 'folderId',
                eventSourceId: '',
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen = runDeleteDeploymentServerSaga(action);

        expect(gen.next().value).toStrictEqual(call(httpDELETE, 'api/v1/config/configId'));
        expect(gen.next().value).toStrictEqual(call(executeLoadLiveDeploymentConfigurationsAction));
        expect(gen.next().done).toBeTruthy();
    });

    it('should handle generic error', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: 'connectionLabel',
                folderId: 'folderId',
                eventSourceId: 'eventSourceId',
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen = runDeleteDeploymentServerSaga(action);
        const error = new Error('generic error');

        gen.next();
        expect(angular.toJson(gen.throw(error).value)).toStrictEqual(angular.toJson(call(toaster.error, 'generic error')));
        expect(gen.next().done).toBeTruthy();
    });

    it('should handle Axios error with 403 status', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: 'connectionLabel',
                folderId: 'folderId',
                eventSourceId: 'eventSourceId',
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const gen = runDeleteDeploymentServerSaga(action);
        const error = {
            isAxiosError: true,
            response: {
                status: 403,
            },
        };

        gen.next();
        expect(angular.toJson(gen.throw(error).value)).toStrictEqual(angular.toJson(call(toaster.error, AUTH_ERROR_MESSAGE)));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('deploymentServerConfigurationSaga', () => {
    it('should start configuration of deployment server to add config', () => {
        const action = {
            payload: {
                connectionLabel: mockArgoDeploymentServer.connectionLabel,
                folderId: 'folderId',
                eventSourceId: mockArgoDeploymentServer.eventSourceId,
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const workflowId = 'workflowId';
        const variables: Array<Variable> = [
            { id: 'a-var-1', key: 'statusWebhookEventSourceId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
        ];
        const updatedWorkflowVariables: Array<Variable> = [
            { id: 'a-var-1', key: 'statusWebhookEventSourceId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
        ];
        const gen: SagaIterator = deploymentServerConfigurationSaga(action);

        expect(gen.next().value).toStrictEqual(
            call(searchWorkflowIdByTags, [mockArgoDeploymentServer.connectionLabel.toLowerCase(), 'deployment server', 'config', 'internal']),
        );
        expect(gen.next(workflowId).value).toStrictEqual(call(getReleaseVariables, workflowId, true));
        expect(gen.next(variables).value).toStrictEqual(
            call(
                createOrUpdateVariable,
                workflowId,
                variables,
                'statusWebhookEventSourceId',
                mockArgoDeploymentServer.eventSourceId,
                'xlrelease.StringVariable',
            ),
        );
        expect(gen.next(updatedWorkflowVariables[0]).value).toStrictEqual(
            put(runWorkflow({ folderId: Ids.toDomainId(action.payload.folderId), workflow: { id: workflowId } } as RunWorkflowAction)),
        );
        expect(gen.next().value).toStrictEqual(
            race({
                error: take(createWorkflowError.type),
                success: take(createWorkflowSuccess.type),
            }),
        );
        expect(gen.next().value).toStrictEqual(all(updatedWorkflowVariables.map((v) => call(updateReleaseVariable, { ...v, value: '' }))));
        expect(gen.next().done).toBeTruthy();
    });
    it('should start configuration of deployment server to edit config', () => {
        const action = {
            payload: {
                configurationId: 'configId',
                connectionLabel: mockArgoDeploymentServer.connectionLabel,
                folderId: 'folderId',
                eventSourceId: mockArgoDeploymentServer.eventSourceId,
            },
        } as PayloadAction<ConfigureDeploymentServer>;
        const workflowId = 'workflowId';
        const variables: Array<Variable> = [
            { id: 'a-var-1', key: 'statusWebhookEventSourceId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
        ];
        const updatedWorkflowVariables: Array<Variable> = [
            { id: 'a-var-1', key: 'statusWebhookEventSourceId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
            { id: 'a-var-2', key: 'liveDeploymentConfigurationId', requiresValue: false, showOnReleaseStart: false, type: 'xlrelease.StringVariable' },
        ];
        const gen: SagaIterator = deploymentServerConfigurationSaga(action);

        expect(gen.next().value).toStrictEqual(
            call(searchWorkflowIdByTags, [mockArgoDeploymentServer.connectionLabel.toLowerCase(), 'deployment server', 'config', 'internal']),
        );
        expect(gen.next(workflowId).value).toStrictEqual(call(getReleaseVariables, workflowId, true));
        expect(gen.next(variables).value).toStrictEqual(
            call(
                createOrUpdateVariable,
                workflowId,
                variables,
                'statusWebhookEventSourceId',
                mockArgoDeploymentServer.eventSourceId,
                'xlrelease.StringVariable',
            ),
        );
        expect(gen.next(updatedWorkflowVariables[0]).value).toStrictEqual(
            call(createOrUpdateVariable, workflowId, variables, 'liveDeploymentConfigurationId', 'configId', 'xlrelease.StringVariable'),
        );
        expect(gen.next(updatedWorkflowVariables[1]).value).toStrictEqual(
            put(runWorkflow({ folderId: Ids.toDomainId(action.payload.folderId), workflow: { id: workflowId } } as RunWorkflowAction)),
        );
        expect(gen.next().value).toStrictEqual(
            race({
                error: take(createWorkflowError.type),
                success: take(createWorkflowSuccess.type),
            }),
        );
        expect(gen.next().value).toStrictEqual(all(updatedWorkflowVariables.map((v) => call(updateReleaseVariable, { ...v, value: '' }))));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('startAddDeploymentServerSaga', () => {
    it('should start add deployment server saga', () => {
        const action = { payload: 'folderId' } as PayloadAction<string>;
        const workflowId = 'workflowId';
        const gen: SagaIterator = runAddDeploymentServerSaga(action);

        expect(gen.next().value).toStrictEqual(call(searchWorkflowIdByTags, ['deployment server', 'new']));
        expect(gen.next(workflowId).value).toStrictEqual(
            put(runWorkflow({ folderId: Ids.toDomainId(action.payload), workflow: { id: workflowId } } as never as RunWorkflowAction)),
        );
        expect(gen.next().done).toBeTruthy();
    });
});

describe('handleError', () => {
    it('should handle axios error', () => {
        const error = {
            isAxiosError: true,
            response: {
                data: 'whops',
                status: 403,
            },
        };
        const gen: SagaIterator = handleError(error);

        expect(gen.next().value).toStrictEqual(put(setDialogError({ error: 'whops', status: 403 })));
        expect(gen.next().done).toBeTruthy();
    });

    it('should handle generic error', () => {
        const error = new Error('error');
        const gen: SagaIterator = handleError(error);

        expect(gen.next().value).toStrictEqual(put(setDialogError({ error: 'error', status: undefined })));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('fetchConnectionServerMetadata', () => {
    it('should fetch connection server metadata', () => {
        const connectionServerMetadata = [mockArgoConnectionServerMetadata];
        const gen: SagaIterator = fetchConnectionServerMetadata();

        expect(gen.next().value).toStrictEqual(call(findConnectionServerMetadata));
        expect(gen.next(connectionServerMetadata).value).toStrictEqual(put(setConnectionServerMetadata(connectionServerMetadata)));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('filterDeploymentServersSaga', () => {
    it('should filter deployment servers', () => {
        const action = { payload: {} } as PayloadAction<DeploymentServerSearch>;
        const gen: SagaIterator = filterDeploymentServersSaga(action);

        expect(gen.next().value).toStrictEqual(call(searchDeploymentServers, action.payload));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('setConfigurationConditionAction', () => {
    it('should set configuration condition action', () => {
        const gen: SagaIterator = setConfigurationConditionAction();

        expect(gen.next().value).toStrictEqual(call(executeLoadLiveDeploymentConfigurationsAction));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('setConfigurationPageAction', () => {
    it('should set configuration page action', () => {
        const gen: SagaIterator = setConfigurationPageAction();

        expect(gen.next().value).toStrictEqual(call(executeLoadLiveDeploymentConfigurationsAction));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('goToLiveDeploymentConfiguration', () => {
    it('should go to live deployment configuration', () => {
        const $state = { go: jest.fn() } as unknown as IStateService;
        const action = { payload: true } as PayloadAction<boolean>;
        const gen: SagaIterator = goToLiveDeploymentConfiguration(action);

        expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
        expect(gen.next($state).value).toStrictEqual($state.go('folders.detail.live-deployment-config', { forceSetup: action.payload }));
        expect(gen.next().done).toBeTruthy();
    });
});

describe('goBackToLiveDeployments', () => {
    it('should go back to live deployments', () => {
        const $state = { go: jest.fn() } as unknown as IStateService;
        const gen: SagaIterator = goBackToLiveDeployments();

        expect(gen.next().value).toStrictEqual(call(getAngularService, '$state'));
        expect(gen.next($state).value).toStrictEqual($state.go('folders.detail.external-deployments'));
        expect(gen.next().done).toBeTruthy();
    });
});
