import { SagaIterator } from 'redux-saga';
import { testSaga } from 'redux-saga-test-plan';
import constant from 'lodash/constant';
import { all, call, put, select, takeEvery } from 'redux-saga/effects';
import externalDeploymentsRootSaga, {
    executeLoadExternalDeploymentsWithRefreshAction,
    executeCreateServerAction,
    executeDeleteApplicationPackageStatusAction,
    executeDeleteWebhookSourceAction,
    executeLoadExternalDeployments,
    executeLoadExternalDeploymentsAction,
    executeLoadFilterEventSourceAction,
    executeLoadServersAction,
    executeLoadWebhookSourceFiltersAction,
    executeLoadWebhookSourcesAction,
    executeSaveEventSourceAction,
    executeSetupLiveUpdateAction,
    executeSseStreamSubscribeAction,
    executeSseStreamUnsubscribeAction,
    executeUpdateApplicationPackageStatusAction,
    getFiltersAction,
    executeUpdateWebhookSourceFiltersAction,
    getWebhookSources,
    toaster,
    storeFiltersAction,
    withLoadingState,
    updateBreadcrumbsAction,
    executePatchExternalDeploymentsAction,
    handleApplicationSseEventAction,
    setLiveDeploymentsPreconditionFilterAction,
    getAvailablePluginsAction,
    updateServerListAction,
    executeCreateApplicationPackageStatusAction,
    setConfigurationPageAction,
    setConfigurationConditionAction,
    executeLoadLiveDeploymentConfigurationsAction,
    executeLoadLiveDeploymentsConfiguration,
} from './external-deployments.saga';
import { folderExternalDeployments, initialPage } from './external-deployments.reducer';
import { ExternalDeploymentEvent, LiveDeploymentData, ExternalDeploymentPage, Server, LiveDeploymentConfigData } from '../external-deployment.types';
import { getFolderExternalDeployments, getWebhookSourcesSelector } from './external-deployments.selectors';
import { CustomConfiguration } from '../../../../../../../../../core/xlr-ui/app/types';
import { APPLICATION_COLUMN_ID, DEFAULT_MAX_AGE, EVENT_TYPE, STATUS_WEBHOOK_EVENT_SOURCE_TYPE, MAX_AGE_NO_CACHE, SSE_REGISTRATION_URL } from '../constants';
import { mapOrderByValue } from '../helper/utils';
import { httpGET, httpDELETE, httpPOST, httpPUT } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import {
    deployServerMock,
    externalDeploymentCreateEventMock,
    externalDeploymentDeleteEventMock,
    externalDeploymentUpdateEventMock,
    mockApplication,
    mockApplications,
    mockApplicationsMap,
    mockConfigDetails,
    mockDeployWebhookEventSource,
    mockDeployWebhookSource,
    mockEnvironment,
    mockEnvironments,
    mockEnvironmentsMap,
    mockLiveDeploymentConfig,
    mockLiveDeploymentConfigData,
    mockLiveDeploymentData,
    mockLiveDeploymentsPreconditionFilter,
    mockSaveWebhookSourceFiltersRequest,
    mockServers,
    mockServersMap,
    mockSetupLiveUpdateRequest,
    mockWebhookSourceFilters,
    mockWebhookSources,
} from '../__mocks__/external-deployments.mocks';
import { withFlagChangingState } from '../../../../../../../../../core/xlr-ui/app/react/utils/saga-utils';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { sse } from '../../../../../../../../../core/xlr-ui/app/features/sse/sse.reducer';
import { Page } from '../../../../../../../../../core/xlr-ui/app/js/http/backend';

const { broadcastSseEvent } = sse.actions;

const {
    addServerToServers,
    createApplicationStatus,
    createServer,
    deleteApplicationStatus,
    deleteWebhookSource,
    getFilters,
    loadExternalDeployments,
    loadFilterEventSource,
    loadServers,
    loadWebhookSourceFilters,
    loadWebhookSources,
    patchExternalDeployments,
    refreshExternalDeployments,
    removeWebhookSource,
    saveEventSource,
    setCondition,
    setConfigDetails,
    setConnectionErrors,
    setConnectionServers,
    setDisconnectedServers,
    setLiveDeploymentConfigs,
    setLiveDeploymentConfigsCount,
    setLiveDeployments,
    setLiveDeploymentsCount,
    setFilterEventSource,
    setIsLoading,
    setLiveUpdate,
    setMaxAge,
    setPage,
    setPreSelectedServer,
    setServers,
    setWebhookSourceFilters,
    setWebhookSources,
    storeFilters,
    subscribeToSseStream,
    unsubscribeFromSseStream,
    updateApplicationStatus,
    updateBreadcrumbs,
    updateWebhookSourceFilters,
    setLiveDeploymentsPreconditionFilter,
    updateServerList,
    getAvailablePlugins,
    setApplications,
    setEnvironments,
    loadLiveDeploymentsConfiguration,
    setConfigurationCondition,
    setConfigurationPage,
} = folderExternalDeployments.actions;

describe('externalDeploymentsSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = externalDeploymentsRootSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeEvery(createApplicationStatus, executeCreateApplicationPackageStatusAction),
                takeEvery(createServer, executeCreateServerAction),
                takeEvery(updateServerList, updateServerListAction),
                takeEvery(getAvailablePlugins, getAvailablePluginsAction),
                takeEvery(deleteApplicationStatus, executeDeleteApplicationPackageStatusAction),
                takeEvery(deleteWebhookSource, executeDeleteWebhookSourceAction),
                takeEvery(getFilters, getFiltersAction),
                takeEvery(loadExternalDeployments, executeLoadExternalDeploymentsAction),
                takeEvery(loadLiveDeploymentsConfiguration, executeLoadLiveDeploymentConfigurationsAction),
                takeEvery(loadFilterEventSource, executeLoadFilterEventSourceAction),
                takeEvery(loadServers, executeLoadServersAction),
                takeEvery(loadWebhookSourceFilters, executeLoadWebhookSourceFiltersAction),
                takeEvery(loadWebhookSources, executeLoadWebhookSourcesAction),
                takeEvery(patchExternalDeployments, executePatchExternalDeploymentsAction),
                takeEvery(refreshExternalDeployments, executeLoadExternalDeploymentsWithRefreshAction),
                takeEvery(saveEventSource, executeSaveEventSourceAction),
                takeEvery(setLiveUpdate, executeSetupLiveUpdateAction),
                takeEvery(storeFilters, storeFiltersAction),
                takeEvery(subscribeToSseStream, executeSseStreamSubscribeAction),
                takeEvery(unsubscribeFromSseStream, executeSseStreamUnsubscribeAction),
                takeEvery(updateApplicationStatus, executeUpdateApplicationPackageStatusAction),
                takeEvery(updateBreadcrumbs, updateBreadcrumbsAction),
                takeEvery(updateWebhookSourceFilters, executeUpdateWebhookSourceFiltersAction),
                takeEvery(broadcastSseEvent, handleApplicationSseEventAction),
                takeEvery(setLiveDeploymentsPreconditionFilter, setLiveDeploymentsPreconditionFilterAction),
                takeEvery(setConfigurationCondition, setConfigurationConditionAction),
                takeEvery(setConfigurationPage, setConfigurationPageAction),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('executeCreateApplicationPackageStatusAction', () => {
        it('should call executeLoadExternalDeploymentsAction', () => {
            const gen: SagaIterator = executeLoadExternalDeploymentsWithRefreshAction();
            expect(gen.next().value).toStrictEqual(put(setMaxAge(MAX_AGE_NO_CACHE)));
            expect(gen.next().value).toStrictEqual(call(executeLoadExternalDeploymentsAction));
            expect(gen.next().value).toStrictEqual(put(setMaxAge(DEFAULT_MAX_AGE)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    const liveDeployments: Array<LiveDeploymentData> = [mockLiveDeploymentData];

    const createStatusEvent: ExternalDeploymentEvent = {
        ...externalDeploymentCreateEventMock,
        deploymentId: 'NewDeploymentId',
    };

    const deleteStatusEvent: ExternalDeploymentEvent = {
        ...externalDeploymentDeleteEventMock,
    };

    const updateStatusEvent: ExternalDeploymentEvent = {
        ...externalDeploymentUpdateEventMock,
        data: {
            ...externalDeploymentUpdateEventMock.data!,
            deploymentState: {
                ...externalDeploymentUpdateEventMock.data!.deploymentState!,
                status: 'EXECUTING',
                statusGroup: 'in-progress',
                versionTag: '2.0',
            },
        },
    };

    describe('executeDeleteApplicationPackageStatusAction', () => {
        const count = 0;
        const condition = '';
        const page: ExternalDeploymentPage = {
            folderId: 'folderId',
            page: 0,
            order: 'asc',
            orderBy: APPLICATION_COLUMN_ID,
            resultsPerPage: 10,
        };

        it('should remove existing deployment', () => {
            const gen: SagaIterator = executeDeleteApplicationPackageStatusAction({
                payload: deleteStatusEvent,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ liveDeployments, page, condition, count }).value).toStrictEqual(
                call(httpGET, `live-deployment/count?folderId=${page.folderId}&condition=${condition}`),
            );
            expect(gen.next({ data: count }).value).toStrictEqual(put(setLiveDeploymentsCount(count)));
            expect(gen.next().value).toStrictEqual(put(setLiveDeployments([])));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeUpdateApplicationPackageStatusAction', () => {
        const updatedDeployments = [
            {
                ...liveDeployments[0],
                ...updateStatusEvent.data,
            },
        ];

        it('should update existing deployment', () => {
            const gen: SagaIterator = executeUpdateApplicationPackageStatusAction({
                payload: updateStatusEvent,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ liveDeployments }).value).toStrictEqual(put(setLiveDeployments(updatedDeployments)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeCreateApplicationPackageStatusAction', () => {
        const count = 0;
        const condition = '';
        const page: ExternalDeploymentPage = {
            folderId: 'folderId',
            page: 0,
            order: 'asc',
            orderBy: APPLICATION_COLUMN_ID,
            resultsPerPage: 10,
        };
        const updatedDeployments = [...liveDeployments, createStatusEvent.data!];
        const applications = mockApplicationsMap;
        const environments = mockEnvironmentsMap;
        const connectionServers = mockServersMap;

        it('should create new deployment', () => {
            const gen: SagaIterator = executeCreateApplicationPackageStatusAction({
                payload: createStatusEvent,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ liveDeployments, page, condition, applications, environments, connectionServers }).value).toStrictEqual(
                call(httpGET, `application-source/${mockApplication.id}`),
            );
            expect(gen.next({ data: mockApplication }).value).toStrictEqual(call(httpGET, `deployment-target/${mockEnvironment.id}`));
            expect(gen.next({ data: mockEnvironment }).value).toStrictEqual(call(httpGET, `api/v1/config/${mockLiveDeploymentConfig.id}`));
            expect(gen.next({ data: mockLiveDeploymentConfig }).value).toStrictEqual(call(httpGET, `api/v1/config/${mockLiveDeploymentConfig.sourceServer}`));
            expect(gen.next({ data: deployServerMock }).value).toStrictEqual(put(setApplications(applications)));
            expect(gen.next(applications).value).toStrictEqual(put(setEnvironments(environments)));
            expect(gen.next(environments).value).toStrictEqual(put(setConnectionServers(connectionServers)));
            expect(gen.next({ liveDeployments, page, condition, count }).value).toStrictEqual(
                call(httpGET, `live-deployment/count?folderId=${page.folderId}&condition=${condition}`),
            );
            expect(gen.next({ data: count }).value).toStrictEqual(put(setLiveDeploymentsCount(count)));
            expect(gen.next().value).toStrictEqual(put(setLiveDeployments(updatedDeployments)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('withLoadingState', () => {
        it('should set and reset isLoading flag', () => {
            const alwaysFalse = constant(false);
            const gen: SagaIterator = withLoadingState(call(alwaysFalse));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(call(alwaysFalse));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadExternalDeploymentsAction', () => {
        it('should call executeLoadExternalDeployments with loading', () => {
            const gen: SagaIterator = executeLoadExternalDeploymentsAction();
            expect(gen.next().value).toStrictEqual(call(withFlagChangingState, call(executeLoadExternalDeployments), setIsLoading));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadExternalDeployments', () => {
        const liveDeploymentPage: Page<LiveDeploymentData> = {
            data: {
                content: [liveDeployments],
                totalElements: 1,
            },
        } as never;
        const disconnectedServers = ['0'];
        const maxAge = 10000;
        const errors = ['Error fetching status on Deploy server'];
        const page: ExternalDeploymentPage = {
            folderId: 'folderId',
            page: 0,
            order: 'asc',
            orderBy: APPLICATION_COLUMN_ID,
            resultsPerPage: 10,
        };
        const condition = 'condition';
        const applications = mockApplications;
        const environments = mockEnvironments;
        const configs = [mockLiveDeploymentConfig];
        const connectionServers = mockServers;
        const applicationsMap = mockApplicationsMap;
        const environmentsMap = mockEnvironmentsMap;
        const serversMap = mockServersMap;
        const connectionServerErrors = new Map<Server, string>([[deployServerMock, deployServerMock.id!]]);

        it('should not fetch anything', () => {
            const gen: SagaIterator = executeLoadExternalDeployments();
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ maxAge }).done).toBeTruthy();
        });

        it('should show toastr', () => {
            const gen: SagaIterator = executeLoadExternalDeployments();
            const conf: CustomConfiguration = {
                id: 'serverId',
                title: 'Deploy Server',
                type: 'xldeploy.XLDeployServer',
            };
            const errorMessage = 'Error fetching status data. Check connection to Deploy Server HTTP Connection or check application logs for more details.';

            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ maxAge, page, condition }).value).toStrictEqual(
                call(httpGET, `live-deployment?max-age=${maxAge}&folderId=${page.folderId}`, true),
            );
            expect(gen.throw?.({ response: { data: conf } }).value).toStrictEqual(call(toaster.error, errorMessage));
            expect(gen.next().done).toBeTruthy();
        });

        it('should fetch deployments and connections', () => {
            const gen: SagaIterator = executeLoadExternalDeployments();
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ maxAge, page, condition }).value).toStrictEqual(
                call(httpGET, `live-deployment?max-age=${maxAge}&folderId=${page.folderId}`, true),
            );
            expect(gen.next({ data: errors, connectionServerErrors }).value).toStrictEqual(
                call(
                    httpGET,
                    `live-deployment/page?max-age=${maxAge}&folderId=${page.folderId}&page=${page.page}&resultsPerPage=${
                        page.resultsPerPage
                    }&orderBy=${mapOrderByValue(page.orderBy)}&order=${page.order.toUpperCase()}&condition=${condition}`,
                    true,
                ),
            );

            expect(gen.next({ data: liveDeploymentPage }).value).toStrictEqual(call(httpGET, `application-source/folder/${page.folderId}`));
            expect(gen.next({ data: applications }).value).toStrictEqual(call(httpGET, `deployment-target/folder/${page.folderId}`));
            expect(gen.next({ data: environments }).value).toStrictEqual(
                call(httpGET, `api/v1/config/byTypeAndTitle?configurationType=${STATUS_WEBHOOK_EVENT_SOURCE_TYPE}&folderId=${page.folderId}&folderOnly=true`),
            );
            expect(gen.next({ data: configs }).value).toStrictEqual(
                call(httpPOST, `api/v1/config/byIds`, ['Configuration/Custom/Configuratione1bf86a7a9a84273bbb5f47700a9f8b8']),
            );
            expect(gen.next({ data: connectionServers }).value).toStrictEqual(put(setConnectionServers(serversMap)));
            expect(gen.next({ data: serversMap }).value).toStrictEqual(put(setConnectionErrors(errors)));
            expect(gen.next({ data: errors }).value).toStrictEqual(put(setDisconnectedServers(disconnectedServers)));
            expect(gen.next({ data: disconnectedServers }).value).toStrictEqual(put(setApplications(applicationsMap)));
            expect(gen.next({ data: applicationsMap }).value).toStrictEqual(put(setEnvironments(environmentsMap)));
            expect(gen.next({ data: environmentsMap }).value).toStrictEqual(put(setLiveDeployments(liveDeploymentPage.content)));
            expect(gen.next({ data: liveDeploymentPage }).value).toStrictEqual(put(setLiveDeploymentsCount(liveDeploymentPage.totalElements)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeSseStreamSubscribeAction', () => {
        it('should call proper url', () => {
            const saga = testSaga(executeSseStreamSubscribeAction);
            saga.next().call(httpPOST, SSE_REGISTRATION_URL, {}, true);
        });
    });

    describe('handleApplicationSseEvent', () => {
        it('should handle EVENT_TYPE.applicationCreated', () => {
            const saga = testSaga(handleApplicationSseEventAction, {
                type: '',
                payload: { eventName: EVENT_TYPE.applicationCreated, payload: createStatusEvent, origin: '' },
            });
            saga.next().put(createApplicationStatus(createStatusEvent));
            saga.next().isDone();
        });
        it('should handle EVENT_TYPE.applicationDeleted', () => {
            const saga = testSaga(handleApplicationSseEventAction, {
                type: '',
                payload: { eventName: EVENT_TYPE.applicationDeleted, payload: deleteStatusEvent, origin: '' },
            });
            saga.next().put(deleteApplicationStatus(deleteStatusEvent));
            saga.next().isDone();
        });
        it('should handle EVENT_TYPE.applicationChanged', () => {
            const saga = testSaga(handleApplicationSseEventAction, {
                type: '',
                payload: { eventName: EVENT_TYPE.applicationChanged, payload: updateStatusEvent, origin: '' },
            });
            saga.next().put(updateApplicationStatus(updateStatusEvent));
            saga.next().isDone();
        });
    });

    describe('executeSseStreamUnsubscribeAction', () => {
        it('should call proper url', () => {
            const gen: SagaIterator = executeSseStreamUnsubscribeAction();
            expect(gen.next().value).toStrictEqual(call(httpDELETE, SSE_REGISTRATION_URL));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeCreateServerAction', () => {
        it('should create and set server', () => {
            const createServerResponse = { data: deployServerMock };
            const gen: SagaIterator = executeCreateServerAction({ payload: deployServerMock, type: 'any' });
            expect(gen.next(createServerResponse).value).toStrictEqual(call(withLoadingState, call(httpPOST, `api/v1/config`, deployServerMock)));
            expect(gen.next(createServerResponse).value).toStrictEqual(call(toaster.success, `Server saved successfully`));
            expect(gen.next(createServerResponse).value).toStrictEqual(put(addServerToServers(createServerResponse.data)));
            expect(gen.next(createServerResponse).value).toStrictEqual(put(setPreSelectedServer(createServerResponse.data)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeSetupLiveUpdateAction', () => {
        it('should create and set server', () => {
            const setupLiveUpdateResponse = { data: mockConfigDetails };

            const gen: SagaIterator = executeSetupLiveUpdateAction({ payload: mockSetupLiveUpdateRequest, type: 'any' });
            expect(gen.next(setupLiveUpdateResponse).value).toStrictEqual(
                call(
                    withLoadingState,
                    call(httpPOST, `live-deployment/setup/${mockSetupLiveUpdateRequest.serverId}?folderId=${mockSetupLiveUpdateRequest.folderId}`),
                ),
            );
            expect(gen.next(setupLiveUpdateResponse).value).toStrictEqual(put(setConfigDetails(setupLiveUpdateResponse.data)));
            expect(gen.next(setupLiveUpdateResponse).value).toStrictEqual(call(getWebhookSources, mockSetupLiveUpdateRequest.folderId));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('getWebhookSources', () => {
        it('should fetch and set webhook sources', () => {
            const folderId = 'Folder11';
            const webhookSourcesResponse = { data: mockWebhookSources };

            const gen: SagaIterator = getWebhookSources(folderId);
            expect(gen.next().value).toStrictEqual(
                call(
                    withLoadingState,
                    call(httpGET, `api/v1/config/byTypeAndTitle?configurationType=events.StatusWebhookEventSource&folderId=${folderId}&folderOnly=true`),
                ),
            );
            expect(gen.next(webhookSourcesResponse).value).toStrictEqual(put(setWebhookSources(webhookSourcesResponse.data)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadWebhookSourceFiltersAction', () => {
        const webhookSourceId = 'Configuration/Custom/Configurationd50e23fc5317443abd7bb07939e31513';
        it('should load webhook source filters', () => {
            const webhookSourcesFiltersResponse = { data: mockWebhookSourceFilters };

            const gen: SagaIterator = executeLoadWebhookSourceFiltersAction({ payload: webhookSourceId, type: 'any' });
            expect(gen.next().value).toStrictEqual(call(withLoadingState, call(httpGET, `live-deployment/filters/${webhookSourceId}`)));
            expect(gen.next(webhookSourcesFiltersResponse).value).toStrictEqual(put(setWebhookSourceFilters(webhookSourcesFiltersResponse.data)));
            expect(gen.next().done).toBeTruthy();
        });

        it('should catch error', () => {
            const message = 'msg';
            const gen: SagaIterator = executeLoadWebhookSourceFiltersAction({ payload: webhookSourceId, type: 'any' });
            expect(gen.next().value).toStrictEqual(call(withLoadingState, call(httpGET, `live-deployment/filters/${webhookSourceId}`)));
            expect(gen.throw?.(new Error(message)).value).toStrictEqual(call(toaster.error, message));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadWebhookSourcesAction', () => {
        it('should call method to load webhook sources', () => {
            const folderId = 'Folder11';

            const gen: SagaIterator = executeLoadWebhookSourcesAction({ payload: folderId, type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getWebhookSources, folderId));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeUpdateWebhookSourceFiltersAction', () => {
        it('should save webhook source filters', () => {
            const saveWebhookSourceResponse = { data: mockDeployWebhookSource };

            const gen: SagaIterator = executeUpdateWebhookSourceFiltersAction({ payload: mockSaveWebhookSourceFiltersRequest, type: 'any' });
            expect(gen.next(saveWebhookSourceResponse).value).toStrictEqual(
                call(
                    withLoadingState,
                    call(
                        httpPUT,
                        `live-deployment/filters/${mockSaveWebhookSourceFiltersRequest.webhookSourceId}`,
                        mockSaveWebhookSourceFiltersRequest.filteredFolders,
                    ),
                ),
            );
            expect(gen.next(saveWebhookSourceResponse).value).toStrictEqual(select(getWebhookSourcesSelector));
            expect(gen.next(mockWebhookSources).value).toStrictEqual(put(setWebhookSources(mockWebhookSources)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadServersAction', () => {
        it('should load and set servers', () => {
            const folderId = 'Folder11';
            const serversResponse = { data: mockServers };

            const gen: SagaIterator = executeLoadServersAction({ payload: folderId, type: 'any' });
            expect(gen.next().value).toStrictEqual(
                call(withLoadingState, call(httpGET, `api/v1/config/byTypeAndTitle?configurationType=configuration.StatusHttpConnection&folderId=${folderId}`)),
            );
            expect(gen.next(serversResponse).value).toStrictEqual(put(setServers(serversResponse.data)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadWebhookSourcesAction', () => {
        it('should call getWebhookSourcesAction', () => {
            const folderId = 'Folder11';

            const gen: SagaIterator = executeLoadWebhookSourcesAction({ payload: folderId, type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getWebhookSources, folderId));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeDeleteWebhookSourceAction', () => {
        it('should delete webhook source', () => {
            const webhookSourceId = 'Configuration/Custom/Configuration1682004056084295907f2e9c654f049b';

            const gen: SagaIterator = executeDeleteWebhookSourceAction({ payload: webhookSourceId, type: 'any' });
            expect(gen.next().value).toStrictEqual(call(withLoadingState, call(httpDELETE, `live-deployment/delete/${webhookSourceId}`)));
            expect(gen.next().value).toStrictEqual(put(removeWebhookSource(webhookSourceId)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadFilterEventSourceAction', () => {
        it('should load webhook endpoint', () => {
            const webhookEndpointResponse = { data: mockDeployWebhookEventSource };

            const gen: SagaIterator = executeLoadFilterEventSourceAction({ payload: mockDeployWebhookEventSource.id || '', type: 'any' });
            expect(gen.next(webhookEndpointResponse).value).toStrictEqual(
                call(withLoadingState, call(httpGET, `api/v1/config/${mockDeployWebhookEventSource.id}`)),
            );
            expect(gen.next(webhookEndpointResponse).value).toStrictEqual(put(setFilterEventSource(webhookEndpointResponse.data)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeSaveEventSourceAction', () => {
        it('should save webhook endpoint', () => {
            const gen: SagaIterator = executeSaveEventSourceAction({ payload: mockDeployWebhookEventSource, type: 'any' });
            expect(gen.next().value).toStrictEqual(
                call(withLoadingState, call(httpPUT, `api/v1/config/${mockDeployWebhookEventSource.id}`, mockDeployWebhookEventSource)),
            );
            expect(gen.next().value).toStrictEqual(call(toaster.success, `Webhook endpoint updated successfully`));
            expect(gen.next().done).toBeTruthy();
        });
    });

    const condition = 'con';
    const page: ExternalDeploymentPage = {
        folderId: 'folder-1',
        order: 'asc',
        orderBy: 'environment',
        page: 0,
        resultsPerPage: 10,
    };
    const filters = {
        condition,
        ...page,
    };
    const setExternalDeploymentsFilters = jest.fn();
    const getExternalDeploymentsFilters = jest.fn(() => filters);
    const update = jest.fn();
    const clientSettings = {
        getExternalDeploymentsFilters,
        setExternalDeploymentsFilters,
    };
    const filtersQueryParams = {
        update,
    };

    describe('storeFiltersAction', () => {
        it('should store filters to local storage', () => {
            const gen: SagaIterator = storeFiltersAction();
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ page, condition }).value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call(getAngularService, 'FiltersQueryParams'));
            expect(gen.next(filtersQueryParams).done).toBeTruthy();
            expect(setExternalDeploymentsFilters).toHaveBeenCalled();
            expect(update).toHaveBeenCalled();
        });
    });

    describe('getFiltersAction', () => {
        it('should get filters from local storage and update state', () => {
            const gen: SagaIterator = getFiltersAction({ payload: 'folder-1', type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next(clientSettings).value).toStrictEqual(call(getAngularService, '$stateParams'));
            expect(gen.next(clientSettings).value).toStrictEqual(put(setCondition(condition)));
            expect(gen.next(clientSettings).value).toStrictEqual(put(setPage(page)));
            expect(gen.next(clientSettings).done).toBeTruthy();
            expect(getExternalDeploymentsFilters).toHaveBeenCalled();
        });
        it('should handle empty filters', () => {
            const getEmptyExternalDeploymentsFilters = jest.fn();
            const gen: SagaIterator = getFiltersAction({ payload: 'folder-1', type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'ClientSettings'));
            expect(gen.next({ ...clientSettings, getExternalDeploymentsFilters: getEmptyExternalDeploymentsFilters }).value).toStrictEqual(
                call(getAngularService, '$stateParams'),
            );
            expect(gen.next({ getExternalDeploymentsFilters: getEmptyExternalDeploymentsFilters }).done).toBeTruthy();
            expect(getEmptyExternalDeploymentsFilters).toHaveBeenCalled();
        });
    });

    describe('executePatchExternalDeploymentsAction', () => {
        it('should patch external deployments', () => {
            const webhookSourceId = 'Configuration/Custom/Configuration1682004056084295907f2e9c654f049b';

            const gen: SagaIterator = executePatchExternalDeploymentsAction({ payload: webhookSourceId, type: 'any' });
            expect(gen.next().value).toStrictEqual(call(withLoadingState, call(httpPOST, `live-deployment/patch/${webhookSourceId}`)));
            expect(gen.next().value).toStrictEqual(call(toaster.success, `External deployments patched successfully`));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('updateBreadcrumbsAction', () => {
        it('should call breadcrumbsService with appropriate conf', () => {
            const folder = { id: 'folderId', title: 'title' };
            const updateBreadcrumbsFromService = jest.fn();
            const gen: SagaIterator = updateBreadcrumbsAction({ payload: folder, type: 'any' });
            expect(gen.next().value).toStrictEqual(call(getAngularService, 'BreadcrumbsService'));
            expect(gen.next({ updateBreadcrumbs: updateBreadcrumbsFromService }).done).toBeTruthy();
            expect(updateBreadcrumbsFromService).toHaveBeenCalledWith(
                [
                    {
                        pathSuffix: 'folders/' + folder.id,
                        label: folder.title,
                        folder,
                    },
                    {
                        pathSuffix: 'folders/' + folder.id + '/external-deployments',
                        label: 'Application deployments',
                        folder,
                    },
                    {
                        label: 'Application Sources',
                    },
                ],
                false,
            );
        });
    });

    describe('setLiveDeploymentsPreconditionFilterAction', () => {
        it('should set folder, condition and store filters to local storage', () => {
            const gen: SagaIterator = setLiveDeploymentsPreconditionFilterAction({ payload: mockLiveDeploymentsPreconditionFilter, type: 'any' });
            expect(gen.next(mockLiveDeploymentsPreconditionFilter).value).toStrictEqual(
                put(setPage({ ...initialPage, folderId: mockLiveDeploymentsPreconditionFilter.folderId })),
            );
            expect(gen.next(mockLiveDeploymentsPreconditionFilter).value).toStrictEqual(
                put(setCondition(mockLiveDeploymentsPreconditionFilter.applicationName)),
            );
            expect(gen.next().value).toStrictEqual(call(storeFiltersAction));
        });
    });

    describe('executeLoadLiveDeploymentConfigurationsAction', () => {
        it('should call executeLoadLiveDeploymentsConfigurationAction with loading', () => {
            const gen: SagaIterator = executeLoadLiveDeploymentConfigurationsAction();
            expect(gen.next().value).toStrictEqual(call(withFlagChangingState, call(executeLoadLiveDeploymentsConfiguration), setIsLoading));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadLiveDeploymentsConfiguration', () => {
        const liveDeploymentConfigPage: Page<LiveDeploymentConfigData> = {
            data: {
                content: [mockLiveDeploymentConfigData],
                totalElements: 1,
            },
        } as never;
        const maxAge = 10000;
        const configurationPage: ExternalDeploymentPage = {
            folderId: 'folderId',
            page: 0,
            order: 'asc',
            orderBy: APPLICATION_COLUMN_ID,
            resultsPerPage: 10,
        };
        const configurationCondition = 'condition';

        it('should not fetch anything', () => {
            const gen: SagaIterator = executeLoadLiveDeploymentsConfiguration();
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ maxAge }).done).toBeTruthy();
        });

        it('should show toastr', () => {
            const gen: SagaIterator = executeLoadLiveDeploymentsConfiguration();
            const conf: CustomConfiguration = {
                id: 'serverId',
                title: 'Deploy Server',
                type: 'xldeploy.XLDeployServer',
            };
            const errorMessage = 'Error fetching status data. Check connection to Deploy Server HTTP Connection or check application logs for more details.';

            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ configurationPage, configurationCondition }).value).toStrictEqual(
                call(
                    httpGET,
                    `live-deployment/configs?folderId=${configurationPage.folderId}&page=0&resultsPerPage=10&orderBy=TITLE&order=ASC&condition=condition`,
                    true,
                ),
            );
            expect(gen.throw?.({ response: { data: conf } }).value).toStrictEqual(call(toaster.error, errorMessage));
            expect(gen.next().done).toBeTruthy();
        });

        it('should fetch deployments and connections', () => {
            const gen: SagaIterator = executeLoadLiveDeploymentsConfiguration();
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ configurationPage, configurationCondition }).value).toStrictEqual(
                call(
                    httpGET,
                    `live-deployment/configs?folderId=${configurationPage.folderId}&page=0&resultsPerPage=10&orderBy=TITLE&order=ASC&condition=condition`,
                    true,
                ),
            );
            expect(gen.next({ data: liveDeploymentConfigPage }).value).toStrictEqual(put(setLiveDeploymentConfigs(liveDeploymentConfigPage.content)));
            expect(gen.next().value).toStrictEqual(put(setLiveDeploymentConfigsCount(liveDeploymentConfigPage.totalElements)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('setConfigurationConditionAction', () => {
        it('should call for live deployment fetch', () => {
            const gen: SagaIterator = setConfigurationConditionAction();
            expect(gen.next().value).toStrictEqual(call(executeLoadLiveDeploymentConfigurationsAction));
        });
    });

    describe('setConfigurationPageAction', () => {
        it('should call for live deployment fetch', () => {
            const gen: SagaIterator = setConfigurationPageAction();
            expect(gen.next().value).toStrictEqual(call(executeLoadLiveDeploymentConfigurationsAction));
        });
    });
});
