package com.xebialabs.xlrelease.status.service

import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.internal.filter.LiveDeploymentFilters
import com.xebialabs.xlrelease.api.internal.views.{LiveDeploymentConfigOrderMode, LiveDeploymentConfigView, LiveDeploymentOrderDirection, LiveDeploymentOrderMode}
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments.{LiveDeployment, LiveDeploymentConfig}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.service.ConfigurationAutoconfigService
import com.xebialabs.xlrelease.status.service.script.{ExternalDeploymentScriptService, PatchScriptService, formatErrorMessage}
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.status.webhook.events.{DeploymentProviderEvent, StatusWebhookEventSource}
import com.xebialabs.xlrelease.webhooks.endpoint.WebhookEndpoint.PostWebhookEndpoint
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, PageRequest, ReleasePageImpl}
import org.springframework.stereotype.Service

import java.lang
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait EndpointExternalDeploymentService {

  def getConnectionErrors(folderId: String, maxAge: Long = 0): Map[StatusHttpConnection, String]

  def getLiveDeployments(folderId: String,
                         maxAge: Long = 0,
                         page: lang.Integer,
                         resultsPerPage: lang.Integer,
                         orderBy: LiveDeploymentOrderMode,
                         direction: LiveDeploymentOrderDirection,
                         condition: String = ""): Page[LiveDeployment]
  def saveWebhookSourceFilters(webhookSourceId: String): StatusWebhookEventSource
  def deleteWebhookSource(webhookSourceId: String): Unit
  def patchExternalDeployments(webhookSourceId: String): Unit
  def countLiveDeployments(folderId: String, condition: String = ""): Integer
  def getLiveDeploymentConfigs(
                                folderId: CiId,
                                page: Integer,
                                resultsPerPage: Integer,
                                orderBy: LiveDeploymentConfigOrderMode,
                                direction: LiveDeploymentOrderDirection,
                                condition: String
                              ): Page[LiveDeploymentConfigView]
  def searchLiveDeployments(liveDeploymentFilters: LiveDeploymentFilters): List[LiveDeployment]
}

@Service
class EndpointExternalDeploymentServiceImpl(configurationRepository: ConfigurationRepository,
                                             externalDeploymentService: LiveDeploymentService,
                                             statusScriptService: ExternalDeploymentScriptService,
                                             autoconfigService: ConfigurationAutoconfigService,
                                             patchScriptService: PatchScriptService
                                           ) extends EndpointExternalDeploymentService with Logging {

  override def getConnectionErrors(folderId: String, maxAge: Long = 0): Map[StatusHttpConnection, String] = {
    configurationRepository
      .findAllByTypeAndTitle[StatusWebhookEventSource](Type.valueOf(classOf[StatusWebhookEventSource]), null, folderId, folderOnly = true)
      .asScala.map(_.getId)
      .map(endpointId => {
        val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](endpointId)
        val connection = statusWebhookEventSource.sourceServer.asInstanceOf[StatusHttpConnection]
        val errorMaybe = if (!externalDeploymentService.exists(folderId, maxAge)) {
          fetchAndPersistExternalDeployment(statusWebhookEventSource.getFolderId, statusWebhookEventSource) match {
            case Left(errorMessage) =>
              Some(errorMessage)
            case _ =>
              None
          }
        }
        else {
          None
        }
        errorMaybe -> connection
      }).collect {
        case (Some(error), connection) => connection -> error
        case (None, connection) => connection -> null
      }.toMap
  }

  override def getLiveDeployments(
                                   folderId: String,
                                   maxAge: Long,
                                   page: lang.Integer,
                                   resultsPerPage: lang.Integer,
                                   orderBy: LiveDeploymentOrderMode,
                                   direction: LiveDeploymentOrderDirection,
                                   condition: String): Page[LiveDeployment] = {

    if (!externalDeploymentService.exists(folderId, maxAge)) {
      val sourceIds = configurationRepository
        .findAllByTypeAndTitle[StatusWebhookEventSource](Type.valueOf(classOf[StatusWebhookEventSource]), null, folderId, folderOnly = true)
        .asScala.map(_.getId).toVector

      sourceIds.foreach(sourceId => {
        val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](sourceId)
        fetchAndPersistExternalDeployment(folderId, statusWebhookEventSource)
      })
    }

    externalDeploymentService.findOnFolder(folderId, resultsPerPage, page, orderBy, direction, condition)
  }


  private def fetchAndPersistExternalDeployment(folderId: CiId,
                                                statusWebhookEventSource: StatusWebhookEventSource): Either[String, Unit]  = {
    Try(statusScriptService.executeScript(statusWebhookEventSource)) match {
      case Success(result: Vector[DeploymentProviderEvent]) =>
        val saved = externalDeploymentService.batchSaveOrUpdate(folderId, result)
        externalDeploymentService.removeStaleDeployments(statusWebhookEventSource.getId, saved)
        Right(None)
      case Success(errorObject) =>
        val msg = s"Illegal response in ${statusWebhookEventSource.getId} statusScript response."
        error(msg + s" Returned type was ${errorObject.getClass.getName}, expected list of states")
        Left(msg)
      case Failure(exception) =>
        val msg = s"Error fetching Application status on ${statusWebhookEventSource.getId} [${statusWebhookEventSource.getTitle}]: ${exception.getMessage}"
        warn(msg, exception)
        Left(msg)
    }
  }

  override def saveWebhookSourceFilters(webhookSourceId: String): StatusWebhookEventSource = {
    val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](webhookSourceId)
    configurationRepository.update(statusWebhookEventSource)

    Try(autoconfigService.autoconfigure(statusWebhookEventSource)) match {
      case Success(_) =>
      //do nothing
      case Failure(exception) =>
        val msg = s"Error updating filters in updateFiltersRequest script on $webhookSourceId " +
          s"[${statusWebhookEventSource.getTitle}]: ${exception.getMessage}"
        warn(msg)
        throw UpdateFiltersError(formatErrorMessage(msg))
    }
    fetchAndPersistExternalDeployment(statusWebhookEventSource.getFolderId, statusWebhookEventSource) match {
      case Left(str) =>
        throw EndpointExternalDeploymentError(str)
      case _ => // nothing
    }

    statusWebhookEventSource
  }

  @IsTransactional
  override def deleteWebhookSource(webhookSourceId: String): Unit = {
    val eventSource: StatusWebhookEventSource = configurationRepository.read(webhookSourceId)
    val httpEndpoint: PostWebhookEndpoint = eventSource.getProperty("eventSource")

    configurationRepository.delete(webhookSourceId)
    configurationRepository.delete(httpEndpoint.getId)
  }

  override def patchExternalDeployments(webhookSourceId: String): Unit = {
    val statusWebhookEventSource = configurationRepository.read[StatusWebhookEventSource](webhookSourceId)
    Try(patchScriptService.executeScript(statusWebhookEventSource)) match {
      case Success(_) =>
      //do nothing
      case Failure(exception) =>
        val msg = s"Error executing patch script on $webhookSourceId [${statusWebhookEventSource.getTitle}]: ${exception.getMessage}"
        warn(msg)
        throw PatchError(formatErrorMessage(msg))
    }
  }

  override def countLiveDeployments(folderId: CiId, condition: String): Integer = {
    externalDeploymentService.count(folderId, condition)
  }

  override def getLiveDeploymentConfigs(
                                         folderId: CiId,
                                         page: Integer,
                                         resultsPerPage: Integer,
                                         orderBy: LiveDeploymentConfigOrderMode,
                                         direction: LiveDeploymentOrderDirection,
                                         condition: String
                                       ): Page[LiveDeploymentConfigView] = {

    val results = configurationRepository
      .findAllByTypeAndTitle[LiveDeploymentConfig](Type.valueOf(classOf[LiveDeploymentConfig]), null, folderId, folderOnly = true)
    val configToSourceMap = configurationRepository
      .findAllByTypeAndTitle[StatusWebhookEventSource](Type.valueOf(classOf[StatusWebhookEventSource]), null, folderId, folderOnly = true)
      .asScala.flatMap(source => {
        source.liveDeploymentConfigs.asScala.map(config => {
          config.getId -> source
        })
      }).toMap

    val views = results.asScala.map(config => {
      val source = configToSourceMap.get(config.getId)
      val configProps = config.getType.getDescriptor.getPropertyDescriptors.asScala.filter(d => d.getCategory == "Filters")
      LiveDeploymentConfigView.from(
        config.getId,
        config.getTitle,
        source.map(_.getSourceServer.getType.toString).getOrElse(""),
        source.map(_.getSourceServer.getTitle).getOrElse(""),
        source.map(_.getSourceServer.getUrl).getOrElse(""),
        configProps.map(pd => (pd.getName -> config.getProperty(pd.getName))).toMap.asJava
      )
    }).filter(v => s"${v.getTitle}###${v.getConnectionTitle}###${v.getFilterProperties.asScala.values.mkString("###")}".contains(condition))
      .sortWith((v1, v2) => {
        val comparison = orderBy match {
          case LiveDeploymentConfigOrderMode.TITLE =>
            v1.getTitle.compareTo(v2.getTitle)
          case LiveDeploymentConfigOrderMode.CONNECTION =>
            v1.getConnectionTitle.compareTo(v2.getConnectionTitle)
        }
        direction match {
          case LiveDeploymentOrderDirection.ASC => comparison < 0
          case LiveDeploymentOrderDirection.DESC => comparison > 0
        }
      }).asJava

    val totalCount = views.size
    val pageable = PageRequest.of(page, resultsPerPage, LiveDeploymentOrderDirection.toDirection(direction), orderBy.toString)
    new ReleasePageImpl[LiveDeploymentConfigView](views, pageable, totalCount)
  }

  override def searchLiveDeployments(liveDeploymentFilters: LiveDeploymentFilters): List[LiveDeployment] = {
    externalDeploymentService.search(liveDeploymentFilters)
  }
}

case class ConnectionServerData(connectionId: String, title: String, serverType: String, serverUrl: String, error: Option[String])

case class EndpointExternalDeploymentError(msg: String) extends DeployitException(msg)

case class WebhookSourceFilters(webhookSourceId: String,
                                folderFilterOptions: List[String],
                                folderFilterValues: List[String])

case class WebhookSourceFiltersError(msg: String) extends DeployitException(msg)

case class UpdateFiltersError(msg: String) extends DeployitException(msg)

case class PatchError(msg: String) extends DeployitException(msg)
