import React, { memo } from 'react';
import Moment from 'react-moment';
import upperFirst from 'lodash/upperFirst';
import camelCase from 'lodash/camelCase';
import deburr from 'lodash/deburr';
import { DotCard, DotCardContent, DotIconButton, DotLink, DotPill } from '@digital-ai/dot-components';
import { Server, LiveDeploymentData, Application, Environment, PluginInfo } from '../external-deployment.types';
import { IMG_SRC } from '../constants';
import './external-deployment.component.less';

export interface ExternalDeploymentProps {
    application: Application;
    connectionServer: Server;
    environment: Environment;
    liveDeployment: LiveDeploymentData;
    pluginInfo: PluginInfo;
}

const ExternalDeploymentComponent = ({
    application,
    connectionServer: { type },
    environment,
    liveDeployment: { deploymentState },
    pluginInfo: pluginInfo,
}: ExternalDeploymentProps) => {
    return (
        <DotCard className="external-deployment-card" key={application.correlationUid + environment.correlationUid}>
            <div data-testid={'app' + application.title}>
                <DotCardContent>
                    <div className="icon-column">
                        <img alt={type} className="server-icon" src={pluginInfo.iconLocation} />
                    </div>
                    <div className="application-name-column" title={application.title}>
                        {application.title}
                    </div>
                    <div className="environment-column">
                        <img alt="environment" className="server-icon" src={IMG_SRC.environment} />
                        <span title={environment.title}>{environment.title}</span>
                    </div>
                    <div className="version-column">
                        <DotLink
                            data-testid={`versionLabel-${application.correlationUid}-${environment.title}`}
                            href={deploymentState.versionLink}
                            target="_blank"
                            tooltip={deploymentState.versionTooltip}
                        >
                            {deploymentState.versionTag}
                        </DotLink>
                    </div>
                    <div className="status-column">
                        <DotPill label={upperFirst(camelCase(deburr(deploymentState.status)))} size="small" status={deploymentState.statusGroup} />
                    </div>
                    <div className="last-updated-column">
                        <Moment fromNow interval={30000}>
                            {deploymentState.lastChangeTime}
                        </Moment>
                    </div>
                    <div className="link-column">
                        <DotIconButton
                            data-testid={`deeplink-${application.correlationUid}-${environment.title}`}
                            iconId="open-new-tab"
                            onClick={() => window.open(deploymentState.deploymentLink, '_blank')}
                            size="small"
                            tooltip="Open application"
                        />
                    </div>
                </DotCardContent>
            </div>
        </DotCard>
    );
};

export const ExternalDeployment = memo(ExternalDeploymentComponent);
