package com.xebialabs.xlrelease.status.service

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.internal.views.{ExternalDeploymentOrderDirection, ExternalDeploymentOrderMode}
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments._
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository, EnvironmentStageRepository}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.status.repository.persistence.LiveDeploymentPersistence
import com.xebialabs.xlrelease.status.repository.persistence.data.LiveDeploymentRow
import com.xebialabs.xlrelease.status.sse.service.ServerSentEventsService
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, PageRequest, ReleasePageImpl}
import org.springframework.stereotype.Service

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait LiveDeploymentService {
  def batchSaveOrUpdate(folderId: CiId, states: Vector[DeploymentProviderEvent]): List[String]

  def findOnFolder(folderId: String,
                   limit: Int, page: Int,
                   orderBy: ExternalDeploymentOrderMode,
                   direction: ExternalDeploymentOrderDirection,
                   condition: String = ""): Page[LiveDeployment]

  def exists(folderId: CiId, maxAge: Long): Boolean

  def saveAndNotify(folderId: String, processedEvent: DeploymentProviderEvent): Unit

  def count(folderId: CiId, condition: String): Integer

  def removeStaleDeployments(eventSourceId: String, exclude: List[String] = List()): Unit
}

@Service
@IsTransactional
class LiveDeploymentServiceImpl(deploymentPersistence: LiveDeploymentPersistence,
                                 applicationRepository: ApplicationRepository,
                                 environmentRepository: EnvironmentRepository,
                                 environmentStageRepository: EnvironmentStageRepository,
                                 serverSentEventsService: ServerSentEventsService,
                                 repositoryAdapter: SqlRepositoryAdapter,
                                 configurationRepository: ConfigurationRepository,
                               ) extends LiveDeploymentService with Logging {
  private val DefaultUnknown: String = "Unknown"

  override def batchSaveOrUpdate(folderId: CiId, states: Vector[DeploymentProviderEvent]): List[String] = {
    implicit val envs: mutable.Map[CiId, Environment] = mutable.Map.empty
    implicit val apps: mutable.Map[CiId, Application] = mutable.Map.empty
    states.flatMap { state =>
      saveOrUpdateStatusEvent(folderId, state, fetchDefaultStage).map(_.getId)
    }.toList
  }

  private def saveOrUpdateStatusEvent(folderId: CiId, event: DeploymentProviderEvent, defaultStage: EnvironmentStage)
                                     (implicit envs: mutable.Map[String, Environment] = mutable.Map.empty[String, Environment],
                                      apps: mutable.Map[String, Application] = mutable.Map.empty[String, Application]): Option[LiveDeployment] = {
    val receivedApp = event.application
    val receivedEnv = event.environment

    val env = envs.getOrElseUpdate(receivedEnv.getCorrelationUid, {
      val envMaybe = environmentRepository.findInFolderByCorrelationId(folderId, receivedEnv.getCorrelationUid)
      envMaybe.map(existingEnv => {
        refreshEnvironment(existingEnv, receivedEnv)
      }).getOrElse(environmentRepository.createEnvironment(constructEnvironment(folderId, receivedEnv, defaultStage)))
    })
    val app = apps.getOrElseUpdate(receivedApp.getCorrelationUid, {
      val appMaybe = applicationRepository.findInFolderByCorrelationId(folderId, receivedApp.getCorrelationUid)
      appMaybe.map(existingApp => {
        refreshApplication(existingApp, receivedApp, env)
      }).getOrElse(applicationRepository.createApplication(constructApplication(folderId, receivedApp, env)))
    })

    Try(deploymentPersistence.save(toLiveDeployment(folderId, event.configId, app, env, event.deploymentState))) match {
      case Failure(exception) =>
        warn(s"Couldn't save fetched external deployment for Endpoint [${folderId}] into database, skipping record: ${exception.getMessage}", exception)
        None
      case Success(row) => Some(toLiveDeployment(row))
    }
  }

  private def refreshEnvironment(original: Environment, update: Environment): Environment = {
    var dirty = false
    if (original == null || update == null) {
      original
    } else {
      if (original.getTitle != update.getTitle) {
        original.setTitle(update.getTitle)
        dirty = true
      }
      val originalTarget = original.getDeploymentTarget
      val updateTarget = update.getDeploymentTarget

      if (originalTarget != null && updateTarget == null) {
        original.setDeploymentTarget(null)
        dirty = true
      }

      // always update Target if not null
      if (updateTarget != null) {
        original.setDeploymentTarget(updateTarget)
        dirty = true
      }
      if (dirty) environmentRepository.updateEnvironment(original) else original
    }
  }

  def refreshApplication(original: Application, update: Application, withEnv: Environment): Application = {
    var dirty = false
    if (original == null || update == null) {
      original
    } else {
      if (original.getTitle != update.getTitle) {
        original.setTitle(update.getTitle)
        dirty = true
      }
      val containedEnvCorrelationIds = original.getEnvironments.asScala.map(e => e.getCorrelationUid)
      if (!containedEnvCorrelationIds.contains(withEnv.getCorrelationUid)) {
        original.getEnvironments.add(withEnv)
        dirty = true
      }

      val originalSource = original.getApplicationSource
      val updateSource = update.getApplicationSource

      if (originalSource != null && updateSource == null) {
        original.setApplicationSource(null)
        dirty = true
      }
      // always update Source if not null
      if (updateSource != null) {
        original.setApplicationSource(updateSource)
        dirty = true
      }
      if (dirty) applicationRepository.updateApplication(original) else original
    }
  }

  //noinspection ScalaStyle
  override def saveAndNotify(folderId: String,
                             externalDeploymentEvent: DeploymentProviderEvent): Unit = {
    val application = externalDeploymentEvent.getApplication()
    val environment = externalDeploymentEvent.getEnvironment()
    val event: Option[LiveDeploymentEvent] = externalDeploymentEvent match {
      case removeEvent: DeploymentProviderEvent if removeEvent.operation == StateOperation.RemoveState =>
        val env = environmentRepository.findInFolderByCorrelationId(folderId, environment.getCorrelationUid)
          .getOrElse(throw new NotFoundException(s"Couldn't find environment with correlation '${environment.getCorrelationUid}'"))
        val app = applicationRepository.findInFolderByCorrelationId(folderId, application.getCorrelationUid)
          .getOrElse(throw new NotFoundException(s"Couldn't find application with correlation '${application.getCorrelationUid}'"))
        val liveDeploymentId = deploymentPersistence.fetchLiveDeploymentId(folderId, app.getId, env.getId)
        liveDeploymentId.map{ deploymentId =>
          deploymentPersistence.delete(deploymentId)
          DeleteLiveDeploymentEvent(deploymentId)
        }
      case createEvent: DeploymentProviderEvent
        if createEvent.operation == StateOperation.CreateState && Option(createEvent.deploymentState).exists(s => Option(s.getStatus).isDefined) =>
        debug(s"Received CreateStatusEvent " +
          s"[${application.getTitle} on " +
          s"${environment.getTitle}] - refreshing live deployments.")
        Some(CreateLiveDeploymentEvent())
      case updateEvent: DeploymentProviderEvent
        if Option(updateEvent.deploymentState).exists(s => Option(s.getStatus).isDefined) =>
        val deployment = saveOrUpdateStatusEvent(folderId, updateEvent, fetchDefaultStage)
        deployment.map(d => UpdateLiveDeploymentEvent(d.getId, d))
      case event: DeploymentProviderEvent
        if !Option(event.getEnvironment).exists(e => Option(e.getCorrelationUid).isDefined) =>
        info(s"Received `${event.getOperation}` event for creation of correlated application ${Option(event.getApplication()).map(_.getCorrelationUid).getOrElse("-")}" +
          s" - skipping status update.")
        None
      case event: DeploymentProviderEvent =>
        info(s"Received `${event.getOperation}` event for correlated application ${Option(event.getApplication()).map(_.getCorrelationUid).getOrElse("-")}" +
          s" on correlated environment ${Option(event.getEnvironment()).map(_.getCorrelationUid).getOrElse("-")} with no status - skipping update.")
        None
      case event => // all the ignored types for now
        warn(s"Couldn't execute ExternalDeploymentEvent[${event.getClass.getName}] persistence - skipping save to database.")
        None
    }
    event.foreach(serverSentEventsService.send)
  }

  override def findOnFolder(folderId: CiId,
                            limit: Int, page: Int,
                            orderBy: ExternalDeploymentOrderMode,
                            direction: ExternalDeploymentOrderDirection,
                            condition: String): Page[LiveDeployment] = {
    implicit val envCache: mutable.Map[CiId, Environment] = mutable.Map.empty
    implicit val appCache: mutable.Map[CiId, Application] = mutable.Map.empty
    val results = deploymentPersistence.findOnFolder(folderId, limit, limit * page, orderBy, direction, condition)
      .map(toLiveDeployment)
    val totalCount = deploymentPersistence.count(folderId, condition)
    val pageable = PageRequest.of(page, limit, ExternalDeploymentOrderDirection.toDirection(direction), orderBy.toString)
    new ReleasePageImpl[LiveDeployment](results.asJava, pageable, totalCount)
  }

  override def exists(folderId: CiId, maxAge: Long): Boolean =
    deploymentPersistence.exists(folderId, maxAge)

  private def toLiveDeployment(row: LiveDeploymentRow)
                              (implicit envCache: mutable.Map[String, Environment],
                               appCache: mutable.Map[String, Application]
                              ): LiveDeployment = {
    val ci = CiSerializerHelper.deserialize(row.json, repositoryAdapter).asInstanceOf[LiveDeployment]
    ci.setFolderId(row.folderId)
    val env = envCache.getOrElseUpdate(
      ci.getEnvironment.getId,
      environmentRepository.findEnvironmentById(ci.getEnvironment.getId)
    )
    val app = appCache.getOrElseUpdate(
      ci.getApplication.getId,
      applicationRepository.findApplicationById(ci.getApplication.getId)
    )
    ci.setEnvironment(env)
    ci.setApplication(app)
    ci
  }

  private def toLiveDeployment(folderId: CiId, eventSourceId: CiId,
                               application: Application, environment: Environment,
                               state: DeploymentState): LiveDeployment = {

    LiveDeployment.create(
      folderId, eventSourceId,
      application, environment,
      state
    )
  }

  private def constructApplication(folderId: String,
                                   app: Application,
                                   environment: Environment
                                  ): Application = {
    app.setId(null)
    app.setFolderId(folderId)
    app.setEnvironments(List.apply(environment).asJava)
    app
  }

  private def constructEnvironment(folderId: CiId, env: Environment, stage: EnvironmentStage): Environment = {
    env.setId(null)
    env.setFolderId(folderId)
    env.setStage(stage)
    env
  }

  private def fetchDefaultStage = {
    Try(environmentStageRepository.findByTitle(DefaultUnknown)) match {
      case Success(stage) => stage
      case Failure(_: NotFoundException) =>
        environmentStageRepository.create(EnvironmentStage.create(DefaultUnknown))
      case Failure(exception) => throw exception
    }
  }

  override def count(folderId: CiId, condition: String): Integer = {
    deploymentPersistence.count(folderId, condition)
  }

  override def removeStaleDeployments(eventSourceId: String, exclude: List[String]): Unit = {
    deploymentPersistence.deleteByEventSourceIdAndDeploymentIdNotIn(eventSourceId, exclude)
  }
}

object StateOperation {
  final val CreateState = "create"
  final val RemoveState = "remove"
}
