package com.xebialabs.xlrelease.status.repository.persistence

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION
import com.xebialabs.xlrelease.api.internal.views.{ExternalDeploymentOrderDirection, ExternalDeploymentOrderMode}
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.environments.{DeploymentState, LiveDeployment}
import com.xebialabs.xlrelease.environments.repository.sql.persistence.{ApplicationPersistence, EnvironmentPersistence, toPersistedId}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, PersistenceSupport, Utils}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.status.repository.persistence.LiveDeploymentSchema.LIVE_DEPLOYMENT
import com.xebialabs.xlrelease.status.repository.persistence.data.LiveDeploymentRow
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.Date
import scala.jdk.CollectionConverters._
import scala.reflect.ClassTag
import scala.util.{Failure, Success, Try}

@Repository
@IsTransactional
class ExternalDeploymentPersistenceImpl @Autowired()(override val folderPersistence: FolderPersistence,
                                                     override val applicationPersistence: ApplicationPersistence,
                                                     override val environmentPersistence: EnvironmentPersistence,
                                                     override val configurationPersistence: ConfigurationPersistence,
                                                     implicit val ciIdService: CiIdService)
                                                    (@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                                     @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends SqlRepository with LiveDeploymentPersistence with Logging {
  private val ID_ROOT: String = CONFIGURATION.getRootNodeName + "/liveDeployments"

  override def save(deployment: LiveDeployment): LiveDeploymentRow = {
    val ciUid: CiUid = findByDeployment(deployment).map(originalDeployment => update(originalDeployment, deployment)).getOrElse(create(deployment))
    findByUid(ciUid)
      .getOrElse(throw new IllegalStateException("Saved or updated instance of Deployment not found."))
  }

  protected def create(deployment: LiveDeployment): CiUid = {
    prepareIdsForSerialization(deployment)
    val content = serialize(deployment)

    sqlInsertWithContent(
      pkName(LIVE_DEPLOYMENT.CI_UID),
      STMT_INSERT_APPLICATION_STATUS,
      liveDeploymentParams(deployment),
      LIVE_DEPLOYMENT.CONTENT -> content,
      identity
    )
  }

  private def prepareIdsForSerialization(liveDeployment: LiveDeployment)(implicit ciIdService: CiIdService): Unit = {
    Option(liveDeployment.getId) match {
      case Some(_) => liveDeployment.setId(Ids.getName(liveDeployment.getId))
      case None => liveDeployment.setId(Ids.getName(ciIdService.getUniqueId(Type.valueOf(implicitly[ClassTag[LiveDeployment]].runtimeClass), ID_ROOT)))
    }
    EnvironmentPersistence.prepareIdsForSerialization(liveDeployment.getEnvironment)
    ApplicationPersistence.prepareIdsForSerialization(liveDeployment.getApplication)
    Option(liveDeployment.getDeploymentState).map(s => Option(s.getId)).get match {
      case Some(id) => liveDeployment.getDeploymentState.setId(Ids.getName(id))
      case None => liveDeployment.getDeploymentState.setId(Ids.getName(ciIdService.getUniqueId(Type.valueOf(implicitly[ClassTag[DeploymentState]].runtimeClass), ID_ROOT)))
    }
  }

  protected def update(originalDeployment: LiveDeploymentRow, deployment: LiveDeployment): CiUid = {
    deployment.setId(originalDeployment.deploymentId)
    prepareIdsForSerialization(deployment)
    val application = deployment.getApplication
    val environment = deployment.getEnvironment
    sqlExecWithContent(
      STMT_UPDATE_APPLICATION_STATUS,
      liveDeploymentParams(deployment) + (LIVE_DEPLOYMENT.CI_UID -> originalDeployment.ciUid),
      LIVE_DEPLOYMENT.CONTENT -> serialize(deployment),
      checkCiUpdated(
        s"on folder ${deployment.getFolderId} with external UIDs " +
          s"${application.getCorrelationUid}/${environment.getCorrelationUid}"))
    originalDeployment.ciUid
  }

  private def liveDeploymentParams(deployment: LiveDeployment): Map[CiId, Any] = {
    val application = deployment.getApplication
    val environment = deployment.getEnvironment
    val state = deployment.getDeploymentState
    val namespace = Try(environment.getDeploymentTarget.getProperty("namespace")) match {
      case Success(ns) => ns
      case Failure(_) => ""
    }
    params(
      LIVE_DEPLOYMENT.DEPLOYMENT_ID -> toPersistedId(deployment.getId),
      LIVE_DEPLOYMENT.FOLDER_CI_UID -> folderPersistence.getUid(deployment.getFolderId),
      LIVE_DEPLOYMENT.APP_CI_UID -> applicationPersistence.findUidById(application.getId)
        .getOrElse(throw new IllegalStateException(s"Cannot find UID for application ${application.getId}")),
      LIVE_DEPLOYMENT.ENV_CI_UID -> environmentPersistence.findUidById(environment.getId)
        .getOrElse(throw new IllegalStateException(s"Cannot find UID for environment ${environment.getId}")),
      LIVE_DEPLOYMENT.APPLICATION_NAME -> application.getTitle,
      LIVE_DEPLOYMENT.DESTINATION -> environment.getTitle,
      LIVE_DEPLOYMENT.NAMESPACE -> namespace,
      LIVE_DEPLOYMENT.STATUS_GROUP -> state.getStatusGroup,
      LIVE_DEPLOYMENT.STATUS -> state.getStatus,
      LIVE_DEPLOYMENT.VERSION_TAG -> state.getVersionTag,
      LIVE_DEPLOYMENT.LAST_EVENT_TIME -> state.getLastChangeTime,
      LIVE_DEPLOYMENT.EVENT_SOURCE_REF -> deployment.getEventSourceId,
    )
  }
}

trait LiveDeploymentPersistence  extends PersistenceSupport with Utils with LimitOffset  with Logging {
  override implicit val dialect: Dialect

  def folderPersistence: FolderPersistence
  def applicationPersistence: ApplicationPersistence
  def environmentPersistence: EnvironmentPersistence
  def configurationPersistence: ConfigurationPersistence

  def save(deployment: LiveDeployment): LiveDeploymentRow

  private final val STMT_BASE_FIND = s"""SELECT
                                        |  f.${FOLDERS.FOLDER_ID} as ${FOLDERS.FOLDER_ID},
                                        |  e.${LIVE_DEPLOYMENT.CI_UID} as ${LIVE_DEPLOYMENT.CI_UID},
                                        |  e.${LIVE_DEPLOYMENT.DEPLOYMENT_ID} as ${LIVE_DEPLOYMENT.DEPLOYMENT_ID},
                                        |  e.${LIVE_DEPLOYMENT.APP_CI_UID} as ${LIVE_DEPLOYMENT.APP_CI_UID},
                                        |  e.${LIVE_DEPLOYMENT.DESTINATION} as ${LIVE_DEPLOYMENT.DESTINATION},
                                        |  e.${LIVE_DEPLOYMENT.ENV_CI_UID} as ${LIVE_DEPLOYMENT.ENV_CI_UID},
                                        |  e.${LIVE_DEPLOYMENT.APPLICATION_NAME} as ${LIVE_DEPLOYMENT.APPLICATION_NAME},
                                        |  e.${LIVE_DEPLOYMENT.NAMESPACE} as ${LIVE_DEPLOYMENT.NAMESPACE},
                                        |  e.${LIVE_DEPLOYMENT.STATUS_GROUP} as ${LIVE_DEPLOYMENT.STATUS_GROUP},
                                        |  e.${LIVE_DEPLOYMENT.STATUS} as ${LIVE_DEPLOYMENT.STATUS},
                                        |  e.${LIVE_DEPLOYMENT.VERSION_TAG} as ${LIVE_DEPLOYMENT.VERSION_TAG},
                                        |  e.${LIVE_DEPLOYMENT.LAST_EVENT_TIME} as ${LIVE_DEPLOYMENT.LAST_EVENT_TIME},
                                        |  e.${LIVE_DEPLOYMENT.CONTENT} as ${LIVE_DEPLOYMENT.CONTENT}
                                        | FROM ${LIVE_DEPLOYMENT.TABLE} e
                                        | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = e.${LIVE_DEPLOYMENT.FOLDER_CI_UID}"""

  private final val STMT_APPLICATION_STATUS_FETCH_UID: String =
    s"""SELECT
       |  ${LIVE_DEPLOYMENT.CI_UID}
       | FROM ${LIVE_DEPLOYMENT.TABLE} e
       | WHERE e.${LIVE_DEPLOYMENT.FOLDER_CI_UID} = :${LIVE_DEPLOYMENT.FOLDER_CI_UID} AND
       | e.${LIVE_DEPLOYMENT.APP_CI_UID} = :${LIVE_DEPLOYMENT.APP_CI_UID} AND
       | e.${LIVE_DEPLOYMENT.ENV_CI_UID} = :${LIVE_DEPLOYMENT.ENV_CI_UID}
       |""".stripMargin

  def fetchUid(deployment: LiveDeployment): Option[CiUid] =
    fetchUid(deployment.getFolderId,
      deployment.getApplication.getId,
      deployment.getEnvironment.getId)

  private final val STMT_LIVE_DEPLOYMENT_BY_UNIQUE_IDS =
    s"""$STMT_BASE_FIND
       | WHERE e.${LIVE_DEPLOYMENT.FOLDER_CI_UID} = :${LIVE_DEPLOYMENT.FOLDER_CI_UID}
       | AND e.${LIVE_DEPLOYMENT.APP_CI_UID} = :${LIVE_DEPLOYMENT.APP_CI_UID}
       | AND e.${LIVE_DEPLOYMENT.ENV_CI_UID} = :${LIVE_DEPLOYMENT.ENV_CI_UID}
       |""".stripMargin
  def fetchLiveDeploymentId(folderId: String, applicationId: String, environmentId: CiId): Option[CiId] = {
    sqlQuery(STMT_LIVE_DEPLOYMENT_BY_UNIQUE_IDS,
      params(
        LIVE_DEPLOYMENT.FOLDER_CI_UID -> folderPersistence.getUid(folderId),
      LIVE_DEPLOYMENT.APP_CI_UID -> applicationPersistence.findUidById(applicationId)
        .getOrElse(throw new IllegalStateException(s"Cannot find UID for application $applicationId")),
      LIVE_DEPLOYMENT.ENV_CI_UID -> environmentPersistence.findUidById(environmentId)
        .getOrElse(throw new IllegalStateException(s"Cannot find UID for environment $environmentId"))),
      externalDeploymentRecordRowMapper
    ).headOption.map(_.deploymentId)
  }

  private def fetchUid(folderId: String, applicationId: CiId, environmentId: CiId): Option[CiUid] =
        sqlQuery(STMT_APPLICATION_STATUS_FETCH_UID, params(
          LIVE_DEPLOYMENT.FOLDER_CI_UID -> folderPersistence.getUid(folderId),
          LIVE_DEPLOYMENT.APP_CI_UID -> applicationPersistence.findUidById(applicationId)
            .getOrElse(throw new IllegalStateException(s"Cannot find UID for application $applicationId")),
          LIVE_DEPLOYMENT.ENV_CI_UID -> environmentPersistence.findUidById(environmentId)
            .getOrElse(throw new IllegalStateException(s"Cannot find UID for environment $environmentId"))
        ), rs => CiUid(rs.getInt(LIVE_DEPLOYMENT.CI_UID))).headOption

  private final val STMT_APPLICATION_CHECK_AGE: String =
    s"""SELECT
       |  COUNT(1)
       | FROM ${LIVE_DEPLOYMENT.TABLE} e
       | WHERE e.${LIVE_DEPLOYMENT.FOLDER_CI_UID} = :${LIVE_DEPLOYMENT.FOLDER_CI_UID}
       | AND e.${LIVE_DEPLOYMENT.LAST_EVENT_TIME} > :${LIVE_DEPLOYMENT.LAST_EVENT_TIME}
       |""".stripMargin

  def exists(folderId: String, maxAge: Long): Boolean =
    sqlQuery(
      STMT_APPLICATION_CHECK_AGE,
      params(
        LIVE_DEPLOYMENT.FOLDER_CI_UID -> folderPersistence.getUid(folderId),
        LIVE_DEPLOYMENT.LAST_EVENT_TIME -> new Date(System.currentTimeMillis - maxAge),
      ),
      _.getInt(1) > 0
    ).head

  private final val STMT_COUNT_BY_ENDPOINT_IDS_AND_FILTER =
    s"""SELECT COUNT(1)
       | FROM ${LIVE_DEPLOYMENT.TABLE} e
       | WHERE e.${LIVE_DEPLOYMENT.FOLDER_CI_UID} = (:folderId)
       |  AND (
       |    LOWER(e.${LIVE_DEPLOYMENT.APPLICATION_NAME}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.DESTINATION}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.NAMESPACE}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.STATUS}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.VERSION_TAG}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  def count(folderId: String, condition: String): Int = sqlQuery(
        STMT_COUNT_BY_ENDPOINT_IDS_AND_FILTER,
        params("folderId" -> folderPersistence.getUid(folderId), "condition" -> s"%$condition%"),
        _.getInt(1)
      ).head

  private final val STMT_FIND_BY_ENDPOINT_IDS =
    s"""$STMT_BASE_FIND
       | WHERE e.${LIVE_DEPLOYMENT.FOLDER_CI_UID} = (:folderUid)
       | AND (
       |    LOWER(e.${LIVE_DEPLOYMENT.APPLICATION_NAME}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.DESTINATION}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.NAMESPACE}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.STATUS}) LIKE LOWER(:condition)
       |    OR LOWER(e.${LIVE_DEPLOYMENT.VERSION_TAG}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  def findOnFolder(folderId: String,
                   limit: Int, offset: Long,
                   orderBy: ExternalDeploymentOrderMode,
                   direction: ExternalDeploymentOrderDirection,
                   condition: String = ""): Vector[LiveDeploymentRow] = {
    val sqlStatement =
      if (limit > 0) {
        addLimitAndOffset(
          addOrdering(
            STMT_FIND_BY_ENDPOINT_IDS,
            orderBy,
            direction
          ), limit, offset
        )
      }
      else {
        addOrdering(
          STMT_FIND_BY_ENDPOINT_IDS,
          orderBy,
          direction
        )
      }

    sqlQuery(
      sqlStatement,
      params("folderUid" -> folderPersistence.getUid(folderId), "condition" -> s"%$condition%"),
      externalDeploymentRecordRowMapper
    ).toVector
  }

  private final val STMT_FIND_BY_EXTERNAL_DEPLOYMENT_UID =
    s"""$STMT_BASE_FIND
       | WHERE e.${LIVE_DEPLOYMENT.CI_UID} = :${LIVE_DEPLOYMENT.CI_UID}
       |""".stripMargin

  private val externalDeploymentRecordRowMapper: RowMapper[LiveDeploymentRow] = (rs: ResultSet, _: Int) => LiveDeploymentRow.fromResultSet(rs)

  def findByUid(id: CiUid): Option[LiveDeploymentRow] =
    sqlQuery(
      STMT_FIND_BY_EXTERNAL_DEPLOYMENT_UID,
      params(
        LIVE_DEPLOYMENT.CI_UID -> id
      ),
      externalDeploymentRecordRowMapper
    ).headOption


  private final val STMT_FIND_BY_EXTERNAL_DEPLOYMENT_PARAMS =
    s"""$STMT_BASE_FIND
       | WHERE e.${LIVE_DEPLOYMENT.FOLDER_CI_UID} = :${LIVE_DEPLOYMENT.FOLDER_CI_UID} AND
       |   e.${LIVE_DEPLOYMENT.APP_CI_UID} = :${LIVE_DEPLOYMENT.APP_CI_UID} AND
       |   e.${LIVE_DEPLOYMENT.ENV_CI_UID} = :${LIVE_DEPLOYMENT.ENV_CI_UID}
       |""".stripMargin

  def findByDeployment(deployment: LiveDeployment): Option[LiveDeploymentRow] =
    sqlQuery(
      STMT_FIND_BY_EXTERNAL_DEPLOYMENT_PARAMS,
      params(
        LIVE_DEPLOYMENT.FOLDER_CI_UID -> folderPersistence.getUid(deployment.getFolderId),
        LIVE_DEPLOYMENT.APP_CI_UID -> applicationPersistence.findUidById(deployment.getApplication.getId)
          .getOrElse(throw new IllegalStateException(s"Cannot find UID for application ${deployment.getApplication.getId}")),
        LIVE_DEPLOYMENT.ENV_CI_UID -> environmentPersistence.findUidById(deployment.getEnvironment.getId)
          .getOrElse(throw new IllegalStateException(s"Cannot find UID for environment ${deployment.getEnvironment.getId}"))
      ),
      externalDeploymentRecordRowMapper
    ).headOption

  protected final val STMT_INSERT_APPLICATION_STATUS =
    s"""INSERT INTO ${LIVE_DEPLOYMENT.TABLE} (
       |   ${LIVE_DEPLOYMENT.DEPLOYMENT_ID},
       |   ${LIVE_DEPLOYMENT.FOLDER_CI_UID},
       |   ${LIVE_DEPLOYMENT.APP_CI_UID},
       |   ${LIVE_DEPLOYMENT.APPLICATION_NAME},
       |   ${LIVE_DEPLOYMENT.ENV_CI_UID},
       |   ${LIVE_DEPLOYMENT.DESTINATION},
       |   ${LIVE_DEPLOYMENT.NAMESPACE},
       |   ${LIVE_DEPLOYMENT.STATUS_GROUP},
       |   ${LIVE_DEPLOYMENT.STATUS},
       |   ${LIVE_DEPLOYMENT.VERSION_TAG},
       |   ${LIVE_DEPLOYMENT.LAST_EVENT_TIME},
       |   ${LIVE_DEPLOYMENT.CONTENT},
       |   ${LIVE_DEPLOYMENT.EVENT_SOURCE_REF}
       |   )
       | VALUES(
       |   :${LIVE_DEPLOYMENT.DEPLOYMENT_ID},
       |   :${LIVE_DEPLOYMENT.FOLDER_CI_UID},
       |   :${LIVE_DEPLOYMENT.APP_CI_UID},
       |   :${LIVE_DEPLOYMENT.APPLICATION_NAME},
       |   :${LIVE_DEPLOYMENT.ENV_CI_UID},
       |   :${LIVE_DEPLOYMENT.DESTINATION},
       |   :${LIVE_DEPLOYMENT.NAMESPACE},
       |   :${LIVE_DEPLOYMENT.STATUS_GROUP},
       |   :${LIVE_DEPLOYMENT.STATUS},
       |   :${LIVE_DEPLOYMENT.VERSION_TAG},
       |   :${LIVE_DEPLOYMENT.LAST_EVENT_TIME},
       |   :${LIVE_DEPLOYMENT.CONTENT},
       |   :${LIVE_DEPLOYMENT.EVENT_SOURCE_REF}
       |   )""".stripMargin

  protected final val STMT_UPDATE_APPLICATION_STATUS =
    s"""UPDATE ${LIVE_DEPLOYMENT.TABLE}
       | SET
       | ${LIVE_DEPLOYMENT.APPLICATION_NAME} = :${LIVE_DEPLOYMENT.APPLICATION_NAME},
       | ${LIVE_DEPLOYMENT.DESTINATION} = :${LIVE_DEPLOYMENT.DESTINATION},
       | ${LIVE_DEPLOYMENT.NAMESPACE} = :${LIVE_DEPLOYMENT.NAMESPACE},
       | ${LIVE_DEPLOYMENT.STATUS_GROUP} = :${LIVE_DEPLOYMENT.STATUS_GROUP},
       | ${LIVE_DEPLOYMENT.STATUS} = :${LIVE_DEPLOYMENT.STATUS},
       | ${LIVE_DEPLOYMENT.VERSION_TAG} = :${LIVE_DEPLOYMENT.VERSION_TAG},
       | ${LIVE_DEPLOYMENT.LAST_EVENT_TIME} = :${LIVE_DEPLOYMENT.LAST_EVENT_TIME},
       | ${LIVE_DEPLOYMENT.CONTENT} = :${LIVE_DEPLOYMENT.CONTENT}
       | WHERE ${LIVE_DEPLOYMENT.CI_UID} = :${LIVE_DEPLOYMENT.CI_UID}
       |""".stripMargin

  private val STMT_DELETE_FOR_EXTERNAL_DEPLOYMENT_UID =
    s"""
       |DELETE FROM ${LIVE_DEPLOYMENT.TABLE}
       |WHERE ${LIVE_DEPLOYMENT.DEPLOYMENT_ID} = :${LIVE_DEPLOYMENT.DEPLOYMENT_ID}
     """.stripMargin

  def delete(deploymentId: CiId): Unit = {
    sqlExec(STMT_DELETE_FOR_EXTERNAL_DEPLOYMENT_UID,
      params(
        LIVE_DEPLOYMENT.DEPLOYMENT_ID -> toPersistedId(deploymentId)
      ),
      _.execute())
  }

  private def addOrdering(query: String, order: ExternalDeploymentOrderMode, direction: ExternalDeploymentOrderDirection): String =
    getOrderCriterionColumn(order).mkString(
      s"$query ORDER BY ",
      s" ${getOrderCriterionDirection(direction)}, ",
      s" ${getOrderCriterionDirection(direction)}")

  private def getOrderCriterionDirection(direction: ExternalDeploymentOrderDirection): String =
    direction match {
      case ExternalDeploymentOrderDirection.ASC => "ASC"
      case ExternalDeploymentOrderDirection.DESC => "DESC"
    }

  private def getOrderCriterionColumn(order: ExternalDeploymentOrderMode): List[String] =
    order match {
      case ExternalDeploymentOrderMode.APPLICATION =>
        List(LIVE_DEPLOYMENT.APPLICATION_NAME,LIVE_DEPLOYMENT.VERSION_TAG, LIVE_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.DESTINATION =>
        List(LIVE_DEPLOYMENT.DESTINATION, LIVE_DEPLOYMENT.NAMESPACE, LIVE_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.VERSION =>
        List(LIVE_DEPLOYMENT.VERSION_TAG, LIVE_DEPLOYMENT.APPLICATION_NAME, LIVE_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.STATUS =>
        List(LIVE_DEPLOYMENT.STATUS_GROUP, LIVE_DEPLOYMENT.STATUS, LIVE_DEPLOYMENT.LAST_EVENT_TIME)
      case ExternalDeploymentOrderMode.TIME =>
        List(LIVE_DEPLOYMENT.LAST_EVENT_TIME)
    }

  private final val STMT_DELETE_BY_CONFIG_ID =
    s"""
       | DELETE FROM ${LIVE_DEPLOYMENT.TABLE}
       | WHERE ${LIVE_DEPLOYMENT.EVENT_SOURCE_REF} = :${LIVE_DEPLOYMENT.EVENT_SOURCE_REF}
       |""".stripMargin

  private final val STMT_DELETE_BY_CONFIG_ID_AND_DEPLOYMENT_ID_NOT_IN =
    s"""
       | DELETE FROM ${LIVE_DEPLOYMENT.TABLE}
       | WHERE ${LIVE_DEPLOYMENT.EVENT_SOURCE_REF} = :${LIVE_DEPLOYMENT.EVENT_SOURCE_REF}
       | AND ${LIVE_DEPLOYMENT.DEPLOYMENT_ID} NOT IN (:deploymentIds)
       |""".stripMargin

  def deleteByEventSourceIdAndDeploymentIdNotIn(eventSourceId: CiId, excludedDeploymentIds: List[String]): Unit = {
    Option(excludedDeploymentIds) match {
      case Some(ids) if ids.nonEmpty =>
        sqlExec(STMT_DELETE_BY_CONFIG_ID_AND_DEPLOYMENT_ID_NOT_IN,
          params(
            LIVE_DEPLOYMENT.EVENT_SOURCE_REF -> eventSourceId,
            "deploymentIds" -> excludedDeploymentIds.map(toPersistedId).asJava
          ),
          _.execute())
      case _ =>
        sqlExec(STMT_DELETE_BY_CONFIG_ID,
          params(
            LIVE_DEPLOYMENT.EVENT_SOURCE_REF -> eventSourceId
          ),
          _.execute())
    }
  }
}
