import React, { memo, useEffect, useState } from 'react';
import { Folder } from '../../../../../../../../../../core/xlr-ui/app/types';
import { ActionToolbarPortalContainer } from '../../../../../../../../../../core/xlr-ui/app/features/main-navigation/action-toolbar/components/action-toolbar.portal';
import { DotButton, DotColumnHeader, DotIcon, DotLink, DotTable, DotThemeProvider, DotTypography, Order } from '@digital-ai/dot-components';
import { Search } from '../../../../../../../../../../core/xlr-ui/app/react/components/search/search';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import './live-deployment-configuration-table.component.less';
import { ExternalDeploymentPage, PluginInfo } from '../../external-deployment.types';
import { AvatarWithTextCell } from '@digital-ai/dot-components/src/lib/components/table/TableCell';
import { folderExternalDeployments } from '../../ducks/external-deployments.reducer';
import {
    getConfigurationCondition,
    getConfigurationPage,
    getIsLoadingSelector,
    getLiveDeploymentConfigsCountSelector,
    getLiveDeploymentConfigsSelector,
    getValidServerCards,
} from '../../ducks/external-deployments.selectors';
import { CONNECTION_COLUMN_ID, FILTERS_COLUMN_ID, NAME_COLUMN_ID } from '../../constants';
import capitalize from 'lodash/capitalize';
import startCase from 'lodash/startCase';

export interface LiveDeploymentConfigurationTableProps {
    folder: Folder;
}

const { setIsLiveDeploymentConfigurationOpen, setConfigurationCondition, setConfigurationPage, loadLiveDeploymentsConfiguration, setConfigurationFolderId } =
    folderExternalDeployments.actions;

const LiveDeploymentConfigurationTableComponent = ({ folder }: LiveDeploymentConfigurationTableProps) => {
    const condition = useAppSelector(getConfigurationCondition);
    const page = useAppSelector(getConfigurationPage);
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const count = useAppSelector(getLiveDeploymentConfigsCountSelector);
    const liveDeploymentConfigs = useAppSelector(getLiveDeploymentConfigsSelector);
    const validServerCards: PluginInfo[] = useAppSelector(getValidServerCards);

    const dispatch = useAppDispatch();

    const [filterTimeout, setFilterTimeout] = useState<ReturnType<typeof setTimeout> | undefined>(undefined);

    useEffect(() => {
        dispatch(setConfigurationFolderId(folder.id));
        dispatch(loadLiveDeploymentsConfiguration());
    }, []);

    const onDisplayFilterChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const value = event.target.value;
        if (filterTimeout) {
            clearTimeout(filterTimeout);
            setFilterTimeout(undefined);
        }
        setFilterTimeout(
            setTimeout(() => {
                dispatch(setConfigurationCondition(value));
            }, 500),
        );
    };

    const onPageChange = (newPage: ExternalDeploymentPage) => {
        dispatch(setConfigurationPage(newPage));
    };

    return (
        <>
            <ActionToolbarPortalContainer></ActionToolbarPortalContainer>
            <DotThemeProvider>
                <div className="live-deployment-configuration-table">
                    <div className="live-deployment-configuration-table-title">
                        <DotButton onClick={() => dispatch(setIsLiveDeploymentConfigurationOpen(false))} type="text">
                            <DotIcon iconId="back" />
                        </DotButton>
                        <DotTypography variant="h1">Live deployments configuration</DotTypography>
                        <Search
                            defaultValue={condition}
                            inputPlaceholder="Filter by values"
                            key={condition}
                            onFilterChange={onDisplayFilterChange}
                            searchClassNames={['mtm', 'mbm']}
                        />
                    </div>
                    <DotTable
                        columns={headerDefinitions}
                        count={count}
                        data={liveDeploymentConfigs.map((config, index) => {
                            const pluginInfo = validServerCards.find((validServerCard) => config.connectionType === validServerCard.type);

                            return {
                                id: index.toString(),
                                rowData: {
                                    name: {
                                        avatarImageSrc: pluginInfo?.iconLocation,
                                        mainText: config.title,
                                    } as AvatarWithTextCell,
                                    connection: (
                                        <>
                                            <DotTypography variant="body1">{config.connectionTitle ? config.connectionTitle : '-'}</DotTypography>
                                            <DotLink href={config.connectionUrl} target="_blank" tooltip="Open in new tab">
                                                {config.connectionUrl}
                                            </DotLink>
                                        </>
                                    ),
                                    filters: (
                                        <>
                                            {Object.entries(config.filterProperties).map(([key, value]) => (
                                                <DotTypography key={index} variant="body1">
                                                    {capitalize(startCase(key))}: {value}
                                                </DotTypography>
                                            ))}
                                        </>
                                    ),
                                },
                            };
                        })}
                        loading={isLoading}
                        onUpdateData={(order: Order, orderBy: string) =>
                            onPageChange({
                                ...page,
                                order,
                                orderBy,
                                folderId: folder.id,
                            })
                        }
                        order={page.order}
                        orderBy={page.orderBy}
                        rowsPerPage={10}
                    />
                </div>
            </DotThemeProvider>
        </>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: 'left',
        id: NAME_COLUMN_ID,
        label: 'Name',
        sortable: true,
        truncate: true,
        width: '20%',
    },
    {
        id: CONNECTION_COLUMN_ID,
        label: 'Connection',
        truncate: true,
        width: '35%',
    },
    {
        id: FILTERS_COLUMN_ID,
        label: 'Filter properties',
        sortable: false,
        width: '35%',
    },
    {
        id: 'actions',
        width: '5%',
    },
];

export const LiveDeploymentConfigurationTable = memo(LiveDeploymentConfigurationTableComponent);
