import React from 'react';
import AceEditor from 'react-ace';
import { ReactWrapper } from 'enzyme';
import { mountWithTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ENDPOINT_TYPE } from '../../../constants';
import { DiscoverModal } from './discover-modal-component';
import { DotDialog, DotAlertBanner, DotIconButton } from '@digital-ai/dot-components';
import { toaster } from '../../../ducks/external-deployments.saga';

describe('Discover modal component', () => {
    let successModal: ReactWrapper;
    let errorModal: ReactWrapper;
    let argoSuccessModal: ReactWrapper;
    const onModalClose = jest.fn();
    const onModalSubmit = jest.fn();
    const successProps = {
        configureDetails: {
            details: '',
            message: 'Created Deploy webhook successfully',
            success: true,
            url: 'Deploy-webhook-url',
            webhookSourceId: 'Deploy-webhook-source-id',
        },
        isModalOpen: true,
        onModalClose,
        onModalSubmit,
        type: ENDPOINT_TYPE.deploy,
    };
    const errorProps = {
        configureDetails: {
            details: '',
            message: '',
            success: false,
            url: '',
            webhookSourceId: 'Deploy-webhook-source-id',
        },
        isModalOpen: true,
        onModalClose,
        onModalSubmit,
        type: ENDPOINT_TYPE.deploy,
    };
    const argoSuccessProps = {
        configureDetails: {
            details: 'Yaml file to copy',
            message: 'Generated yaml file',
            success: true,
            url: '',
            webhookSourceId: '',
        },
        isModalOpen: true,
        onModalClose,
        onModalSubmit,
        type: ENDPOINT_TYPE.argoCd,
    };

    beforeEach(() => {
        successModal = mountWithTheme(<DiscoverModal {...successProps} />);
        errorModal = mountWithTheme(<DiscoverModal {...errorProps} />);
        argoSuccessModal = mountWithTheme(<DiscoverModal {...argoSuccessProps} />);
    });

    afterEach(() => {
        successModal.unmount();
        errorModal.unmount();
        argoSuccessModal.unmount();
    });

    it('should display success Discover modal', () => {
        window.open = jest.fn();
        expect(argoSuccessModal.find(DotDialog).props().title).toStrictEqual('Configuration ready');
        expect(argoSuccessModal.find(DotAlertBanner)).toExist();
        expect(argoSuccessModal.find(DotAlertBanner).props().severity).toStrictEqual('success');
        expect(argoSuccessModal.find(DotAlertBanner).find(DotIconButton)).toExist();
        expect(argoSuccessModal.find(DotDialog).find(`.configuration-details-wrapper`)).toExist();
        argoSuccessModal.find(DotIconButton).at(0).invoke('onClick')?.('' as never);
        expect(window.open).toHaveBeenCalledWith(argoSuccessProps.configureDetails.url, '_blank');
    });

    it('should open icon for new tab in modal', () => {
        window.open = jest.fn().mockReturnValue({});
        expect(successModal.find(`button[data-testid="modal-new-tab"]`)).toExist();
        successModal.find(`button[data-testid="modal-new-tab"]`).simulate('click');
        expect(window.open).toHaveBeenCalledWith('Deploy-webhook-url', '_blank');
    });

    it('should handle view CI in Deploy', () => {
        window.open = jest.fn().mockReturnValue({});
        const button = successModal.find('[data-testid="configure-success-banner"] button');
        button.simulate('click');
        expect(window.open).toHaveBeenCalledWith('Deploy-webhook-url', '_blank');
    });

    it('should display error Discover modal', () => {
        expect(errorModal.find(DotDialog).props().title).toStrictEqual('Configuration failed');
        expect(errorModal.find(DotAlertBanner).props().severity).toStrictEqual('error');
    });

    it('should display cancel and submit buttons', () => {
        expect(successModal.find(DotDialog).props().onSubmit).toBeDefined();
        expect(successModal.find(DotDialog).props().onCancel).toBeDefined();
        expect(successModal.find(DotDialog).props().submitButtonProps?.label).toEqual('Next');
    });

    it('should display ace editor in argo success modal', () => {
        expect(argoSuccessModal.find(AceEditor)).toExist();
        expect(argoSuccessModal.find(AceEditor).props().value).toEqual('Yaml file to copy');
    });

    it('should display copy icon in argo success modal', () => {
        Object.assign(navigator, {
            clipboard: {
                writeText: () => Promise.resolve(),
            },
        });
        const success = jest.fn();
        Object.assign(toaster, { success });
        jest.spyOn(navigator.clipboard, 'writeText');
        expect(argoSuccessModal.find('button[data-testid="copy-btn"]')).toExist();
        argoSuccessModal.find('button[data-testid="copy-btn"]').simulate('click');
        expect(navigator.clipboard.writeText).toHaveBeenCalledWith('Yaml file to copy');
    });

    it('should handle copy error', () => {
        Object.assign(navigator, {
            clipboard: {
                writeText: () =>
                    new Promise((resolve, reject) => {
                        reject('error');
                    }),
            },
        });
        jest.spyOn(navigator.clipboard, 'writeText');
        expect(argoSuccessModal.find('button[data-testid="copy-btn"]')).toExist();
        argoSuccessModal.find('button[data-testid="copy-btn"]').simulate('click');
        expect(navigator.clipboard.writeText).toHaveBeenCalledWith('Yaml file to copy');
    });
});
