package com.xebialabs.xlrelease.status.sse.service

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.joda.JodaModule
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.service.SseService
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

trait ServerSentEventsService {
  def add(): Unit

  def send(id: String, event: ExternalDeploymentEvent): Unit

  def remove(): Unit
}

@Service
class ServerSentEventsServiceImpl @Autowired()(sseService: SseService) extends ServerSentEventsService with Logging {

  private val mapper = new ObjectMapper()
  mapper.registerModule(DefaultScalaModule)
  mapper.registerModule(new JodaModule)

  private val applicationEventId = "applications-sse"

  override def add(): Unit = {
    sseService.subscribeTopicToUser(applicationEventId)
  }


  override def remove(): Unit = {
    sseService.unsubscribeTopicToUser(applicationEventId)
  }

  def send(id: String, event: ExternalDeploymentEvent): Unit = {
    val sseEvent = buildSseEvent(EndpointExternalDeploymentEvent(id, event))
    sseService.sendEvent(applicationEventId, sseEvent._1, sseEvent._2)
  }

  private def buildSseEvent(event: EndpointExternalDeploymentEvent): (String, String) = {

    var name: String = ""

    event.state match {
      case e: CreateStatusEvent if isApplicationEvent(e) => name = "application-created"
      case e: DeleteStatusEvent if isApplicationEvent(e) => name = "application-deleted"
      case _: CreateStatusEvent => name = "application-package-created"
      case _: DeleteStatusEvent => name = "application-package-deleted"
      case _: UpdateStatusEvent => name = "application-changed"
      case _ => // ignore
    }

    (name, mapper.writeValueAsString(event))
  }

  private def isApplicationEvent(event: BaseExternalDeploymentEvent) = {
    Option(event.destination).isDefined || Option(event.namespace).isDefined
  }
}

sealed trait ServerSentEvent {
  val endpointId: String
}

case class EndpointExternalDeploymentEvent(override val endpointId: String, state: ExternalDeploymentEvent) extends ServerSentEvent
