import React, { memo, ChangeEvent, useState, useEffect } from 'react';
import {
    DotColumnHeader,
    DotIcon,
    DotLink,
    DotTable,
    DotTablePagination,
    DotThemeProvider,
    DotTooltip,
    DotTypography,
    Order,
    RowsPerPageOption,
} from '@digital-ai/dot-components';
import { ExternalDeployment as ExternalDeploymentType, ExternalDeploymentPage } from '../external-deployment.types';
import { folderExternalDeployments } from '../ducks/external-deployments.reducer';
import { Folder } from '../../../../../../../../../core/xlr-ui/app/types';
import {
    getConditionSelector,
    getConnectionErrorsSelector,
    getConnectionServersSelector,
    getDisconnectedServers,
    getExternalDeploymentCountSelector,
    getExternalDeploymentPageSelector,
    getExternalDeploymentsSelector,
    getIsLoadingSelector,
    getServersSelector,
    getValidServerCards,
} from '../ducks/external-deployments.selectors';
import './external-deployments-table.component.less';
import { ExternalDeployment } from './external-deployment.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import { ExternalDeploymentComponentPortal } from './external-deployment-component.portal';
import {
    APPLICATION_COLUMN_ID,
    DEFAULT_PAGE,
    DESTINATION_COLUMN_ID,
    STATUS_COLUMN_ID,
    STATUS_HTTP_CONNECTION,
    TIME_COLUMN_ID,
    VERSION_COLUMN_ID,
} from '../constants';
import { Search } from '../../../../../../../../../core/xlr-ui/app/react/components/search/search';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { PluginInfo } from './wizard/basic-wizard/application-discovery-basic.component';
import { CreateServerForm } from './wizard/create-server/application-create-server-component';

const { navigateTo } = navigation.actions;

export interface ExternalDeploymentsTableProps {
    folder: Folder;
    onViewChange: () => void;
}

const { getFilters, loadExternalDeployments, setCondition, storeFilters, setPage, subscribeToSseStream, unsubscribeFromSseStream, getAvailablePlugins } =
    folderExternalDeployments.actions;

const ExternalDeploymentsTableComponent = ({ folder, onViewChange }: ExternalDeploymentsTableProps) => {
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const externalDeployments = useAppSelector(getExternalDeploymentsSelector);
    const connectionServers = useAppSelector(getConnectionServersSelector);
    const count = useAppSelector(getExternalDeploymentCountSelector);
    const page = useAppSelector(getExternalDeploymentPageSelector);
    const condition = useAppSelector(getConditionSelector);
    const connectionErrors = useAppSelector(getConnectionErrorsSelector);
    const validServerCards: PluginInfo[] = useAppSelector(getValidServerCards);
    const disconnectedServers = useAppSelector(getDisconnectedServers);
    const servers = useAppSelector(getServersSelector);
    const dispatch = useAppDispatch();

    const [filterTimeout, setFilterTimeout] = useState<ReturnType<typeof setTimeout> | undefined>(undefined);
    const [selectedErrorServer, setSelectedErrorServer] = useState<string>('');
    const [isServerFormOpen, setServerFormOpen] = useState(false);

    useEffect(() => {
        dispatch(getFilters(folder.id));
        dispatch(subscribeToSseStream());
        dispatch(getAvailablePlugins(STATUS_HTTP_CONNECTION));
        return () => {
            dispatch(unsubscribeFromSseStream());
        };
    }, []);

    useEffect(() => {
        dispatch(storeFilters());
    }, [page, condition]);

    useEffect(() => {
        dispatch(loadExternalDeployments());
    }, [servers]);

    const onPageChange = (newPage: ExternalDeploymentPage) => {
        dispatch(setPage(newPage));
        dispatch(loadExternalDeployments());
    };

    const onDisplayFilterChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const value = event.target.value;
        if (filterTimeout) {
            clearTimeout(filterTimeout);
            setFilterTimeout(undefined);
        }
        setFilterTimeout(
            setTimeout(() => {
                dispatch(setCondition(value));
            }, 500),
        );
    };

    const onServerErrorClick = (err: string) => {
        const errServer = disconnectedServers.find((s) => err.includes(s));
        if (errServer) {
            setSelectedErrorServer(errServer);
            setServerFormOpen(true);
        }
    };

    const onServerFormClose = () => {
        setServerFormOpen(false);
    };

    return (
        <>
            {isServerFormOpen ? (
                <DotThemeProvider>
                    <CreateServerForm closeForm={() => onServerFormClose()} folder={folder} serverId={selectedErrorServer} />
                </DotThemeProvider>
            ) : (
                <>
                    <ExternalDeploymentComponentPortal
                        folder={folder}
                        onViewChange={onViewChange}
                        openApplicationSources={() =>
                            dispatch(navigateTo({ pathSuffix: `folders/${folder.id}/application-pipelines/external-deployments/application-sources` }))
                        }
                        reSyncEnabled={true}
                    />
                    <DotThemeProvider>
                        <div className="external-deployments-table">
                            <div className="external-deployments-table-title">
                                <DotTypography variant="h1">Application deployments ({count})</DotTypography>
                                <Search
                                    defaultValue={condition}
                                    inputPlaceholder="Type to filter"
                                    key={condition}
                                    onFilterChange={onDisplayFilterChange}
                                    searchClassNames={['mtm', 'mbm']}
                                />
                            </div>
                            {connectionErrors.length > 0 && (
                                <DotTooltip
                                    className="external-deployments-connection-errors"
                                    placement="bottom-start"
                                    title={
                                        <div>
                                            {connectionErrors.map((e) => (
                                                <div key={e}>
                                                    <DotLink onClick={() => onServerErrorClick(e)}>
                                                        <p style={{ color: 'white', textDecoration: 'underline' }}>{e}</p>
                                                    </DotLink>
                                                </div>
                                            ))}
                                        </div>
                                    }
                                >
                                    <DotTypography>
                                        <DotIcon iconId="error-solid" />
                                        {connectionErrors.length} disconnected servers
                                    </DotTypography>
                                </DotTooltip>
                            )}
                            <DotTable
                                className="external-deployments-table-header"
                                columns={headerDefinitions}
                                data={[]}
                                loading={isLoading}
                                onUpdateData={(order: Order, orderBy: string) =>
                                    onPageChange({
                                        ...page,
                                        folderId: folder.id,
                                        order,
                                        orderBy,
                                    })
                                }
                                order={page.order}
                                orderBy={page.orderBy}
                            />
                            {count > 0
                                ? externalDeployments.map((externalDeployment: ExternalDeploymentType) => {
                                      const connectionServer = connectionServers.get(externalDeployment.endpointId);
                                      const pluginInfo = validServerCards.find((validServerCard) => connectionServer?.type === validServerCard.type);

                                      return (
                                          connectionServer &&
                                          pluginInfo && (
                                              <ExternalDeployment
                                                  connectionServer={connectionServer}
                                                  externalDeployment={externalDeployment}
                                                  key={`${externalDeployment.applicationUid}-${externalDeployment.state.destination}`}
                                                  pluginInfo={pluginInfo}
                                              />
                                          )
                                      );
                                  })
                                : condition &&
                                  condition.length > 0 && (
                                      <div className="empty-external-deployments-table">
                                          No applications match the current filter criteria. Consider adjusting the filter values.
                                      </div>
                                  )}
                            <DotTablePagination
                                count={count}
                                onPageChange={(newPage: number) =>
                                    onPageChange({
                                        ...page,
                                        folderId: folder.id,
                                        page: newPage,
                                    })
                                }
                                onRowsPerPageChange={(event: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) =>
                                    onPageChange({
                                        ...page,
                                        folderId: folder.id,
                                        page: DEFAULT_PAGE,
                                        resultsPerPage: parseInt(event.target.value) as RowsPerPageOption,
                                    })
                                }
                                page={page.page}
                                rowsPerPage={page.resultsPerPage}
                                typography="body1"
                            />
                        </div>
                    </DotThemeProvider>
                </>
            )}
        </>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: 'left',
        id: 'icon',
        sortable: false,
        truncate: false,
        width: '3%',
    },
    {
        align: undefined,
        id: APPLICATION_COLUMN_ID,
        label: 'Application name',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: DESTINATION_COLUMN_ID,
        label: 'Environment',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: VERSION_COLUMN_ID,
        label: 'Version',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: STATUS_COLUMN_ID,
        label: 'Status',
        truncate: false,
        width: '16%',
    },
    {
        align: undefined,
        id: TIME_COLUMN_ID,
        label: 'Updated',
        truncate: false,
        width: '16%',
    },
    {
        align: 'right',
        id: 'deeplink',
        sortable: false,
        truncate: false,
        width: '5%',
    },
];

export const ExternalDeploymentsTable = memo(ExternalDeploymentsTableComponent);
