package com.xebialabs.xlrelease.applications.management.repository.persistence

import com.xebialabs.xlrelease.applications.management.repository.persistence.ManagedApplicationSchema.MANAGED_APPLICATION
import com.xebialabs.xlrelease.applications.management.service.ManagedApplication
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{CONFIGURATIONS, FOLDERS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{FolderPersistence, PersistenceSupport, Utils}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.{Date, UUID}

trait ApplicationsManagementPersistence extends PersistenceSupport with Utils with LimitOffset with Logging {


  override implicit val dialect: Dialect

  def configurationPersistence: ConfigurationPersistence

  def folderPersistence: FolderPersistence

  def findById(applicationId: String): Option[ManagedApplication]

  def findForFolder(folderId: String, limit: Long, offset: Long, condition: String): Vector[ManagedApplication]

  def findWithFolderConnections: Vector[ManagedApplication]

  def countForFolder(folderId: String): Int

  def countForFolderWithCondition(folderId: String, condition: String): Int

  def create(managedApplication: ManagedApplication): String

  def update(managedApplication: ManagedApplication): Unit

  def delete(applicationId: String): Unit

  def existsByApplicationReferenceAndFolderId(applicationReference: String, folderId: String): Boolean

  def findConnectionFolderId(applicationId: String): Option[String]
}

@Repository
@IsTransactional
class ApplicationsManagementPersistenceImpl @Autowired()(override val configurationPersistence: ConfigurationPersistence,
                                                         override val folderPersistence: FolderPersistence)
                                                        (@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                                         @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends SqlRepository with ApplicationsManagementPersistence with Logging {

  override def findById(applicationId: String): Option[ManagedApplication] = {
    sqlQuery(
      STMT_FIND_BY_ID,
      params(MANAGED_APPLICATION.ID -> applicationId),
      managedApplicationRecordRowMapper
    ).headOption
  }

  override def findForFolder(folderId: String, limit: Long, offset: Long, condition: String = ""): Vector[ManagedApplication] = {
    val sqlStatement =
      if (limit > 0) {
        addLimitAndOffset(STMT_FIND_BY_FOLDER, limit, offset)
      } else {
        STMT_FIND_BY_FOLDER
      }

    sqlQuery(
      sqlStatement,
      params(MANAGED_APPLICATION.FOLDER_UID -> folderPersistence.getUid(folderId), "condition" -> s"%$condition%"),
      managedApplicationRecordRowMapper
    ).toVector
  }

  override def findWithFolderConnections: Vector[ManagedApplication] =
    sqlQuery(
      STMT_FIND_WITH_FOLDER_CONNECTIONS,
      params(),
      managedApplicationRecordRowMapper
    ).toVector

  override def countForFolder(folderId: String): Int = {
    sqlQuery(
      STMT_COUNT_BY_FOLDER,
      params(MANAGED_APPLICATION.FOLDER_UID -> folderPersistence.getUid(folderId)),
      _.getInt(1)
    ).head
  }

  override def countForFolderWithCondition(folderId: String, condition: String): Int = {
    sqlQuery(
      STMT_COUNT_BY_FOLDER_WITH_CONDITION,
      params(MANAGED_APPLICATION.FOLDER_UID -> folderPersistence.getUid(folderId), "condition" -> s"%$condition%"),
      _.getInt(1)
    ).head
  }

  override def create(managedApplication: ManagedApplication): String = {
    val applicationId = UUID.randomUUID().toString
    val serverConfigurationUid = configurationPersistence.getUid(managedApplication.connectionServer).get
    val folderUid = if(managedApplication.folderId != null) {
      folderPersistence.getUid(managedApplication.folderId)
    } else {
      findConfigurationFolderId(managedApplication.connectionServer).get
    }

    sqlExec(
      STMT_INSERT,
      params(
        MANAGED_APPLICATION.ID -> applicationId,
        MANAGED_APPLICATION.APPLICATION_NAME -> managedApplication.applicationName,
        MANAGED_APPLICATION.MANAGED_BY -> managedApplication.managedBy,
        MANAGED_APPLICATION.DATE_CREATED -> managedApplication.dateCreated,
        MANAGED_APPLICATION.ENVIRONMENT_TAG -> managedApplication.environmentTag,
        MANAGED_APPLICATION.SERVER_CONFIGURATION_UID -> serverConfigurationUid,
        MANAGED_APPLICATION.APPLICATION_REFERENCE -> managedApplication.applicationReference,
        MANAGED_APPLICATION.WORKFLOW_ID -> managedApplication.workflowId,
        MANAGED_APPLICATION.DELETE_WORKFLOW_ID -> managedApplication.deleteWorkflowId,
        MANAGED_APPLICATION.UPDATE_WORKFLOW_ID -> managedApplication.updateWorkflowId,
        MANAGED_APPLICATION.FOLDER_UID -> folderUid,
      ),
      _.execute()
    )

    applicationId
  }

  override def update(managedApplication: ManagedApplication): Unit = {
    sqlExec(
      STMT_UPDATE,
      params(
        MANAGED_APPLICATION.ID -> managedApplication.id,
        MANAGED_APPLICATION.APPLICATION_NAME -> managedApplication.applicationName,
        MANAGED_APPLICATION.MANAGED_BY -> managedApplication.managedBy,
        MANAGED_APPLICATION.DATE_CREATED -> managedApplication.dateCreated,
        MANAGED_APPLICATION.ENVIRONMENT_TAG -> managedApplication.environmentTag,
        MANAGED_APPLICATION.SERVER_CONFIGURATION_UID -> configurationPersistence.getUid(managedApplication.connectionServer).get,
        MANAGED_APPLICATION.APPLICATION_REFERENCE -> managedApplication.applicationReference,
        MANAGED_APPLICATION.WORKFLOW_ID -> managedApplication.workflowId,
        MANAGED_APPLICATION.DELETE_WORKFLOW_ID -> managedApplication.deleteWorkflowId,
        MANAGED_APPLICATION.UPDATE_WORKFLOW_ID -> managedApplication.updateWorkflowId,
        MANAGED_APPLICATION.FOLDER_UID -> folderPersistence.getUid(managedApplication.folderId),
      ),
      _.execute()
    )
  }

  override def delete(applicationId: String): Unit =
    sqlExec(
      STMT_DELETE_BY_ID,
      params(MANAGED_APPLICATION.ID -> applicationId),
      _.execute()
    )

  override def findConnectionFolderId(applicationId: String): Option[String] =
    sqlQuery(
      STMT_FIND_CONNECTION_FOLDER_ID,
      params(MANAGED_APPLICATION.ID -> applicationId),
      rs => rs.getString(FOLDERS.FOLDER_ID)
    ).headOption

  override def existsByApplicationReferenceAndFolderId(applicationReference: String, folderId: String): Boolean = {
    sqlQuery(
      STMT_EXISTS_BY_APP_REF_AND_FOLDER_UID,
      params(
        MANAGED_APPLICATION.APPLICATION_REFERENCE -> applicationReference,
        MANAGED_APPLICATION.FOLDER_UID -> folderPersistence.getUid(folderId),
      ),
      _.getInt(1) > 0
    ).head
  }

  private def findConfigurationFolderId(configurationId: String): Option[String] =
    sqlQuery(
      STMT_FIND_CONFIGURATION_FOLDER_UID,
      params(CONFIGURATIONS.ID -> configurationId),
      rs => rs.getString(CONFIGURATIONS.FOLDER_CI_UID)
    ).headOption

  private final val STMT_BASE_FIND =
    s"""SELECT
       |  m.${MANAGED_APPLICATION.ID} as ${MANAGED_APPLICATION.ID},
       |  m.${MANAGED_APPLICATION.APPLICATION_NAME} as ${MANAGED_APPLICATION.APPLICATION_NAME},
       |  m.${MANAGED_APPLICATION.MANAGED_BY} as ${MANAGED_APPLICATION.MANAGED_BY},
       |  m.${MANAGED_APPLICATION.DATE_CREATED} as ${MANAGED_APPLICATION.DATE_CREATED},
       |  m.${MANAGED_APPLICATION.ENVIRONMENT_TAG} as ${MANAGED_APPLICATION.ENVIRONMENT_TAG},
       |  c.${CONFIGURATIONS.ID} as $CONNECTION_ID_ALIAS,
       |  m.${MANAGED_APPLICATION.APPLICATION_REFERENCE} as ${MANAGED_APPLICATION.APPLICATION_REFERENCE},
       |  m.${MANAGED_APPLICATION.WORKFLOW_ID} as ${MANAGED_APPLICATION.WORKFLOW_ID},
       |  m.${MANAGED_APPLICATION.DELETE_WORKFLOW_ID} as ${MANAGED_APPLICATION.DELETE_WORKFLOW_ID},
       |  m.${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID} as ${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID},
       |  f.${FOLDERS.FOLDER_ID} as ${FOLDERS.FOLDER_ID}
       | FROM ${CONFIGURATIONS.TABLE} c
       | JOIN ${MANAGED_APPLICATION.TABLE} m ON c.${CONFIGURATIONS.CI_UID} = m.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
       | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = m.${MANAGED_APPLICATION.FOLDER_UID}
     """.stripMargin

  private val STMT_INSERT =
    s"""|INSERT INTO ${MANAGED_APPLICATION.TABLE}
        |   ( ${MANAGED_APPLICATION.ID}
        |   , ${MANAGED_APPLICATION.APPLICATION_NAME}
        |   , ${MANAGED_APPLICATION.MANAGED_BY}
        |   , ${MANAGED_APPLICATION.DATE_CREATED}
        |   , ${MANAGED_APPLICATION.ENVIRONMENT_TAG}
        |   , ${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
        |   , ${MANAGED_APPLICATION.APPLICATION_REFERENCE}
        |   , ${MANAGED_APPLICATION.WORKFLOW_ID}
        |   , ${MANAGED_APPLICATION.DELETE_WORKFLOW_ID}
        |   , ${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID}
        |   , ${MANAGED_APPLICATION.FOLDER_UID}
        |   )
        | VALUES
        |   ( :${MANAGED_APPLICATION.ID}
        |   , :${MANAGED_APPLICATION.APPLICATION_NAME}
        |   , :${MANAGED_APPLICATION.MANAGED_BY}
        |   , :${MANAGED_APPLICATION.DATE_CREATED}
        |   , :${MANAGED_APPLICATION.ENVIRONMENT_TAG}
        |   , :${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
        |   , :${MANAGED_APPLICATION.APPLICATION_REFERENCE}
        |   , :${MANAGED_APPLICATION.WORKFLOW_ID}
        |   , :${MANAGED_APPLICATION.DELETE_WORKFLOW_ID}
        |   , :${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID}
        |   , :${MANAGED_APPLICATION.FOLDER_UID}
        |   )
      """.stripMargin

  private val STMT_UPDATE =
    s"""|UPDATE ${MANAGED_APPLICATION.TABLE}
        | SET
        |    ${MANAGED_APPLICATION.APPLICATION_NAME} = :${MANAGED_APPLICATION.APPLICATION_NAME}
        |  , ${MANAGED_APPLICATION.MANAGED_BY} = :${MANAGED_APPLICATION.MANAGED_BY}
        |  , ${MANAGED_APPLICATION.DATE_CREATED} = :${MANAGED_APPLICATION.DATE_CREATED}
        |  , ${MANAGED_APPLICATION.ENVIRONMENT_TAG} = :${MANAGED_APPLICATION.ENVIRONMENT_TAG}
        |  , ${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID} = :${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
        |  , ${MANAGED_APPLICATION.APPLICATION_REFERENCE} = :${MANAGED_APPLICATION.APPLICATION_REFERENCE}
        |  , ${MANAGED_APPLICATION.WORKFLOW_ID} = :${MANAGED_APPLICATION.WORKFLOW_ID}
        |  , ${MANAGED_APPLICATION.DELETE_WORKFLOW_ID} = :${MANAGED_APPLICATION.DELETE_WORKFLOW_ID}
        |  , ${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID} = :${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID}
        |  , ${MANAGED_APPLICATION.FOLDER_UID} = :${MANAGED_APPLICATION.FOLDER_UID}
        | WHERE
        |  ${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
        """.stripMargin

  private val STMT_DELETE_BY_ID =
    s"""|DELETE FROM ${MANAGED_APPLICATION.TABLE}
        | WHERE ${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
       """.stripMargin

  private final val STMT_FIND_BY_FOLDER =
    s"""$STMT_BASE_FIND
       | WHERE m.${MANAGED_APPLICATION.FOLDER_UID} = :${MANAGED_APPLICATION.FOLDER_UID}
       | AND (
       |    LOWER(m.${MANAGED_APPLICATION.APPLICATION_NAME}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  private final val STMT_COUNT_BY_FOLDER =
    s"""SELECT COUNT(1)
       | FROM ${MANAGED_APPLICATION.TABLE} m
       | WHERE m.${MANAGED_APPLICATION.FOLDER_UID} = :${MANAGED_APPLICATION.FOLDER_UID}
       |""".stripMargin

  private final val STMT_COUNT_BY_FOLDER_WITH_CONDITION =
    s"""$STMT_COUNT_BY_FOLDER
       |  AND (
       |    LOWER(m.${MANAGED_APPLICATION.APPLICATION_NAME}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  private final val STMT_FIND_BY_ID =
    s"""$STMT_BASE_FIND
       | WHERE m.${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
       |""".stripMargin

  private final val STMT_FIND_CONFIGURATION_FOLDER_UID =
    s"""SELECT
       |  c.${CONFIGURATIONS.FOLDER_CI_UID} as ${CONFIGURATIONS.FOLDER_CI_UID}
       | FROM ${CONFIGURATIONS.TABLE} c
       | WHERE c.${CONFIGURATIONS.ID} = :${CONFIGURATIONS.ID}
       | """.stripMargin

  private val managedApplicationRecordRowMapper: RowMapper[ManagedApplication] = (rs: ResultSet, _: Int) =>
    ManagedApplication(
      rs.getString(MANAGED_APPLICATION.ID),
      rs.getString(MANAGED_APPLICATION.APPLICATION_NAME),
      rs.getString(MANAGED_APPLICATION.MANAGED_BY),
      Option(rs.getDate(MANAGED_APPLICATION.DATE_CREATED)).map(dc => new Date(dc.getTime)).orNull,
      rs.getString(MANAGED_APPLICATION.ENVIRONMENT_TAG),
      rs.getString(MANAGED_APPLICATION.APPLICATION_REFERENCE),
      rs.getString(CONNECTION_ID_ALIAS),
      rs.getString(MANAGED_APPLICATION.WORKFLOW_ID),
      rs.getString(MANAGED_APPLICATION.DELETE_WORKFLOW_ID),
      rs.getString(MANAGED_APPLICATION.UPDATE_WORKFLOW_ID),
      rs.getString(FOLDERS.FOLDER_ID)
    )

  private final val STMT_EXISTS_BY_APP_REF_AND_FOLDER_UID: String =
    s"""SELECT
       |  COUNT(1)
       | FROM ${MANAGED_APPLICATION.TABLE} m
       | WHERE m.${MANAGED_APPLICATION.APPLICATION_REFERENCE} = :${MANAGED_APPLICATION.APPLICATION_REFERENCE}
       | AND m.${MANAGED_APPLICATION.FOLDER_UID} = :${MANAGED_APPLICATION.FOLDER_UID}
       |""".stripMargin

  private final val STMT_FIND_WITH_FOLDER_CONNECTIONS =
    s"""$STMT_BASE_FIND
       | WHERE c.${CONFIGURATIONS.FOLDER_CI_UID} IS NOT NULL
       | AND m.${MANAGED_APPLICATION.FOLDER_UID} IS NULL
       |""".stripMargin

  private final val STMT_FIND_CONNECTION_FOLDER_ID =
    s"""SELECT
       |  f.${FOLDERS.FOLDER_ID} as ${FOLDERS.FOLDER_ID}
       | FROM ${CONFIGURATIONS.TABLE} c
       | JOIN ${MANAGED_APPLICATION.TABLE} m ON c.${CONFIGURATIONS.CI_UID} = m.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
       | JOIN ${FOLDERS.TABLE} f ON c.${CONFIGURATIONS.FOLDER_CI_UID} = f.${FOLDERS.CI_UID}
       | WHERE m.${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
       | """.stripMargin

  private final val CONNECTION_ID_ALIAS = "CON_ID"
}

