import { all, call, CallEffect, put, race, select, take, takeEvery, takeLatest } from 'redux-saga/effects';
import { IHttpResponse } from 'angular';
import { PayloadAction } from '@reduxjs/toolkit';
import IdsFactory from '../../../../../../../../../core/xlr-ui/app/js/util/ids';
import { httpDELETE, httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { applicationManagement, ApplicationManagementState, NavigateToWorkflowPayload } from './managed-application.reducer';
import { ManagedApplicationsFilters, WebhookSourceWithFolder } from '../managed-application.types';
import { AxiosError } from 'axios';
import { ClientSettings, CustomConfiguration, FilterQueryParams, Folder, Variable } from '../../../../../../../../../core/xlr-ui/app/types';
import { RunWorkflowAction, workflow } from '../../../../../../../../../core/xlr-ui/app/features/workflow/ducks/workflow.reducer';
import { getReleaseVariables, updateReleaseVariable } from '../../../../../../../../../core/xlr-ui/app/features/tasks/ducks/variables-service.saga';
import { getApplicationManagementState } from './managed-application.selectors';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import ToastrFactory from '../../../../../../../../../core/xlr-ui/app/js/util/toastrFactory';
import { IStateParamsService } from 'angular-ui-router';
import { DELETE_BUTTON_ACTION } from '../components/created-applications-table/managed-application.component';
import { hasAnyOfPermissions } from '../../../../../../../../../core/xlr-ui/app/js/auth/authenticator.saga';
import { Permissions } from '../../../../../../../../../core/xlr-ui/app/js/security/permissions-constants';

const {
    createManagedApplicationsFromWebhookSource,
    getFilters,
    loadManagedApplications,
    navigateToWorkflow,
    setCondition,
    setIsLoading,
    setManagedApplications,
    setManagedApplicationsCount,
    setManagedApplicationsTotalCount,
    setPage,
    storeFilters,
    loadEditPermission,
    setEditPermission,
} = applicationManagement.actions;

const { createWorkflowError, createWorkflowSuccess, runWorkflow } = workflow.actions;
const Ids = IdsFactory();

export const toaster = ToastrFactory();

export function* withLoadingState<R>(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        const result: R = yield effect;
        return result;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* storeFiltersAction() {
    const { page, condition } = yield select(getApplicationManagementState);
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    clientSettings.setManagedApplicationsFilters({ ...page, condition });
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    FiltersQueryParams.update({ ...page, condition });
}

export function* executeFetchManagedApplicationsAction() {
    const { page, condition }: ApplicationManagementState = yield select(getApplicationManagementState);
    try {
        const {
            data: { totalCount, count, managedApplications },
        } = yield call(
            withLoadingState,
            call(
                httpGET,
                `api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${page.resultsPerPage}&condition=${
                    condition ? encodeURIComponent(condition) : ''
                }`,
                true,
            ),
        );
        yield put(setManagedApplications(managedApplications));
        yield put(setManagedApplicationsCount(count));
        yield put(setManagedApplicationsTotalCount(totalCount));
    } catch (e: unknown) {
        const err = e as AxiosError<CustomConfiguration, unknown>;
        const errServerData = err.response?.data;
        const errorMessage = `Error fetching managed applications data. Check connection to ${errServerData?.title} HTTP Connection or check application logs for more details.`;
        yield call(toaster.error, errorMessage);
    }
}

export function* getFiltersAction(action: PayloadAction<string>) {
    const folderId: string = action.payload;
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    const filters: ManagedApplicationsFilters = clientSettings.getManagedApplicationsFilters();

    const stateParams: IStateParamsService = yield call(getAngularService, '$stateParams');
    const condition = stateParams.condition;
    if (condition) filters.condition = condition;

    if (filters) {
        yield put(setCondition(filters.condition));
        yield put(
            setPage({
                folderId,
                page: filters.page,
                order: filters.order,
                orderBy: filters.orderBy,
                resultsPerPage: filters.resultsPerPage,
            }),
        );
    }
}

export function* executeCreateManagedApplicationsFromWebhookSourceAction(action: PayloadAction<WebhookSourceWithFolder>) {
    const { webhookSourceId, folderId } = action.payload;
    yield call(withLoadingState, call(httpPOST, `/api/v1/managed-application/create-from-webhook-source/${webhookSourceId}?folderId=${folderId}`));
    yield call(toaster.success, `Successfully created managed applications from discovered applications`);
}

export const managedApplicationIdVariableKey = 'managedApplicationId';

export function* navigateToWorkflowSaga(action: PayloadAction<NavigateToWorkflowPayload>) {
    const { actionName, managedApplicationId, workflowId } = action.payload;
    if (workflowId) {
        const { page }: ApplicationManagementState = yield select(getApplicationManagementState);
        const variables: Array<Variable> = yield call(getReleaseVariables, workflowId, true);
        let appIdVariable = variables.find((v) => v.key === managedApplicationIdVariableKey);
        if (appIdVariable) {
            yield call(updateReleaseVariable, { ...appIdVariable, value: managedApplicationId });
        } else {
            const variable: Variable = {
                id: null,
                key: managedApplicationIdVariableKey,
                requiresValue: false,
                showOnReleaseStart: false,
                type: 'xlrelease.StringVariable',
                value: managedApplicationId,
            };
            const { data }: IHttpResponse<Variable> = yield call(httpPOST, `api/v1/releases/${Ids.toDomainId(workflowId)}/variables`, variable);
            appIdVariable = data;
        }
        yield put(runWorkflow({ folderId: page.folderId, workflow: { id: workflowId } } as never as RunWorkflowAction));
        // wait until the workflow creation is finished and then clean the template variable
        yield race({
            error: take(createWorkflowError.type),
            success: take(createWorkflowSuccess.type),
        });
        yield call(updateReleaseVariable, { ...appIdVariable, value: '' });
    } else {
        if (actionName === DELETE_BUTTON_ACTION) {
            yield call(httpDELETE, `api/v1/managed-application/${managedApplicationId}`);
            yield call(
                toaster.info,
                `Selected managed application has no connected ${actionName} workflow,
            this action triggered deletion of Managed Application only in the Digital.a Release system`,
            );
            yield put(loadManagedApplications());
        } else {
            yield call(toaster.error, `Selected managed application has no connected ${actionName} workflow`);
        }
    }
}

export function* loadEditPermissionAction(action: PayloadAction<Folder>) {
    const folder: Folder = action.payload;
    const hasEditPermission: boolean = yield call(hasAnyOfPermissions, [Permissions.EDIT_APP_PIPELINES], folder.$metadata);
    yield put(setEditPermission(hasEditPermission));
}

export function* managedApplicationSaga() {
    yield all([
        takeEvery(createManagedApplicationsFromWebhookSource, executeCreateManagedApplicationsFromWebhookSourceAction),
        takeEvery(getFilters, getFiltersAction),
        takeLatest(loadManagedApplications, executeFetchManagedApplicationsAction),
        takeLatest(navigateToWorkflow, navigateToWorkflowSaga),
        takeEvery(storeFilters, storeFiltersAction),
        takeEvery(loadEditPermission, loadEditPermissionAction),
    ]);
}
