import React from 'react';
import moment from 'moment';
import { ReactWrapper } from 'enzyme';
import upperFirst from 'lodash/upperFirst';
import camelCase from 'lodash/camelCase';
import deburr from 'lodash/deburr';
import { mountWithTheme } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ExternalDeployment } from './external-deployment.component';
import {
    connectionArgoServerMock,
    connectionDeployServerMock,
    externalDeploymentMock,
    externalDeploymentMockDeploy,
} from '../__mocks__/external-deployments.mocks';
import { generateEnvironmentTitle, getDeepLink } from '../helper/utils';
import { IMG_SRC } from '../constants';
import { pluginInfoArgo, pluginInfoDeploy } from '../__mocks__/discovery-wizard.mocks';

describe('ExternalDeployment component', () => {
    let wrapper: ReactWrapper;
    const openDeepLink = () =>
        wrapper.find(`button[data-testid="deeplink-${externalDeploymentMock.applicationUid}-${externalDeploymentMock.state.destination}"]`);

    beforeEach(() => {
        wrapper = mountWithTheme(
            <ExternalDeployment
                connectionServer={connectionDeployServerMock}
                externalDeployment={externalDeploymentMockDeploy}
                pluginInfo={pluginInfoDeploy}
            />,
        );
    });
    const searchDotCard = (selector: string) => wrapper.find(`.external-deployment-card ${selector}`);

    it('should show the server icon', () => {
        expect(searchDotCard('.icon-column .server-icon').props().src).toStrictEqual('icon');
    });

    it('should show the application name', () => {
        expect(searchDotCard('.application-name-column').text()).toStrictEqual(externalDeploymentMock.applicationName);
        expect(searchDotCard('.application-name-column').props().title).toStrictEqual(externalDeploymentMock.applicationName);
    });

    it('should show the environment icon and name deploy', () => {
        expect(searchDotCard('.environment-column .server-icon').props().src).toStrictEqual(IMG_SRC.environment);
        expect(searchDotCard('.environment-column > span').text()).toStrictEqual(externalDeploymentMock.state.destination);
        expect(searchDotCard('.environment-column > span').props().title).toStrictEqual(externalDeploymentMock.state.destination);
    });

    it('should show the version', () => {
        expect(searchDotCard('.version-column').text()).toStrictEqual(externalDeploymentMock.state.versionTag.label);
        expect(searchDotCard('.version-column .dot-link').at(0).prop('href')).toStrictEqual(
            getDeepLink(connectionDeployServerMock.url, externalDeploymentMock.applicationPath),
        );
    });

    it('should show the status inside the pill component', () => {
        expect(searchDotCard('.status-column .MuiChip-label').at(0).text()).toStrictEqual(
            upperFirst(camelCase(deburr(externalDeploymentMock.state.deploymentStatus))),
        );
        expect(searchDotCard('.status-column .MuiChip-label').at(0).hasClass('MuiChip-labelSmall')).toBeTruthy();
        expect(searchDotCard('.status-column .MuiChip-root').at(0).hasClass('MuiChip-sizeSmall')).toBeTruthy();
        expect(searchDotCard('.status-column .MuiChip-root').at(0).hasClass('success')).toBeTruthy();
    });

    it('should show the time passed from last update', () => {
        expect(searchDotCard('.last-updated-column').at(0).text()).toStrictEqual(moment(externalDeploymentMock.state.lastChangeTime).fromNow());
    });

    it('should show the deep link icon with popover', () => {
        expect(searchDotCard('.link-column .dot-i').hasClass('icon-open-new-tab')).toBeTruthy();
        openDeepLink().simulate('mouseenter');
        expect(searchDotCard('.link-column .dot-tooltip').at(0).props().title).toStrictEqual('Open application');
    });

    it('should handle on click of deep link', () => {
        window.open = jest.fn().mockReturnValue({});
        openDeepLink().simulate('click');
        expect(window.open).toHaveBeenCalledWith(getDeepLink(connectionDeployServerMock.url, externalDeploymentMock.applicationPath), '_blank');
    });

    it('should show the environment icon and name argo', () => {
        wrapper = mountWithTheme(
            <ExternalDeployment connectionServer={connectionArgoServerMock} externalDeployment={externalDeploymentMock} pluginInfo={pluginInfoArgo} />,
        );
        expect(searchDotCard('.environment-column .server-icon').props().src).toStrictEqual(IMG_SRC.environment);
        expect(searchDotCard('.environment-column > span').text()).toStrictEqual(generateEnvironmentTitle(externalDeploymentMock.state));
        expect(searchDotCard('.environment-column > span').props().title).toStrictEqual(generateEnvironmentTitle(externalDeploymentMock.state));
    });
});
