import React, { ChangeEvent, memo, useEffect, useState } from 'react';
import { Folder } from '../../../../../../../../../../core/xlr-ui/app/types';
import { ManagedApplication } from './managed-application.component';
import { ManagedApplicationTableComponentPortal } from './managed-application-table-component.portal';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { applicationManagement } from '../../ducks/managed-application.reducer';
import { ManagedApplicationType, ManagedApplicationsPage } from '../../managed-application.types';
import {
    getCreatedApplicationsCountSelector,
    getCreatedApplicationsLoadingSelector,
    getCreatedApplicationsPageSelector,
    getCreatedApplicationsSelector,
    getConditionSelector,
} from '../../ducks/managed-application.selectors';
import { DotColumnHeader, DotTable, DotTablePagination, DotTypography, Order, RowsPerPageOption } from '@digital-ai/dot-components';
import {
    COLUMN_ACTIONS,
    COLUMN_APPLICATION_NAME,
    COLUMN_DATE_CREATED,
    COLUMN_DEEP_LINK,
    COLUMN_ENV_TAG,
    COLUMN_MANAGED_BY,
    COLUMN_TARGET_CONNECTION,
    COLUMN_TARGET_DEPLOYMENT,
    COLUMN_WIDTH,
} from '../../constants';
import { Search } from '../../../../../../../../../../core/xlr-ui/app/react/components/search/search';

export interface ManagedApplicationTableProps {
    folder: Folder;
    setCreateApplicationView: (_: boolean) => void;
}

const { loadManagedApplications, setPage, storeFilters, setCondition } = applicationManagement.actions;

const ManagedApplicationTableComponent = ({ folder, setCreateApplicationView }: ManagedApplicationTableProps) => {
    const createdApplications: ManagedApplicationType[] = useAppSelector(getCreatedApplicationsSelector);
    const count: number = useAppSelector(getCreatedApplicationsCountSelector);
    const isLoading: boolean = useAppSelector(getCreatedApplicationsLoadingSelector);
    const page = useAppSelector(getCreatedApplicationsPageSelector);
    const condition = useAppSelector(getConditionSelector);
    const [filterTimeout, setFilterTimeout] = useState<ReturnType<typeof setTimeout> | undefined>(undefined);
    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(storeFilters());
    }, [page, condition]);

    useEffect(() => {
        dispatch(loadManagedApplications());
    }, []);

    const onPageChange = (newPage: ManagedApplicationsPage) => {
        dispatch(setPage(newPage));
        dispatch(loadManagedApplications());
    };

    const onDisplayFilterChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const value = event.target.value;
        if (filterTimeout) {
            clearTimeout(filterTimeout);
            setFilterTimeout(undefined);
        }
        setFilterTimeout(
            setTimeout(() => {
                dispatch(setCondition(value));
            }, 500),
        );
    };

    return (
        <>
            <ManagedApplicationTableComponentPortal folder={folder} onViewChange={() => setCreateApplicationView(true)} />
            <div className="external-deployments-table">
                <div className="external-deployments-table-title">
                    <DotTypography variant="h1">Applications ({count})</DotTypography>
                    <Search
                        defaultValue={condition}
                        inputPlaceholder="Type to filter"
                        key={condition}
                        onFilterChange={onDisplayFilterChange}
                        searchClassNames={['mtm', 'mbm']}
                    />
                </div>

                <DotTable
                    className="external-deployments-table-header"
                    columns={headerDefinitions}
                    data={[]}
                    loading={isLoading}
                    onUpdateData={(order: Order, orderBy: string) =>
                        onPageChange({
                            ...page,
                            folderId: folder.id,
                            order,
                            orderBy,
                        })
                    }
                    order={page.order}
                    orderBy={page.orderBy}
                />

                {count > 0
                    ? createdApplications.map((createdApplication: ManagedApplicationType) => {
                          return <ManagedApplication folder={folder} key={createdApplication.applicationName} managedApplication={createdApplication} />;
                      })
                    : condition &&
                      condition.length > 0 && (
                          <div className="empty-external-deployments-table">
                              No applications match the current filter criteria. Consider adjusting the filter values.
                          </div>
                      )}

                <DotTablePagination
                    count={count}
                    onPageChange={(newPage: number) =>
                        onPageChange({
                            ...page,
                            folderId: folder.id,
                            page: newPage,
                        })
                    }
                    onRowsPerPageChange={(event: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) =>
                        onPageChange({
                            ...page,
                            folderId: folder.id,
                            page: 0,
                            resultsPerPage: parseInt(event.target.value) as RowsPerPageOption,
                        })
                    }
                    page={page.page}
                    rowsPerPage={page.resultsPerPage}
                    typography="body1"
                />
            </div>
        </>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: undefined,
        id: COLUMN_APPLICATION_NAME,
        label: 'Application name',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_MANAGED_BY,
        label: 'Managed by',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_DATE_CREATED,
        label: 'Date created',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_TARGET_DEPLOYMENT,
        label: 'Target deployment',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_ENV_TAG,
        label: 'Environment tag',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_TARGET_CONNECTION,
        label: 'Target Connection',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_DEEP_LINK,
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_ACTIONS,
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
];

export const ManagedApplicationTable = memo(ManagedApplicationTableComponent);
