package com.xebialabs.xlrelease.applications.management.repository.persistence

import com.xebialabs.xlrelease.applications.management.repository.persistence.ManagedApplicationSchema.MANAGED_APPLICATION
import com.xebialabs.xlrelease.applications.management.service.ManagedApplication
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.CONFIGURATIONS
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.{FolderPersistence, PersistenceSupport, Utils}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.{Date, UUID}

trait ApplicationsManagementPersistence extends PersistenceSupport with Utils with LimitOffset with Logging {


  override implicit val dialect: Dialect

  def configurationPersistence: ConfigurationPersistence

  def folderPersistence: FolderPersistence

  def findById(applicationId: String): Option[ManagedApplication]

  def findForFolder(folderId: String, limit: Long, offset: Long, condition: String): Vector[ManagedApplication]

  def countForFolder(folderId: String): Int

  def countForFolderWithCondition(folderId: String, condition: String): Int

  def create(managedApplication: ManagedApplication): String

  def update(managedApplication: ManagedApplication): Unit

  def delete(applicationId: String): Unit

  def existsByApplicationReference(applicationReference: String): Boolean

  def findConnectionId(applicationId: String): Option[String]
}

@Repository
@IsTransactional
class ApplicationsManagementPersistenceImpl @Autowired()(override val configurationPersistence: ConfigurationPersistence,
                                                         override val folderPersistence: FolderPersistence)
                                                        (@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                                         @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends SqlRepository with ApplicationsManagementPersistence with Logging {

  override def findById(applicationId: String): Option[ManagedApplication] = {
    sqlQuery(
      STMT_FIND_BY_ID,
      params(MANAGED_APPLICATION.ID -> applicationId),
      managedApplicationRecordRowMapper
    ).headOption
  }

  override def findForFolder(folderId: String, limit: Long, offset: Long, condition: String = ""): Vector[ManagedApplication] = {
    val sqlStatement =
      if (limit > 0) {
        addLimitAndOffset(STMT_FIND_BY_FOLDER, limit, offset)
      } else {
        STMT_FIND_BY_FOLDER
      }

    sqlQuery(
      sqlStatement,
      params(CONFIGURATIONS.FOLDER_CI_UID -> folderPersistence.getUid(folderId), "condition" -> s"%$condition%"),
      managedApplicationRecordRowMapper
    ).toVector
  }

  override def countForFolder(folderId: String): Int = {
    sqlQuery(
      STMT_COUNT_BY_FOLDER,
      params(CONFIGURATIONS.FOLDER_CI_UID -> folderPersistence.getUid(folderId)),
      _.getInt(1)
    ).head
  }

  override def countForFolderWithCondition(folderId: String, condition: String): Int = {
    sqlQuery(
      STMT_COUNT_BY_FOLDER_WITH_CONDITION,
      params(CONFIGURATIONS.FOLDER_CI_UID -> folderPersistence.getUid(folderId), "condition" -> s"%$condition%"),
      _.getInt(1)
    ).head
  }

  override def create(managedApplication: ManagedApplication): String = {
    val applicationId = UUID.randomUUID().toString
    sqlExec(
      STMT_INSERT,
      params(
        MANAGED_APPLICATION.ID -> applicationId,
        MANAGED_APPLICATION.APPLICATION_NAME -> managedApplication.applicationName,
        MANAGED_APPLICATION.MANAGED_BY -> managedApplication.managedBy,
        MANAGED_APPLICATION.DATE_CREATED -> managedApplication.dateCreated,
        MANAGED_APPLICATION.ENVIRONMENT_TAG -> managedApplication.environmentTag,
        MANAGED_APPLICATION.SERVER_CONFIGURATION_UID -> configurationPersistence.getUid(managedApplication.connectionServer).get,
        MANAGED_APPLICATION.APPLICATION_REFERENCE -> managedApplication.applicationReference,
        MANAGED_APPLICATION.WORKFLOW_ID -> managedApplication.workflowId,
        MANAGED_APPLICATION.DELETE_WORKFLOW_ID -> managedApplication.deleteWorkflowId,
        MANAGED_APPLICATION.UPDATE_WORKFLOW_ID -> managedApplication.updateWorkflowId,
      ),
      _.execute()
    )

    applicationId
  }

  override def update(managedApplication: ManagedApplication): Unit = {
    sqlExec(
      STMT_UPDATE,
      params(
        MANAGED_APPLICATION.ID -> managedApplication.id,
        MANAGED_APPLICATION.APPLICATION_NAME -> managedApplication.applicationName,
        MANAGED_APPLICATION.MANAGED_BY -> managedApplication.managedBy,
        MANAGED_APPLICATION.DATE_CREATED -> managedApplication.dateCreated,
        MANAGED_APPLICATION.ENVIRONMENT_TAG -> managedApplication.environmentTag,
        MANAGED_APPLICATION.SERVER_CONFIGURATION_UID -> configurationPersistence.getUid(managedApplication.connectionServer).get,
        MANAGED_APPLICATION.APPLICATION_REFERENCE -> managedApplication.applicationReference,
        MANAGED_APPLICATION.WORKFLOW_ID -> managedApplication.workflowId,
        MANAGED_APPLICATION.DELETE_WORKFLOW_ID -> managedApplication.deleteWorkflowId,
        MANAGED_APPLICATION.UPDATE_WORKFLOW_ID -> managedApplication.updateWorkflowId,
      ),
      _.execute()
    )
  }

  override def delete(applicationId: String): Unit =
    sqlExec(
      STMT_DELETE_BY_ID,
      params(MANAGED_APPLICATION.ID -> applicationId),
      _.execute()
    )


  override def findConnectionId(applicationId: String): Option[String] =
    sqlQuery(
      STMT_FIND_CONNECTION_ID,
      params(MANAGED_APPLICATION.ID -> applicationId),
      rs => rs.getString(CONFIGURATIONS.ID)
    ).headOption


  override def existsByApplicationReference(applicationReference: String): Boolean = {
    sqlQuery(
      STMT_EXISTS_BY_APP_REF,
      params(
        MANAGED_APPLICATION.APPLICATION_REFERENCE -> applicationReference,
      ),
      _.getInt(1) > 0
    ).head
  }

  private final val STMT_BASE_FIND =
    s"""SELECT
       |  m.${MANAGED_APPLICATION.ID} as ${MANAGED_APPLICATION.ID},
       |  m.${MANAGED_APPLICATION.APPLICATION_NAME} as ${MANAGED_APPLICATION.APPLICATION_NAME},
       |  m.${MANAGED_APPLICATION.MANAGED_BY} as ${MANAGED_APPLICATION.MANAGED_BY},
       |  m.${MANAGED_APPLICATION.DATE_CREATED} as ${MANAGED_APPLICATION.DATE_CREATED},
       |  m.${MANAGED_APPLICATION.ENVIRONMENT_TAG} as ${MANAGED_APPLICATION.ENVIRONMENT_TAG},
       |  m.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID} as ${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID},
       |  m.${MANAGED_APPLICATION.APPLICATION_REFERENCE} as ${MANAGED_APPLICATION.APPLICATION_REFERENCE},
       |  m.${MANAGED_APPLICATION.WORKFLOW_ID} as ${MANAGED_APPLICATION.WORKFLOW_ID},
       |  m.${MANAGED_APPLICATION.DELETE_WORKFLOW_ID} as ${MANAGED_APPLICATION.DELETE_WORKFLOW_ID},
       |  m.${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID} as ${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID}
       | FROM ${CONFIGURATIONS.TABLE} c
       | JOIN ${MANAGED_APPLICATION.TABLE} m ON c.${CONFIGURATIONS.CI_UID} = m.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}"""

  private val STMT_INSERT =
    s"""|INSERT INTO ${MANAGED_APPLICATION.TABLE}
        |   ( ${MANAGED_APPLICATION.ID}
        |   , ${MANAGED_APPLICATION.APPLICATION_NAME}
        |   , ${MANAGED_APPLICATION.MANAGED_BY}
        |   , ${MANAGED_APPLICATION.DATE_CREATED}
        |   , ${MANAGED_APPLICATION.ENVIRONMENT_TAG}
        |   , ${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
        |   , ${MANAGED_APPLICATION.APPLICATION_REFERENCE}
        |   , ${MANAGED_APPLICATION.WORKFLOW_ID}
        |   , ${MANAGED_APPLICATION.DELETE_WORKFLOW_ID}
        |   , ${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID}
        |   )
        | VALUES
        |   ( :${MANAGED_APPLICATION.ID}
        |   , :${MANAGED_APPLICATION.APPLICATION_NAME}
        |   , :${MANAGED_APPLICATION.MANAGED_BY}
        |   , :${MANAGED_APPLICATION.DATE_CREATED}
        |   , :${MANAGED_APPLICATION.ENVIRONMENT_TAG}
        |   , :${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
        |   , :${MANAGED_APPLICATION.APPLICATION_REFERENCE}
        |   , :${MANAGED_APPLICATION.WORKFLOW_ID}
        |   , :${MANAGED_APPLICATION.DELETE_WORKFLOW_ID}
        |   , :${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID}
        |   )
      """.stripMargin

  private val STMT_UPDATE =
    s"""|UPDATE ${MANAGED_APPLICATION.TABLE}
        | SET
        |    ${MANAGED_APPLICATION.APPLICATION_NAME} = :${MANAGED_APPLICATION.APPLICATION_NAME}
        |  , ${MANAGED_APPLICATION.MANAGED_BY} = :${MANAGED_APPLICATION.MANAGED_BY}
        |  , ${MANAGED_APPLICATION.DATE_CREATED} = :${MANAGED_APPLICATION.DATE_CREATED}
        |  , ${MANAGED_APPLICATION.ENVIRONMENT_TAG} = :${MANAGED_APPLICATION.ENVIRONMENT_TAG}
        |  , ${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID} = :${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
        |  , ${MANAGED_APPLICATION.APPLICATION_REFERENCE} = :${MANAGED_APPLICATION.APPLICATION_REFERENCE}
        |  , ${MANAGED_APPLICATION.WORKFLOW_ID} = :${MANAGED_APPLICATION.WORKFLOW_ID}
        |  , ${MANAGED_APPLICATION.DELETE_WORKFLOW_ID} = :${MANAGED_APPLICATION.DELETE_WORKFLOW_ID}
        |  , ${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID} = :${MANAGED_APPLICATION.UPDATE_WORKFLOW_ID}
        | WHERE
        |  ${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
        """.stripMargin

  private val STMT_DELETE_BY_ID =
    s"""|DELETE FROM ${MANAGED_APPLICATION.TABLE}
        | WHERE ${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
       """.stripMargin

  private final val STMT_FIND_BY_FOLDER =
    s"""$STMT_BASE_FIND
       | WHERE c.${CONFIGURATIONS.FOLDER_CI_UID} = :${CONFIGURATIONS.FOLDER_CI_UID}
       | AND (
       |    LOWER(m.${MANAGED_APPLICATION.APPLICATION_NAME}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  private final val STMT_COUNT_BY_FOLDER =
    s"""SELECT COUNT(1)
       | FROM ${CONFIGURATIONS.TABLE} c
       | JOIN ${MANAGED_APPLICATION.TABLE} m ON c.${CONFIGURATIONS.CI_UID} = m.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
       | WHERE c.${CONFIGURATIONS.FOLDER_CI_UID} = :${CONFIGURATIONS.FOLDER_CI_UID}
       |""".stripMargin

  private final val STMT_COUNT_BY_FOLDER_WITH_CONDITION =
    s"""$STMT_COUNT_BY_FOLDER
       |  AND (
       |    LOWER(m.${MANAGED_APPLICATION.APPLICATION_NAME}) LIKE LOWER(:condition)
       | )
       |""".stripMargin

  private final val STMT_FIND_BY_ID =
    s"""$STMT_BASE_FIND
       | WHERE m.${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
       |""".stripMargin

  private final val STMT_FIND_CONNECTION_ID =
    s"""SELECT
       |  c.${CONFIGURATIONS.ID} as ${CONFIGURATIONS.ID}
       | FROM ${CONFIGURATIONS.TABLE} c
       | JOIN ${MANAGED_APPLICATION.TABLE} m ON c.${CONFIGURATIONS.CI_UID} = m.${MANAGED_APPLICATION.SERVER_CONFIGURATION_UID}
       | WHERE m.${MANAGED_APPLICATION.ID} = :${MANAGED_APPLICATION.ID}
       | """.stripMargin

  private val managedApplicationRecordRowMapper: RowMapper[ManagedApplication] = (rs: ResultSet, _: Int) =>
    ManagedApplication(
      rs.getString(MANAGED_APPLICATION.ID),
      rs.getString(MANAGED_APPLICATION.APPLICATION_NAME),
      rs.getString(MANAGED_APPLICATION.MANAGED_BY),
      Option(rs.getDate(MANAGED_APPLICATION.DATE_CREATED)).map(dc => new Date(dc.getTime)).orNull,
      rs.getString(MANAGED_APPLICATION.ENVIRONMENT_TAG),
      rs.getString(MANAGED_APPLICATION.APPLICATION_REFERENCE),
      rs.getString(MANAGED_APPLICATION.SERVER_CONFIGURATION_UID),
      rs.getString(MANAGED_APPLICATION.WORKFLOW_ID),
      rs.getString(MANAGED_APPLICATION.DELETE_WORKFLOW_ID),
      rs.getString(MANAGED_APPLICATION.UPDATE_WORKFLOW_ID)
    )

  private final val STMT_EXISTS_BY_APP_REF: String =
    s"""SELECT
       |  COUNT(1)
       | FROM ${MANAGED_APPLICATION.TABLE} m
       | WHERE m.${MANAGED_APPLICATION.APPLICATION_REFERENCE} = :${MANAGED_APPLICATION.APPLICATION_REFERENCE}
       |""".stripMargin
}

