import { all, call, CallEffect, put, select, takeEvery, takeLatest } from 'redux-saga/effects';
import { PayloadAction } from '@reduxjs/toolkit';
import { httpGET } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { applicationManagement, ApplicationManagementState } from './managed-application.reducer';
import { ManagedApplicationsPage } from '../managed-application.types';
import { toaster } from '../../external-deployments/ducks/external-deployments.saga';
import { AxiosError } from 'axios';
import { ClientSettings, CustomConfiguration, FilterQueryParams } from '../../../../../../../../../core/xlr-ui/app/types';
import { getApplicationManagementState } from './managed-application.selectors';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

const { loadManagedApplications, setIsLoading, setManagedApplications, setManagedApplicationsCount, setPage, storeFilters, getFilters } =
    applicationManagement.actions;

export function* withLoadingState<R>(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        const result: R = yield effect;
        return result;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* storeFiltersAction() {
    const { page } = yield select(getApplicationManagementState);
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    clientSettings.setManagedApplicationsFilters({ ...page });
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    FiltersQueryParams.update({ ...page });
}

export function* executeFetchManagedApplicationsAction() {
    const { page }: ApplicationManagementState = yield select(getApplicationManagementState);
    try {
        const {
            data: { count, managedApplications },
        } = yield call(
            withLoadingState,
            call(httpGET, `api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${page.resultsPerPage}`, true),
        );
        yield put(setManagedApplications(managedApplications));
        yield put(setManagedApplicationsCount(count));
    } catch (e: unknown) {
        const err = e as AxiosError<CustomConfiguration, unknown>;
        const errServerData = err.response?.data;
        const errorMessage = `Error fetching managed applications data. Check connection to ${errServerData?.title} HTTP Connection or check application logs for more details.`;
        yield call(toaster.error, errorMessage);
    }
}

export function* getFiltersAction(action: PayloadAction<string>) {
    const folderId: string = action.payload;
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    const filters: ManagedApplicationsPage = clientSettings.getManagedApplicationsFilters();
    if (filters) {
        yield put(
            setPage({
                folderId,
                page: filters.page,
                order: filters.order,
                orderBy: filters.orderBy,
                resultsPerPage: filters.resultsPerPage,
            }),
        );
    }
}

export function* managedApplicationSaga() {
    yield all([
        takeLatest(loadManagedApplications, executeFetchManagedApplicationsAction),
        takeEvery(storeFilters, storeFiltersAction),
        takeEvery(getFilters, getFiltersAction),
    ]);
}
