import React from 'react';
import { ReactWrapper } from 'enzyme';
import { mountComponentWithStore } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ENDPOINT_TYPE, IMG_SRC } from '../../../constants';
import { initialState as wizardInitialState } from '../../../ducks/application-wizard.reducer';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialState } from '../../../ducks/external-deployments.reducer';
import { BasicDiscovery } from './application-discovery-basic.component';
import { argoServerMock, deployServerMock, mockArgoWebhookSource, mockConfigDetails } from '../../../__mocks__/external-deployments.mocks';
import { DotAutoComplete } from '@digital-ai/dot-components';
import { DiscoverModal } from '../discover-modal/discover-modal-component';
import { FilterApplications } from '../filter-applications/filter-applications-component';
import { CreateServerForm } from '../create-server/application-create-server-component';

const dispatch = jest.fn();
const { setConfigDetails, setLiveUpdate } = folderExternalDeployments.actions;
const { loadServers, loadWebhookSources } = folderExternalDeployments.actions;
describe('Basic discovery component', () => {
    let wrapper: ReactWrapper;
    const closeWizard = jest.fn();
    const defaultProps = {
        folder: { id: 'folderId', title: 'Folder-1' },
        closeWizard,
    };
    const defaultState = {
        ...initialState,
        servers: [deployServerMock],
    };

    const mountComponent = (state: FolderExternalDeploymentsState = defaultState) => {
        wrapper = mountComponentWithStore(<BasicDiscovery {...defaultProps} />, dispatch, {
            externalApplicationWizard: wizardInitialState,
            folderExternalDeployments: state,
        });
        expect(dispatch).toHaveBeenCalledWith(loadServers(defaultProps.folder.id));
        expect(dispatch).toHaveBeenCalledWith(loadWebhookSources(defaultProps.folder.id));
    };

    const searchDotCard = (selector: string) => wrapper.find(`.discovery ${selector}`);

    describe('basic setup', () => {
        beforeEach(() => {
            mountComponent();
        });

        afterEach(() => {
            wrapper.unmount();
        });

        it('should contain title', () => {
            expect(searchDotCard(`h2`).text()).toStrictEqual('Discover applications');
        });

        it('should display argo server card', () => {
            expect(searchDotCard(`.server-cards .dot-card`).at(0)).toExist();
            expect(searchDotCard('.server-cards .dot-card .dot-card-header img').at(0).props().src).toStrictEqual(IMG_SRC.argoCd);
            expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(0).text()).toStrictEqual('ArgoCD');
            expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(1).text()).toStrictEqual('by Argo');
        });

        it('should show options for discovering argo servers, click it and close it', () => {
            const createServerButton = searchDotCard(`.server-cards .dot-card .dot-card-content button[data-testid="create-server-btn"]`).at(0);
            expect(createServerButton).toExist();
            expect(createServerButton.text()).toStrictEqual('Create connection');
            createServerButton.simulate('click');
            wrapper.update();
            const createServerForm = wrapper.find(CreateServerForm);
            expect(createServerForm).toExist();
            expect(createServerForm.props().folder).toBe(defaultProps.folder);
            expect(createServerForm.props().serverType).toBe(ENDPOINT_TYPE.argoCd);
            createServerForm.invoke('closeForm')?.();
            wrapper.update();
            expect(wrapper.find(CreateServerForm)).not.toExist();
        });

        it('should display deploy server card', () => {
            expect(searchDotCard(`.server-cards .dot-card`).at(1)).toExist();
            expect(searchDotCard(`.server-cards .dot-card .dot-card-header img`).at(1).props().src).toStrictEqual(IMG_SRC.deploy);
            expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(2).text()).toStrictEqual('Deploy');
            expect(searchDotCard(`.server-cards .dot-card .dot-card-header span`).at(3).text()).toStrictEqual('by Digital.ai');
        });

        it('should show autocomplete for choosing deploy servers', () => {
            expect(wrapper.find(DotAutoComplete)).toExist();
            expect(wrapper.find(DotAutoComplete)).toHaveLength(1);
            expect(wrapper.find(DotAutoComplete).props().options).toStrictEqual([deployServerMock]);
        });

        it('should show button for creating deploy server', () => {
            expect(searchDotCard(`button[data-testid="discover-server-btn"]`)).toExist();
            expect(searchDotCard(`button[data-testid="discover-server-btn"]`).text()).toStrictEqual('Discover applications');
        });

        it('should disable discover applications button for deploy servers', () => {
            expect(searchDotCard(`button[data-testid="discover-server-btn"]`)).toBeDisabled();
        });

        it('should enable discover applications button for deploy server', () => {
            wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, deployServerMock, '');
            wrapper.update();
            expect(searchDotCard(`button[data-testid="discover-server-btn"]`)).not.toBeDisabled();
        });

        it('should enable discover applications button for deploy servers', () => {
            wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, deployServerMock, '');
            wrapper.update();
            expect(searchDotCard(`.server-cards .dot-card .dot-card-content button[data-testid="discover-server-btn"]`)).not.toBeDisabled();
        });

        it('should disable discover applications button when server removed', () => {
            wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, null, '');
            wrapper.update();
            expect(searchDotCard(`.server-cards .dot-card .dot-card-content button[data-testid="discover-server-btn"]`)).toBeDisabled();
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            searchDotCard(`button[data-testid="discover-server-btn"]`).invoke('onClick')?.();
            expect(dispatch).not.toBeCalledWith(setLiveUpdate({ folderId: 'folderId', serverId: '' }));
        });

        it('should handle discover applications button click', () => {
            wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, { ...deployServerMock }, '');
            searchDotCard(`button[data-testid="discover-server-btn"]`).simulate('click');
            expect(dispatch).toBeCalledWith(setLiveUpdate({ folderId: 'folderId', serverId: deployServerMock.id || '' }));
        });

        it('should show cancel button', () => {
            expect(searchDotCard(`button[data-testid="discovery-cancel-btn"]`)).toExist();
            expect(searchDotCard(`button[data-testid="discovery-cancel-btn"]`).text()).toEqual('Cancel');
        });

        it('should handle cancel button click', () => {
            searchDotCard(`button[data-testid="discovery-cancel-btn"]`).simulate('click');
            expect(closeWizard).toHaveBeenCalled();
        });
    });
    describe('with argo server', () => {
        beforeEach(() => {
            mountComponent({ ...defaultState, servers: [argoServerMock] });
        });

        it('should enable discover applications button for argo server', () => {
            wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, argoServerMock, '');
            wrapper.update();
            expect(searchDotCard(`button[data-testid="discover-server-btn"]`)).not.toBeDisabled();
        });

        it('should disable discover applications button when server removed', () => {
            wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, null, '');
            wrapper.update();
            expect(searchDotCard(`.server-cards .dot-card .dot-card-content button[data-testid="discover-server-btn"]`)).toBeDisabled();
        });

        it('should handle discover applications button click', () => {
            wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, argoServerMock, '');
            searchDotCard(`button[data-testid="discover-server-btn"]`).simulate('click');
            expect(dispatch).toBeCalledWith(setLiveUpdate({ folderId: 'folderId', serverId: argoServerMock.id || '' }));
        });
    });
    describe('with config details', () => {
        it('should close discovery modal', () => {
            mountComponent({ ...defaultState, configDetails: mockConfigDetails, servers: [argoServerMock] });
            const modal = wrapper.find(DiscoverModal);
            expect(modal.props().isModalOpen).toBe(true);
            modal.invoke('onModalClose')?.();
            wrapper.update();
            expect(dispatch).toHaveBeenCalledWith(setConfigDetails(undefined));
            expect(wrapper.find(DiscoverModal).props().isModalOpen).toBe(false);
        });

        it('should submit discovery modal and filter back', () => {
            mountComponent({
                ...defaultState,
                configDetails: mockConfigDetails,
                servers: [argoServerMock],
                webhookSources: [mockArgoWebhookSource],
            });
            const modal = wrapper.find(DiscoverModal);
            expect(modal.props().isModalOpen).toBe(true);
            modal.invoke('onModalSubmit')?.();
            wrapper.update();
            expect(dispatch).toHaveBeenCalledWith(setConfigDetails(undefined));
            const filterApps = wrapper.find(FilterApplications);
            expect(filterApps.props().webhookSource).toBe(mockArgoWebhookSource);
            expect(filterApps.props().server).toBe(undefined);
            filterApps.invoke('onFilterBack')?.();
            wrapper.update();
            expect(wrapper.find(FilterApplications)).not.toExist();
        });

        it('should handle unsuccessful config', () => {
            mountComponent({
                ...defaultState,
                configDetails: { ...mockConfigDetails, success: false },
                preSelectedServer: argoServerMock,
                servers: [argoServerMock],
                webhookSources: [mockArgoWebhookSource],
            });
            expect(wrapper.find(DiscoverModal)).not.toExist();
        });

        it('should handle config with null id', () => {
            mountComponent({ ...defaultState, configDetails: { ...mockConfigDetails, webhookSourceId: '' }, servers: [argoServerMock] });
            expect(wrapper.find(DiscoverModal).props().isModalOpen).toBe(true);
        });
    });
});
