import React, { useEffect, useState, memo } from 'react';
import { DotThemeProvider, DotTableActions, DotTable, DotTypography, DotChip, DotIcon, DotConfirmationDialog, DotIconButton } from '@digital-ai/dot-components';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { folderExternalDeployments } from '../../ducks/external-deployments.reducer';
import { Folder } from '../../../../../../../../../../core/xlr-ui/app/types';
import { getDisconnectedServers, getServersSelector, getWebhookSourcesSelector } from '../../ducks/external-deployments.selectors';
import { getServerIcon } from '../../helper/utils';
import './application-source.component.less';
import { Server, WebhookSource } from '../../external-deployment.types';
import { IMG_SRC } from '../../constants';
import { FilterApplications } from '../wizard/filter-applications/filter-applications-component';
import { navigation } from '../../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
const { navigateTo } = navigation.actions;
import noop from 'lodash/noop';

export interface ApplicationSourceComponentProp {
    folder: Folder;
}

const { loadServers, loadWebhookSources, loadExternalDeployments, getFilters, deleteWebhookSource, updateBreadcrumbs } = folderExternalDeployments.actions;

const APPLICATION_SOURCE_TABLE_COLUMNS = [
    {
        id: 'server',
        label: 'Server',
        sortable: false,
        truncate: true,
        width: '30%',
    },
    {
        id: 'webhook',
        label: 'Webhook',
        sortable: false,
        truncate: true,
        width: '30%',
    },
    {
        id: 'status',
        label: 'Status',
        sortable: false,
        truncate: true,
        width: '25%',
    },
    {
        id: 'action',
        label: 'Action',
        sortable: false,
        truncate: true,
        width: '15%',
    },
];

const ApplicationSourceComponent = (props: ApplicationSourceComponentProp) => {
    const { folder } = props;

    const servers = useAppSelector(getServersSelector);
    const webhookSources = useAppSelector(getWebhookSourcesSelector);
    const disconnectedServers = useAppSelector(getDisconnectedServers);

    const [isDialogOpen, setIsDialogOpen] = useState<boolean>(false);
    const [isEditMode, setIsEditMode] = useState<boolean>(false);
    const [selectedServer, setSelectedServer] = useState<Server | undefined>(undefined);
    const [selectedWebhookSource, setSelectedWebhookSource] = useState<WebhookSource | undefined>(undefined);

    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(getFilters(folder.id));
        dispatch(loadWebhookSources(folder.id));
        dispatch(loadExternalDeployments());
        dispatch(updateBreadcrumbs(folder));
        dispatch(loadServers(folder.id));
    }, []);

    const goBack = () => dispatch(navigateTo({ pathSuffix: `folders/${folder.id}/application-pipelines/external-deployments` }));

    const onDeleteButtonClick = (webhookSource: WebhookSource) => {
        setSelectedWebhookSource(webhookSource);
        setIsDialogOpen(true);
    };

    const onDeleteConfirmed = () => {
        if (selectedWebhookSource && selectedWebhookSource.id) {
            dispatch(deleteWebhookSource(selectedWebhookSource.id));
            setIsDialogOpen(false);
        }
    };

    const onEditButtonClick = (webhookSource: WebhookSource) => {
        const serverSelected = servers.find((server) => server.id === webhookSource.sourceServer);

        setSelectedServer(serverSelected);
        setSelectedWebhookSource(webhookSource);
        setIsEditMode(true);
    };

    const getImage = (src: string, title: string) => {
        return (
            <div className="application-source-img">
                <img alt={title} src={src} />
                {title}
            </div>
        );
    };

    const getServerTitleByServerId = (serverId: string | null) => {
        const serverSelected = servers.find((server) => server.id === serverId);
        return serverSelected ? getImage(getServerIcon(serverSelected.type), serverSelected.title) : '';
    };

    const getTableActions = (webhookSource: WebhookSource) => {
        return (
            <DotTableActions
                actions={[
                    { iconId: 'edit', id: 'edit', label: 'Edit', onClick: () => onEditButtonClick(webhookSource), tooltip: 'Edit' },
                    { iconId: 'delete', id: 'delete', label: 'Delete', onClick: () => onDeleteButtonClick(webhookSource), tooltip: 'Delete' },
                ]}
                id={webhookSource.id || ''}
            />
        );
    };

    const getConnectionStatus = (serverId: string) => {
        const connected = !disconnectedServers.includes(serverId);
        return (
            <DotChip
                className={connected ? 'application-source-status-success' : ''}
                error={!connected}
                isDeletable={false}
                size="small"
                startIcon={<DotIcon className={connected ? 'success' : 'error'} iconId={connected ? 'check-solid' : 'error-solid'} />}
            >
                {connected ? 'Connected' : 'Disconnected'}
            </DotChip>
        );
    };

    const tableData = webhookSources.map((webhookSource) => ({
        id: webhookSource.id ? webhookSource.id : undefined,
        rowData: {
            action: getTableActions(webhookSource),
            status: getConnectionStatus(webhookSource.sourceServer || ''),
            webhook: getImage(IMG_SRC.webhookSource, webhookSource.title),
            server: getServerTitleByServerId(webhookSource.sourceServer),
        },
    }));

    return (
        <DotThemeProvider>
            {!isEditMode && (
                <>
                    <div className="application-source">
                        <DotTypography variant="h1">
                            <DotIconButton iconId="back" iconSize="medium" onClick={goBack} size="medium" tooltip="Back" />
                            Application sources
                        </DotTypography>
                        <DotTable className="application-source-table" columns={APPLICATION_SOURCE_TABLE_COLUMNS} data={tableData} onUpdateData={noop} />
                    </div>

                    <DotConfirmationDialog
                        className="application-source-dialog"
                        message={
                            <p>
                                You are about to <b>DELETE Application source and endpoint</b>. Click <b>Delete source and endpoint</b> to confirm{' '}
                            </p>
                        }
                        onCancel={() => setIsDialogOpen(false)}
                        onSubmit={() => onDeleteConfirmed()}
                        open={isDialogOpen}
                        submitButtonProps={{ label: 'Delete source and endpoint', startIcon: <DotIcon iconId="delete" />, type: 'destructive' }}
                        title={<DotTypography variant="h1">Delete application source</DotTypography>}
                    />
                </>
            )}

            {isEditMode && (
                <FilterApplications
                    onCancel={() => setIsEditMode(false)}
                    onFilterBack={() => setIsEditMode(false)}
                    onSave={() => setIsEditMode(false)}
                    server={selectedServer}
                    webhookSource={selectedWebhookSource as WebhookSource}
                />
            )}
        </DotThemeProvider>
    );
};

export const ApplicationSource = memo(ApplicationSourceComponent);
