import { all, call, CallEffect, put, select, takeEvery, takeLatest } from 'redux-saga/effects';
import moment from 'moment';
import { IHttpResponse } from 'angular';
import { PayloadAction } from '@reduxjs/toolkit';
import { httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { applicationManagement, ApplicationManagementState, CreateReleaseAction } from './managed-application.reducer';
import { DevOpsDeploymentsFeatureResponse, FetchTemplatesResponse, ManagedApplicationsPage, TemplateFilter } from '../managed-application.types';
import { toaster } from '../../external-deployments/ducks/external-deployments.saga';
import { AxiosError } from 'axios';
import { ClientSettings, CustomConfiguration, FilterQueryParams, Release } from '../../../../../../../../../core/xlr-ui/app/types';
import { DEV_OPS_DEPLOYMENT_FEATURE_SETTINGS_TYPE } from '../../../../../../../../../core/xlr-ui/app/features/feature-settings/feature-settings-utils';
import { getApplicationManagementState } from './managed-application.selectors';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { UserProfile } from '../../../../../../../../../core/xlr-ui/app/js/profile/user-profile';
import IdsFactory from '../../../../../../../../../core/xlr-ui/app/js/util/ids';

const Ids = IdsFactory();

const {
    setTemplates,
    fetchTemplates,
    loadManagedApplications,
    setIsLoading,
    setManagedApplications,
    setManagedApplicationsCount,
    setPage,
    storeFilters,
    getFilters,
    createRelease,
} = applicationManagement.actions;

const { navigateTo } = navigation.actions;

export function* withLoadingState<R>(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        const result: R = yield effect;
        return result;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* storeFiltersAction() {
    const { page } = yield select(getApplicationManagementState);
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    clientSettings.setManagedApplicationsFilters({ ...page });
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    FiltersQueryParams.update({ ...page });
}

export function* executeFetchTemplatesAction(action: PayloadAction<TemplateFilter>) {
    const { parentId } = action.payload;
    const {
        data: { applicationPipelineTag },
    }: IHttpResponse<DevOpsDeploymentsFeatureResponse> = yield call(httpGET, `/settings/features/${DEV_OPS_DEPLOYMENT_FEATURE_SETTINGS_TYPE}`);
    const {
        data: { cis },
    }: IHttpResponse<FetchTemplatesResponse> = yield call(httpPOST, `releases/templates/search`, {
        parentId,
        tags: [applicationPipelineTag],
    });
    yield put(setTemplates(cis));
}

export function* executeFetchManagedApplicationsAction() {
    const { page }: ApplicationManagementState = yield select(getApplicationManagementState);
    try {
        const {
            data: { count, managedApplications },
        } = yield call(
            withLoadingState,
            call(httpGET, `/api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${page.resultsPerPage}`, true),
        );
        yield put(setManagedApplications(managedApplications));
        yield put(setManagedApplicationsCount(count));
    } catch (e: unknown) {
        const err = e as AxiosError<CustomConfiguration, unknown>;
        const errServerData = err.response?.data;
        const errorMessage = `Error fetching managed applications data. Check connection to ${errServerData?.title} HTTP Connection or check application logs for more details.`;
        yield call(toaster.error, errorMessage);
    }
}

export function* getFiltersAction(action: PayloadAction<string>) {
    const folderId: string = action.payload;
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    const filters: ManagedApplicationsPage = clientSettings.getManagedApplicationsFilters();
    if (filters) {
        yield put(
            setPage({
                folderId,
                page: filters.page,
                order: filters.order,
                orderBy: filters.orderBy,
                resultsPerPage: filters.resultsPerPage,
            }),
        );
    }
}

export function* createReleaseAction(action: PayloadAction<CreateReleaseAction>) {
    const { template, parentId } = action.payload;

    let folderId = parentId;
    if (!folderId) {
        folderId = Ids.getParentId(Ids.toDomainId(template.id));
    }

    const releaseToCreate = {
        autoStart: true,
        abortOnFailure: false,
        allowPasswordsInAllFields: false,
        archiveRelease: false,
        description: template.description,
        disableNotifications: false,
        dueDate: moment().add(60, 'minutes').toDate(),
        flag: template.flag,
        tags: template.tags,
        owner: UserProfile.getCurrentUser(),
        parentId: folderId,
        plannedDuration: null,
        scheduledStartDate: moment().toDate(),
        scriptUsername: template.scriptUsername,
        scriptUserPassword: template.scriptUserPassword,
        syntheticProperties: template.syntheticProperties,
        variableMapping: template.variableMapping,
        templateId: template.id,
        title: template.title,
        variables: template.variables,
    };

    const { data }: IHttpResponse<Release> = yield call(httpPOST, 'releases', releaseToCreate);
    yield put(navigateTo({ pathSuffix: `stepper/${data.id}` }));
}

export function* managedApplicationSaga() {
    yield all([
        takeLatest(fetchTemplates, executeFetchTemplatesAction),
        takeLatest(loadManagedApplications, executeFetchManagedApplicationsAction),
        takeEvery(storeFilters, storeFiltersAction),
        takeEvery(getFilters, getFiltersAction),
        takeLatest(createRelease, createReleaseAction),
    ]);
}
