import React from 'react';
import AceEditor from 'react-ace';
import { DotDialog, DotIconButton, DotAlertBanner } from '@digital-ai/dot-components';
import { toaster } from '../../../ducks/external-deployments.saga';
import { WebhookSourceAutoConfigDetails } from '../../../external-deployment.types';
import { ENDPOINT_TYPE } from '../../../constants';
import { isArgoEndpoint } from '../../../helper/utils';

export interface DiscoverModalProps {
    configureDetails: WebhookSourceAutoConfigDetails;
    isModalOpen: boolean;
    onModalClose: () => void;
    onModalSubmit: () => void;
    type: string;
}

export const DiscoverModal = (props: DiscoverModalProps) => {
    const { configureDetails, isModalOpen, onModalClose, onModalSubmit, type } = props;
    const title = `Configuration ${configureDetails.success ? 'ready' : 'failed'}`;

    const editorOptions = {
        readOnly: true,
    };

    const copyCode = () => {
        navigator.clipboard
            .writeText(configureDetails.details)
            .then(() => toaster.success('Configuration code was copied to the clipboard!', { preventDuplicates: true }))
            .catch((_) => toaster.error('Configuration code cannot be copied to the clipboard. Please, check your browser settings.', 'Error', {}));
    };

    const renderArgoDetails = (
        <div className="configuration-details-wrapper">
            <DotIconButton className="copy-icon" data-testid="copy-btn" iconId="duplicate" onClick={copyCode} tooltip="Copy to clipboard" />
            <AceEditor
                fontSize={12}
                height="100%"
                mode="yaml"
                name="configure-run-ace-editor"
                placeholder="Configuration template was empty"
                setOptions={editorOptions}
                theme="tomorrow"
                value={configureDetails.details}
                width="100%"
            />
        </div>
    );

    const renderDeployDetails = configureDetails.success && (
        <div className="configuration-details-wrapper" style={{ height: 'fit-content' }}>
            Webhook and CI have been created.
            <br />
            View CI in Deploy
            <DotIconButton iconId="open-new-tab" onClick={() => window.open(configureDetails.url, '_blank')} tooltip="Open webhook link" />
        </div>
    );

    return (
        <DotDialog
            className="configure-run-modal"
            onCancel={onModalClose}
            onSubmit={onModalSubmit}
            open={isModalOpen}
            submitButtonProps={{ disabled: !configureDetails.success, label: 'Next' }}
            title={title}
        >
            <DotAlertBanner action={null} className="configure-success-banner" severity={configureDetails.success ? 'success' : 'error'}>
                {configureDetails.message}
                {configureDetails.success && (
                    <DotIconButton
                        data-testid="modal-new-tab"
                        iconId="open-new-tab"
                        onClick={() => window.open(configureDetails.url, '_blank')}
                        tooltip="Open webhook link"
                    />
                )}
            </DotAlertBanner>
            {isArgoEndpoint(type) && renderArgoDetails}
            {type === ENDPOINT_TYPE.deploy && renderDeployDetails}
        </DotDialog>
    );
};
