package com.xebialabs.xlrelease.status.sse.service

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.joda.JodaModule
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.repository.SSERepository
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import javax.ws.rs.core.MediaType
import javax.ws.rs.sse.{OutboundSseEvent, SseEventSink}

trait ServerSentEventsService {
  def add(sseEventSink: SseEventSink): Unit

  def send(id: String, event: ExternalDeploymentEvent): Unit
}

@Service
class ServerSentEventsServiceImpl @Autowired()(sseRepository: SSERepository) extends ServerSentEventsService with Logging {

  private val mapper = new ObjectMapper()
  mapper.registerModule(DefaultScalaModule)
  mapper.registerModule(new JodaModule)

  private val applicationEventId = "applications-sse"

  override def add(sseEventSink: SseEventSink): Unit = {
    sseRepository.addSink(applicationEventId, sseEventSink)

    logger.info(s"New emitter $sseEventSink has been added")

  }

  def send(id: String, event: ExternalDeploymentEvent): Unit = {
    val sseEvent = buildSseEvent(EndpointExternalDeploymentEvent(id, event))
    sseRepository.sendEventToSink(applicationEventId, sseEvent)
  }

  private def buildSseEvent(event: EndpointExternalDeploymentEvent): OutboundSseEvent = {

    var name :String = ""

    event.state match {
      case e: CreateStatusEvent if isApplicationEvent(e) => name = "application-created"
      case e: DeleteStatusEvent if isApplicationEvent(e) => name = "application-deleted"
      case _: CreateStatusEvent => name = "application-package-created"
      case _: DeleteStatusEvent => name = "application-package-deleted"
      case _: UpdateStatusEvent => name = "application-changed"
      case _ => // ignore
    }

    sseRepository.newEventBuilder()
          .name(name)
          .mediaType(MediaType.APPLICATION_JSON_TYPE)
          .data(mapper.writeValueAsString(event))
          .reconnectDelay(3000)
          .build()
  }

  private def isApplicationEvent(event: BaseExternalDeploymentEvent) = {
    Option(event.destination).isDefined || Option(event.namespace).isDefined
  }
}

sealed trait ServerSentEvent {
  val endpointId: String
}

case class EndpointExternalDeploymentEvent(override val endpointId: String, state: ExternalDeploymentEvent) extends ServerSentEvent
