import { navigation } from '../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { call, put } from 'redux-saga/effects';
import { getFolderOrWaitForIt } from '../../../../../../../../core/xlr-ui/app/features/folders/ducks/folders.saga';
import { FolderPathService, MainNavigationItem } from '../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import getAngularService from '../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { ApplicationPipelinesSidebarProvider } from './application-pipelines-sidebar-provider';
import { SagaIterator } from 'redux-saga';

const { setSidebarMenu, setBackItem, updateBreadcrumbs } = navigation.actions;

describe('ApplicationPipelinesSidebarProvider', () => {
    const folderSidebarProvider = new ApplicationPipelinesSidebarProvider();

    describe('isSupported', () => {
        it('should be false for /somethingsomething path', () => {
            expect(folderSidebarProvider.isSupported('/somethingsomething')).toBe(false);
        });

        it('should be false for paths inside /folders that dont contain /application-pipelines', () => {
            expect(folderSidebarProvider.isSupported('/folders/Applications/somethingsomething/blabla')).toBe(false);
        });

        it('should be true for paths inside /folders that contain /application-pipelines', () => {
            expect(folderSidebarProvider.isSupported('/folders/Applications/somethingsomething/blabla/application-pipelines')).toBe(true);
        });
    });

    describe('provide', () => {
        const UiExtensionsService = {
            getMenuItems: jest.fn(),
        };

        const folderPathService: FolderPathService = {
            getPathFromParentFolderId: () => [
                {
                    id: 'Applications',
                    title: 'My Root',
                },
                {
                    id: 'Applications/Folder1',
                    title: 'My Folder',
                },
            ],
            getPathFromReleaseId: jest.fn(),
        };

        const selectedFolder = {
            id: 'Applications/Folder1',
            title: 'My Folder',
        };

        const folderTabs: MainNavigationItem[] = [
            {
                icon: 'applications',
                label: 'Applications management',
                pathSuffix: 'applications-management',
                weight: 10,
                permitted: true,
            },
            {
                icon: 'package',
                label: 'External deployments',
                pathSuffix: 'external-deployments',
                weight: 20,
                permitted: false,
            },
        ];

        it('should call proper saga', () => {
            const generator: SagaIterator = folderSidebarProvider.provide('Applications/Folder1/application-pipelines');
            expect(generator.next().value).toMatchObject(call(getFolderOrWaitForIt));
            expect(generator.next(selectedFolder).value).toMatchObject(call(getAngularService, 'UiExtensionsService'));
            expect(generator.next(UiExtensionsService).value).toMatchObject(
                call([UiExtensionsService, UiExtensionsService.getMenuItems], 'xlrelease.sidebar.Folder.ApplicationPipelines', selectedFolder),
            );
            expect(generator.next(folderTabs).value).toMatchObject(
                put(
                    setSidebarMenu({
                        sidebarMenu: [
                            {
                                pathSuffix: 'folders/Applications/Folder1/application-pipelines/applications-management',
                                selected: true,
                                startIconId: 'applications',
                                text: 'Applications management',
                            },
                        ],
                        withFolder: false,
                    }),
                ),
            );
            expect(generator.next().value).toMatchObject(
                put(
                    setBackItem({
                        pathSuffix: `folders/${selectedFolder.id}`,
                        title: selectedFolder.title,
                        text: selectedFolder.title,
                    }),
                ),
            );
            expect(generator.next().value).toMatchObject(call(getAngularService, 'FolderPathService'));
            expect(generator.next(folderPathService).value).toMatchObject(
                call([folderPathService, folderPathService.getPathFromParentFolderId], selectedFolder.id),
            );
            expect(generator.next(folderPathService.getPathFromParentFolderId('')).value).toMatchObject(
                put(
                    updateBreadcrumbs([
                        {
                            href: '#/folders/Applications',
                            text: 'My Root',
                            underline: 'hover',
                        },
                        {
                            href: '#/folders/Applications/Folder1',
                            text: 'My Folder',
                            underline: 'hover',
                        },
                    ]),
                ),
            );
            expect(generator.next().done).toBe(true);
        });
    });
});
