package com.xebialabs.xlrelease.status.webhook.events

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.platform.script.jython.ScriptSource
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlplatform.webhooks.events.handlers.EventProcessorHandler
import com.xebialabs.xlplatform.webhooks.queue.JmsEventPublisherHandler
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.status.repository.persistence.{ExternalDeploymentPersistence, ExternalDeploymentUid}
import com.xebialabs.xlrelease.status.service.script.{FilterScriptService, MapScriptService}
import com.xebialabs.xlrelease.status.sse.service.ServerSentEventsService
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import com.xebialabs.xlrelease.webhooks.consumers.BaseConsumerHandler
import com.xebialabs.xlrelease.webhooks.utils.JythonScriptEventProcessingUtils
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Value}
import org.springframework.jms.core.JmsTemplate
import org.springframework.stereotype.Component
import org.springframework.util.ResourceUtils

import java.io.FileNotFoundException
import java.util.Date
import scala.util.{Failure, Success, Try}

@Component
class StatusWebhookConsumerHandler @Autowired()(val jmsTemplate: JmsTemplate,
                                                @Value("${xl.queue.queueName}") val queueName: String,
                                                configurationRepository: ConfigurationRepository,
                                                externalDeploymentPersistence: ExternalDeploymentPersistence,
                                                serverSentEventsService: ServerSentEventsService,
                                                jythonScriptUtils: JythonScriptEventProcessingUtils,
                                                filterScriptService: FilterScriptService,
                                                mapScriptService: MapScriptService
                                               )
  extends EventProcessorHandler[Event, Event, StatusWebhookEventSource]
    with JmsEventPublisherHandler[Event, StatusWebhookEventSource]
    with BaseConsumerHandler[StatusWebhookEventSource]
    with Logging {

  import jythonScriptUtils._

  override def consumerConfigType: Type = Type.valueOf(classOf[StatusWebhookEventSource])

  override def consumeEvent(config: StatusWebhookEventSource, event: Event): Boolean = {
    event match {
      case httpEvent: HttpRequestEvent if config.consumerEnabled =>
        try {
          handleStatusEvent(config, httpEvent)
          true
        } catch {
          case e: Throwable =>
            throw e
        }
      case _ => true
    }
  }

  override def filter(config: StatusWebhookEventSource, event: Event): Boolean = {
    if(config.isInstanceOf[RemoteStatusWebhookEventSource]) {
      filterScriptService.executeScript(config, event)
    } else {
      nonEmptyScriptLocation(config.filterScript).forall { path =>
        Try(ResourceUtils.getFile(path)).map { _ =>
          executeFilter(config, event, sandbox = false)
            .apply(logger)
            .apply(ScriptSource.byResource(path))
        }.recover {
          case e: FileNotFoundException =>
            logger.error("Filter caused exception, assuming false", e)
            false
        }.get
      }
    }
  }

  def map(config: StatusWebhookEventSource, event: Event): Event = {
    if(config.isInstanceOf[RemoteStatusWebhookEventSource]) {
      mapScriptService.executeScript(config, event)
    } else {
      executeMap(config, event, sandbox = false)
        .apply(logger)
        .apply(ScriptSource.byResource(scriptLocationOrDefault(config.getType, config.mapScript)))
    }
  }

  def handleStatusEvent(config: StatusWebhookEventSource, event: HttpRequestEvent): Unit = {
    logger.trace(s"receivedEvent from ${event.sourceId} [${event.content}]")
    val sourceServerId = config.sourceServer.getId
    val connection = configurationRepository.read[StatusHttpConnection](sourceServerId)
    val externalDeploymentEvent: Option[ExternalDeploymentEvent] = Try(map(config, event)) match {
      case Success(event: StatusWebhookEvent) if event.operation == StateOperation.CreateState =>
        Some(CreateStatusEvent(
          event.applicationName,
          event.applicationUid,
          event.applicationPath,
          VersionState(event.versionLabel, event.versionState),
          event.destination,
          event.destinationUid,
          event.namespace,
          event.statusGroup
        ))
      case Success(event: StatusWebhookEvent) if event.operation == StateOperation.RemoveState =>
        Some(DeleteStatusEvent(
          event.applicationName,
          event.applicationUid,
          VersionState(event.versionLabel, event.versionState),
          event.destination,
          event.destinationUid,
          event.namespace)
        )
      case Success(event: StatusWebhookEvent) if Option(event.deploymentMetadata).isDefined =>
        Some(UpdateStatusEvent(
          event.applicationName,
          event.applicationUid,
          event.applicationPath,
          event.statusGroup,
          StateMetadata(
            event.destination,
            event.destinationUid,
            event.namespace,
            VersionState(event.versionLabel, event.versionState),
            event.deploymentMetadata.deploymentStatus,
            event.deploymentMetadata.deploymentType,
            event.deploymentMetadata.user,
            event.deploymentMetadata.lastChangeTime
          )
        ))
      case Success(errorObject) =>
        val errorType = Option(errorObject).map(_.getClass.getName).getOrElse("null")
        error(s"Illegal response in ${connection.getId} transformerScript response. " +
          s"Returned type was $errorType, expected state or package objects.")
        None
      case Failure(exception) =>
        error(s"Exception thrown from transformerScript for ${connection.getType.toString} endpoint.", exception)
        None
    }
    externalDeploymentEvent.foreach(processedEvent => {
      serverSentEventsService.send(sourceServerId, processedEvent)
      externalDeploymentPersistence.persist(
        ExternalDeploymentUid(
          config.getId, processedEvent.applicationUid, processedEvent.destinationUid),
          processedEvent
        )
    })
  }
}

sealed class ExternalDeploymentEvent(val applicationUid: String, val destinationUid: String)

abstract class BaseExternalDeploymentEvent(
                                          val applicationName: String,
                                          override val applicationUid: String,
                                          val applicationPath: String,
                                          val destination: String,
                                          override val destinationUid: String,
                                          val namespace: String
                                          ) extends ExternalDeploymentEvent(applicationUid, destinationUid)

final case class CreateStatusEvent(
                                    override val applicationName: String,
                                    override val applicationUid: String,
                                    override val applicationPath: String,
                                    versionTag: VersionState,
                                    override val destination: String,
                                    override val destinationUid: String,
                                    override val namespace: String,
                                    statusGroup: String
                                  )
  extends BaseExternalDeploymentEvent(applicationName, applicationUid, applicationPath, destination, destinationUid, namespace)

final case class DeleteStatusEvent(
                                    override val applicationName: String,
                                    override val applicationUid: String,
                                    versionTag: VersionState,
                                    override val destination: String,
                                    override val destinationUid: String,
                                    override val namespace: String
                                  )
  extends BaseExternalDeploymentEvent(applicationName, applicationUid, null, destination, destinationUid, namespace)

final case class UpdateStatusEvent(
                                    override val applicationName: String,
                                    override val applicationUid: String,
                                    override val applicationPath: String,
                                    statusGroup: String,
                                    state: StateMetadata
                                  )
  extends BaseExternalDeploymentEvent(applicationName, applicationUid, applicationPath, state.destination,  state.destinationUid, state.namespace)

final case class VersionState(label: String, state: String)

final case class StateMetadata(destination: String,
                               destinationUid: String,
                               namespace: String,
                               versionTag: VersionState,
                               deploymentStatus: String,
                               deploymentType: String,
                               user: String,
                               lastChangeTime: Date
                              )

object StateOperation {
  final val CreateState = "create"
  final val RemoveState = "remove"
}
