package com.xebialabs.xlrelease.applications.management.service

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.applications.management.repository.persistence.ApplicationsManagementPersistence
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.status.service.ConnectionServerData
import com.xebialabs.xlrelease.status.webhook.configuration.StatusHttpConnection
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import java.lang
import java.util.Date

trait ApplicationsManagementService {
  def getManagedApplications(folderId: String, page: lang.Long, resultsPerPage: lang.Long): ApplicationsManagement
  def addManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData
  def updateManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData
  def deleteManagedApplication(managedApplicationId: String): Unit
}

@Service
class ApplicationManagementServiceImpl(
                                        configurationRepository: ConfigurationRepository,
                                        applicationsManagementPersistence: ApplicationsManagementPersistence
                                      ) extends ApplicationsManagementService with Logging {
  override def getManagedApplications(folderId: String, page: lang.Long, resultsPerPage: lang.Long): ApplicationsManagement = {
    val count = applicationsManagementPersistence.countForFolder(folderId)
    val managedApplications =
      applicationsManagementPersistence.findForFolder(folderId, resultsPerPage, resultsPerPage * page).map(ma => {
        managedApplicationToManagedApplicationData(ma)
      })
    ApplicationsManagement(count, managedApplications)
  }

  override def addManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData = {
    validateManagedApplication(managedApplication)
    val managedApplicationId = applicationsManagementPersistence.create(managedApplication)
    val created = applicationsManagementPersistence.findById(managedApplicationId).getOrElse(
      throw new NotFoundException(s"Managed Application with id [$managedApplicationId] not found")
    )
    managedApplicationToManagedApplicationData(created)
  }

  override def updateManagedApplication(managedApplication: ManagedApplication): ManagedApplicationData = {
    checkArgument(hasText(managedApplication.id), "ID is required")
    validateManagedApplication(managedApplication)
    applicationsManagementPersistence.update(managedApplication)
    managedApplicationToManagedApplicationData(managedApplication)
  }

  override def deleteManagedApplication(managedApplicationId: String): Unit = {
    checkArgument(hasText(managedApplicationId), "ID is required")
    applicationsManagementPersistence.delete(managedApplicationId)
  }

  private def validateManagedApplication(managedApplication: ManagedApplication): Unit = {
    checkArgument(hasText(managedApplication.applicationName), "Application name cannot be blank")
    checkArgument(Option(managedApplication.connectionServer).isDefined, "Connection server must be defined")
  }

  private def managedApplicationToManagedApplicationData(managedApplication: ManagedApplication): ManagedApplicationData = {
    val connection = configurationRepository.read[StatusHttpConnection](applicationsManagementPersistence.findConnectionId(managedApplication.id).get)
    ManagedApplicationData(
      managedApplication.id,
      managedApplication.applicationName,
      managedApplication.managedBy,
      managedApplication.dateCreated,
      managedApplication.environmentTag,
      managedApplication.applicationReference,
      ConnectionServerData(connection.getId, connection.getTitle, connection.getType.toString, connection.getUrl, None))
  }
}

case class ApplicationsManagement(count: Int, managedApplications: Vector[ManagedApplicationData])
case class ManagedApplicationData(id: String,
                                  applicationName: String,
                                  managedBy: String,
                                  dateCreated: Date,
                                  environmentTag: String,
                                  applicationReference: String,
                                  connectionServer: ConnectionServerData)
case class ManagedApplication(id: String,
                              applicationName: String,
                              managedBy: String,
                              dateCreated: Date,
                              environmentTag: String,
                              applicationReference: String,
                              connectionServer: String)
