import React, { ChangeEvent, memo, useEffect } from 'react';
import { Folder } from '../../../../../../../../../../core/xlr-ui/app/types';
import { ManagedApplication } from './managed-application.component';
import { ManagedApplicationTableComponentPortal } from './managed-application-table-component.portal';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { applicationManagement } from '../../ducks/managed-application.reducer';
import { ManagedApplicationType, ManagedApplicationsPage } from '../../managed-application.types';
import {
    getCreatedApplicationsCountSelector,
    getCreatedApplicationsLoadingSelector,
    getCreatedApplicationsPageSelector,
    getCreatedApplicationsSelector,
} from '../../ducks/managed-application.selectors';
import { DotColumnHeader, DotTable, DotTablePagination, DotThemeProvider, DotTypography, Order, RowsPerPageOption } from '@digital-ai/dot-components';
import {
    COLUMN_ACTIONS,
    COLUMN_APPLICATION_NAME,
    COLUMN_DATE_CREATED,
    COLUMN_DEEP_LINK,
    COLUMN_ENV_TAG,
    COLUMN_MANAGED_BY,
    COLUMN_TARGET_CONNECTION,
    COLUMN_TARGET_DEPLOYMENT,
    COLUMN_WIDTH,
} from '../../constants';

export interface ManagedApplicationTableProps {
    folder: Folder;
    onViewChange: () => void;
}

const { loadManagedApplications, setPage, storeFilters } = applicationManagement.actions;

const ManagedApplicationTableComponent = ({ folder, onViewChange }: ManagedApplicationTableProps) => {
    const createdApplications: ManagedApplicationType[] = useAppSelector(getCreatedApplicationsSelector);
    const count: number = useAppSelector(getCreatedApplicationsCountSelector);
    const isLoading: boolean = useAppSelector(getCreatedApplicationsLoadingSelector);
    const page = useAppSelector(getCreatedApplicationsPageSelector);
    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(storeFilters());
    }, [page]);

    const onPageChange = (newPage: ManagedApplicationsPage) => {
        dispatch(setPage(newPage));
        dispatch(loadManagedApplications());
    };

    return (
        <>
            <ManagedApplicationTableComponentPortal folder={folder} onViewChange={onViewChange} />
            <DotThemeProvider>
                <div className="external-deployments-table">
                    <div className="external-deployments-table-title">
                        <DotTypography variant="h1">Applications ({count})</DotTypography>
                    </div>

                    <DotTable
                        className="external-deployments-table-header"
                        columns={headerDefinitions}
                        data={[]}
                        loading={isLoading}
                        onUpdateData={(order: Order, orderBy: string) =>
                            onPageChange({
                                ...page,
                                folderId: folder.id,
                                order,
                                orderBy,
                            })
                        }
                        order={page.order}
                        orderBy={page.orderBy}
                    />

                    {createdApplications.map((createdApplication: ManagedApplicationType) => {
                        return <ManagedApplication key={createdApplication.applicationName} {...createdApplication} />;
                    })}

                    <DotTablePagination
                        count={count}
                        onPageChange={(newPage: number) =>
                            onPageChange({
                                ...page,
                                folderId: folder.id,
                                page: newPage,
                            })
                        }
                        onRowsPerPageChange={(event: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) =>
                            onPageChange({
                                ...page,
                                folderId: folder.id,
                                page: 0,
                                resultsPerPage: parseInt(event.target.value) as RowsPerPageOption,
                            })
                        }
                        page={page.page}
                        rowsPerPage={page.resultsPerPage}
                        typography="body1"
                    />
                </div>
            </DotThemeProvider>
        </>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: undefined,
        id: COLUMN_APPLICATION_NAME,
        label: 'Application name',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_MANAGED_BY,
        label: 'Managed by',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_DATE_CREATED,
        label: 'Date created',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_TARGET_DEPLOYMENT,
        label: 'Target deployment',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_ENV_TAG,
        label: 'Environment tag',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_TARGET_CONNECTION,
        label: 'Target Connection',
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_DEEP_LINK,
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
    {
        align: undefined,
        id: COLUMN_ACTIONS,
        sortable: false,
        truncate: false,
        width: COLUMN_WIDTH,
    },
];

export const ManagedApplicationTable = memo(ManagedApplicationTableComponent);
