import { all, call, CallEffect, put, select, takeEvery, takeLatest } from 'redux-saga/effects';
import { IHttpResponse } from 'angular';
import { PayloadAction } from '@reduxjs/toolkit';
import { httpGET, httpPOST } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import { applicationManagement, ApplicationManagementState } from './managed-application.reducer';
import { DevOpsDeploymentsFeatureResponse, FetchTemplatesResponse, ManagedApplicationsPage, TemplateFilter } from '../managed-application.types';
import { toaster } from '../../external-deployments/ducks/external-deployments.saga';
import { AxiosError } from 'axios';
import { ClientSettings, CustomConfiguration, FilterQueryParams } from '../../../../../../../../../core/xlr-ui/app/types';
import { DEV_OPS_DEPLOYMENT_FEATURE_SETTINGS_TYPE } from '../../../../../../../../../core/xlr-ui/app/features/feature-settings/feature-settings-utils';
import { getApplicationManagementState } from './managed-application.selectors';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

const {
    setTemplates,
    fetchTemplates,
    loadManagedApplications,
    setIsLoading,
    setManagedApplications,
    setManagedApplicationsCount,
    setPage,
    storeFilters,
    getFilters,
} = applicationManagement.actions;

export function* withLoadingState<R>(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        const result: R = yield effect;
        return result;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* storeFiltersAction() {
    const { page } = yield select(getApplicationManagementState);
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    clientSettings.setManagedApplicationsFilters({ ...page });
    const FiltersQueryParams: FilterQueryParams = yield call(getAngularService, 'FiltersQueryParams');
    FiltersQueryParams.update({ ...page });
}

export function* executeFetchTemplatesAction(action: PayloadAction<TemplateFilter>) {
    const { parentId } = action.payload;
    const {
        data: { applicationPipelineTag },
    }: IHttpResponse<DevOpsDeploymentsFeatureResponse> = yield call(httpGET, `/settings/features/${DEV_OPS_DEPLOYMENT_FEATURE_SETTINGS_TYPE}`);
    const {
        data: { cis },
    }: IHttpResponse<FetchTemplatesResponse> = yield call(httpPOST, `releases/templates/search`, {
        parentId,
        tags: [applicationPipelineTag],
    });
    yield put(setTemplates(cis));
}

export function* executeFetchManagedApplicationsAction() {
    const { page }: ApplicationManagementState = yield select(getApplicationManagementState);
    try {
        const {
            data: { count, managedApplications },
        } = yield call(
            withLoadingState,
            call(httpGET, `/api/v1/managed-application?page=${page.page}&folderId=${page.folderId}&resultsPerPage=${page.resultsPerPage}`, true),
        );
        yield put(setManagedApplications(managedApplications));
        yield put(setManagedApplicationsCount(count));
    } catch (e: unknown) {
        const err = e as AxiosError;
        const errServerData: CustomConfiguration = err.response?.data;
        const errorMessage = `Error fetching managed applications data. Check connection to ${errServerData.title} HTTP Connection or check application logs for more details.`;
        yield call(toaster.error, errorMessage);
    }
}

export function* getFiltersAction(action: PayloadAction<string>) {
    const folderId: string = action.payload;
    const clientSettings: ClientSettings = yield call(getAngularService, 'ClientSettings');
    const filters: ManagedApplicationsPage = clientSettings.getManagedApplicationsFilters();
    if (filters) {
        yield put(
            setPage({
                folderId,
                page: filters.page,
                order: filters.order,
                orderBy: filters.orderBy,
                resultsPerPage: filters.resultsPerPage,
            }),
        );
    }
}

export function* managedApplicationSaga() {
    yield all([
        takeLatest(fetchTemplates, executeFetchTemplatesAction),
        takeLatest(loadManagedApplications, executeFetchManagedApplicationsAction),
        takeEvery(storeFilters, storeFiltersAction),
        takeEvery(getFilters, getFiltersAction),
    ]);
}
