import React, { memo, useState } from 'react';
import { DotAvatar, DotButton, DotCard, DotCardContent, DotIcon, DotIconButton, DotMenu, DotPill, MenuItemProps } from '@digital-ai/dot-components';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';
import { ManagedApplicationType } from '../../managed-application.types';
import './managed-application.component.less';
import { getServerIcon } from '../../../external-deployments/helper/utils';

const ManagedApplicationComponent = ({
    applicationName,
    managedBy,
    dateCreated,
    applicationReference: targetDeployment,
    environmentTag,
    connectionServer,
}: ManagedApplicationType) => {
    const isManagedByRelease = managedBy === 'Release';
    const { name: targetConnection, url: deploymentsUrl, type: serverType } = connectionServer;
    const serverIcon = getServerIcon(serverType);
    const [open, setOpen] = useState(false);
    const [anchorEl, setAnchorEl] = useState<undefined | HTMLElement>(undefined);
    const handleToggle = (event: React.MouseEvent<HTMLButtonElement, MouseEvent>) => {
        setAnchorEl(event.currentTarget);
        setOpen(!open);
    };
    const menuItems: MenuItemProps[] = [
        { key: 'edit', children: 'Edit' },
        { key: 'delete', children: 'Delete' },
    ];

    return (
        <DotCard className="created-application-card" key={applicationName}>
            <div data-testid={`app${applicationName}`}>
                <DotCardContent>
                    <div className="application-name-column" title={applicationName}>
                        {applicationName}
                    </div>

                    <div className="managed-by-column">
                        <DotPill
                            labelcolor="#3B485C !important"
                            bordercolor={isManagedByRelease ? '#3D6C9E' : undefined}
                            backgroundcolor={isManagedByRelease ? '#E8EDF3 !important' : '#F3F5F6 !important'}
                            variant="outlined"
                            label={upperFirst(managedBy)}
                            size="small"
                            icon={<DotIcon iconId={isManagedByRelease ? 'release' : 'lock'} />}
                            status="success"
                        />
                    </div>

                    <div className="date-created-column">{moment(dateCreated).format('MMM DD, YYYY')}</div>

                    <div className="target-deployment-column">
                        <DotAvatar size="small" imageSrc={serverIcon} alt="ArgoCD Logo" />
                        {targetDeployment}
                    </div>

                    <div className="environment-tag-column">
                        <DotPill
                            labelcolor="#3B485C !important"
                            bordercolor="#498500"
                            backgroundcolor="#E9F0E0 !important"
                            variant="outlined"
                            label={environmentTag}
                            size="small"
                            icon={<DotIcon iconId="environment" />}
                            status="success"
                        />
                    </div>

                    <div className="target-connection-column">{targetConnection}</div>

                    <div className="link-column">
                        <DotButton onClick={() => window.open(deploymentsUrl, '_blank')} type="outlined" tooltip="View deployments">
                            View deployments
                        </DotButton>
                    </div>

                    <div className="actions-column">
                        <div className="actions">
                            <DotIconButton iconId="options" onClick={handleToggle} />
                        </div>
                        <DotMenu
                            id="actions-menu"
                            onLeave={() => setOpen(false)}
                            onSelect={() => setOpen(false)}
                            anchorEl={anchorEl}
                            disablePortal={true}
                            open={open}
                            menuItems={menuItems}
                        />
                    </div>
                </DotCardContent>
            </div>
        </DotCard>
    );
};

export const ManagedApplication = memo(ManagedApplicationComponent);
