import React from 'react';
import { DashboardView, DotCoreApiProvider, DotDashboardHeader, DotMetadataApiProvider, DotProgress } from '@digital-ai/dot-components';

import { mountWithStoreAndTheme, preparePortalContainer, ReactWrapper } from '@xlr-ui/tests/unit/testing-utils';
import { AnalyticsDashboardPage, AnalyticsDashboardPageProps } from './analytics-dashboard-page.component';
import { analytics } from '../../ducks/analytics.reducer';
import { MicrostrategyDashboard } from './microstrategy/microstrategy-dashboard.component';

const { loadDashboard, clearSelectedDashboard, favoriteDashboardFromDetails } = analytics.actions;

describe('AnalyticsDashboardPage', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();
    const dashboardId = '1';
    const defaultState = {
        analytics: {
            selectedDashboard: undefined,
            intelligenceConfiguration: undefined,
            updatingDashboardIds: [],
        },
        profile: {
            favoriteDashboards: [],
        },
    };

    const prepareBody = () => {
        const portal = global.document.createElement('div');
        portal.setAttribute('id', 'dashboard-placeholder');
        const body = global.document.querySelector('body');
        body?.appendChild(portal);
    };

    const defaultProps: AnalyticsDashboardPageProps = {
        dashboardId,
    };

    const originalDashboard = {
        author_fullname: 'Dashboard1 Author',
        author_id: 'id1a',
        target_apps: ['RELEASE'],
        external_id: '1',
        name: 'Test dashboard 1',
        description: '',
        dashboard_url: 'https://test.com',
        server_url: 'https://test.com',
        external_embedding_id: null,
        id: 'id1',
        lifecycle_state: DashboardView.lifecycle_state.DRAFT,
        application_instances: [],
        categories: [],
        bi_type: DashboardView.bi_type.MICROSTRATEGY,
        is_ootb_dashboard: false,
        thumbnail: '',
        external_object: { page_count: 3 },
        embed_config: {},
        help_content_id: 'test-help-content-1',
        filter_configuration: [],
        favorite: false,
    };

    const mount = (props = defaultProps, state = defaultState) => {
        wrapper = mountWithStoreAndTheme(
            <DotCoreApiProvider apiUrl="http://demo">
                <DotMetadataApiProvider apiUrl="http://demo">
                    <AnalyticsDashboardPage {...props} />
                </DotMetadataApiProvider>
            </DotCoreApiProvider>,
            dispatch,
            state,
        );
    };

    const getDotProgress = () => wrapper.find(DotProgress);
    const getDotDashboardHeader = () => wrapper.find(DotDashboardHeader);

    beforeEach(() => {
        preparePortalContainer();
        prepareBody();
    });

    afterEach(() => {
        jest.clearAllMocks();
    });

    it('should dispatch actions on init', () => {
        mount();
        expect(dispatch).toHaveBeenCalledWith(loadDashboard(dashboardId));
    });

    it('should dispatch actions on unmount', () => {
        mount();
        dispatch.mockReset();
        wrapper.unmount();
        expect(dispatch).toHaveBeenCalledWith(clearSelectedDashboard());
    });

    it('should show loader if there is no dashboard loaded', () => {
        mount();
        const progress = getDotProgress();
        expect(progress).toExist();
        expect(progress.props().ariaLabel).toBe('Loading dashboard data');
        expect(progress.props().size).toBe(100);
    });

    it('should render MicrostrategyDashboard if we have a dashboard selected and token', () => {
        const folder = { id: 'Folder1', title: 'My folder' };
        mount({ ...defaultProps, folder }, {
            ...defaultState,
            analytics: {
                tokenValue: 'token',
                selectedDashboard: { embedConfig: { url: 'url' }, serverUrl: 'http', originalDashboard },
                updatingDashboardIds: [],
            },
        } as never);
        const dashboard = wrapper.find(MicrostrategyDashboard);
        expect(dashboard).toExist();
        expect(dashboard.prop('embedConfig')).toStrictEqual({ url: 'url' });
        expect(dashboard.prop('folder')).toStrictEqual(folder);
        expect(dashboard.prop('serverUrl')).toStrictEqual('http');
    });

    it('should not render MicrostrategyDashboard if we do not have a dashboard selected', () => {
        mount(defaultProps, {
            ...defaultState,
            analytics: {
                selectedDashboard: undefined,
                updatingDashboardIds: [],
            },
        } as never);
        const dashboard = wrapper.find(MicrostrategyDashboard);
        expect(dashboard).not.toExist();
    });

    describe('DotDashboardHeader', () => {
        it('should render DotDashboardHeader if we have a dashboard selected and token', () => {
            const folder = { id: 'Folder1', title: 'My folder' };
            mount({ ...defaultProps, folder }, {
                ...defaultState,
                analytics: {
                    tokenValue: 'token',
                    selectedDashboard: { embedConfig: { url: 'url' }, serverUrl: 'http', originalDashboard },
                    updatingDashboardIds: [],
                },
            } as never);

            const header = getDotDashboardHeader();
            expect(header).toExist();
            expect(header.prop('canEdit')).toBe(false);
            expect(header.prop('dashboard')).toBe(originalDashboard);
            expect(header.prop('isEdit')).toBe(false);
        });

        it('should handle go back when there is folder', () => {
            mount({ ...defaultProps, folder: { id: 'Folder1', title: 'My folder' } }, {
                ...defaultState,
                analytics: {
                    tokenValue: 'token',
                    selectedDashboard: { embedConfig: { url: 'url' }, serverUrl: 'http', originalDashboard },
                    updatingDashboardIds: [],
                },
            } as never);

            const header = getDotDashboardHeader();
            header.invoke('onClose')();
            expect(window.location.href).toBe('http://localhost/#/folders/Folder1/analytics');
        });

        it('should handle go back when there is no folder', () => {
            mount(defaultProps, {
                ...defaultState,
                analytics: {
                    tokenValue: 'token',
                    selectedDashboard: { embedConfig: { url: 'url' }, serverUrl: 'http', originalDashboard },
                    updatingDashboardIds: [],
                },
            } as never);

            const header = getDotDashboardHeader();
            header.invoke('onClose')();
            expect(window.location.href).toBe('http://localhost/#/analytics');
        });

        it('should handle favorite click', () => {
            mount(defaultProps, {
                ...defaultState,
                analytics: {
                    tokenValue: 'token',
                    selectedDashboard: { embedConfig: { url: 'url' }, serverUrl: 'http', originalDashboard },
                    updatingDashboardIds: [],
                },
            } as never);

            const header = getDotDashboardHeader();
            header.invoke('onFavorite')();
            expect(dispatch).toHaveBeenCalledWith(favoriteDashboardFromDetails({ id: dashboardId, isUnfavorite: false }));
        });
    });
});
