import React from 'react';
import noop from 'lodash/noop';
import {
    AutoCompleteValue,
    DotChip,
    AutoCompleteOption,
    DotAutoComplete,
    DotButton,
    DotAvatar,
    DotCheckbox,
    DotClickAwayListener,
    DotDrawer,
    DotIconButton,
    DotTypography,
    DotAnnotation,
} from '@digital-ai/dot-components';
import { CardSearch } from '../ducks/analytics.reducer';
import { AvatarWithName } from '@xlr-ui/app/features/tasks/components/common/avatar-with-name.component';
import { removeAuthorFromFilters } from '../helpers';
import { AnalyticsCategoriesSkeleton } from './analytics-categories.skeleton';
import { DEFAULT_PAGINATION } from '@xlr-ui/app/constants/pagination';
import './analytics-page-filter-drawer.component.less';

export interface AnalyticsPageFilterDrawerProps {
    authors: string[];
    cardSearch?: CardSearch;
    categories: string[];
    isCategoriesLoading: boolean;
    isDrawerOpened: boolean;
    numberOfFiltersApplied: number;
    onCardFilter: (currentCardSearch: CardSearch) => void;
    onClearAll: () => void;
    onDrawerClose: () => void;
}

export const AnalyticsPageFilterDrawer = ({
    authors,
    cardSearch,
    categories,
    isDrawerOpened,
    isCategoriesLoading,
    numberOfFiltersApplied,
    onCardFilter,
    onClearAll,
    onDrawerClose,
}: AnalyticsPageFilterDrawerProps) => {
    const renderCategoryCheckboxes = () => {
        return categories.map((category) => {
            const filterCategories = cardSearch?.categories || [];
            const isInFilters = filterCategories.some((filterCategory) => filterCategory === category);
            return (
                <DotCheckbox
                    checked={isInFilters}
                    key={category}
                    label={category}
                    onChange={(e) => {
                        const isChecked = e.target.checked;
                        onCardFilter({
                            ...cardSearch,
                            ...DEFAULT_PAGINATION,
                            categories: isChecked
                                ? [...filterCategories, category]
                                : filterCategories.filter((currentCategory) => currentCategory !== category),
                        });
                    }}
                />
            );
        });
    };

    const renderAuthorAutocomplete = () => {
        const authorsSearched = cardSearch?.authorNames || [];
        const authorsOptions: Array<AutoCompleteOption> = authors.map((name) => ({ title: name }));
        const filteredAuthorNames: Array<AutoCompleteOption> = authorsSearched.map((name) => ({ title: name }));
        return (
            <DotAutoComplete
                freesolo={false}
                inputId="authors"
                isOptionEqualToValue={(option: AutoCompleteOption, value: AutoCompleteOption) => option.title === value.title}
                multiple={true}
                onChange={(_ev, value) => {
                    const selectedAuthors = (value as Array<AutoCompleteOption>) ?? [];
                    onCardFilter({
                        ...cardSearch,
                        ...DEFAULT_PAGINATION,
                        authorNames: selectedAuthors.map((author) => author.title),
                    });
                }}
                options={authorsOptions}
                placeholder="Start typing to filter by authors"
                renderOption={(props, { title }: AutoCompleteOption) => (
                    <li key={title} {...props}>
                        <AvatarWithName id={`id-${title}`} title={title} />
                    </li>
                )}
                renderTags={(options: AutoCompleteValue[]) =>
                    options.map((option: AutoCompleteValue, index) => {
                        const author = option as AutoCompleteOption;
                        return (
                            <DotChip
                                avatar={<DotAvatar alt={`Image for user ${author.title}`} size="small" text={author.title} type="text" />}
                                className="author-chip"
                                key={index}
                                onDelete={() => {
                                    onCardFilter({
                                        ...cardSearch,
                                        ...DEFAULT_PAGINATION,
                                        authorNames: removeAuthorFromFilters(author.title, authorsSearched),
                                    });
                                }}
                                size="small"
                            >
                                {author.title}
                            </DotChip>
                        );
                    })
                }
                value={filteredAuthorNames}
            />
        );
    };

    return (
        <DotClickAwayListener onClickAway={isDrawerOpened ? onDrawerClose : noop}>
            <div>
                <DotDrawer
                    ModalProps={{ hideBackdrop: true }}
                    PaperProps={{
                        style: {
                            top: '104px',
                            padding: '0',
                            height: 'calc(100% - 104px)',
                        },
                    }}
                    anchor="right"
                    className="analytics-drawer"
                    open={isDrawerOpened}
                    variant="persistent"
                    width="320px"
                >
                    {isDrawerOpened ? (
                        <>
                            <div className="analytics-drawer-header">
                                <DotTypography variant="h3">Filters</DotTypography>
                                <DotIconButton
                                    iconId="close"
                                    onClick={onDrawerClose}
                                    tooltip={
                                        <>
                                            Close
                                            <DotAnnotation content="ESC" type="secondary" />
                                        </>
                                    }
                                />
                            </div>
                            <div className="analytics-drawer-actions">
                                <DotTypography variant="subtitle2">{`Applied filters (${numberOfFiltersApplied})`}</DotTypography>
                                <DotButton onClick={onClearAll} type="outlined">
                                    Clear all
                                </DotButton>
                            </div>
                            <div className="filter-group-title">
                                <DotTypography variant="overline">Filter by category</DotTypography>
                            </div>
                            <div className="filter-checkbox-group">{isCategoriesLoading ? <AnalyticsCategoriesSkeleton /> : renderCategoryCheckboxes()}</div>
                            <div className="filter-group-title">
                                <DotTypography variant="overline">Filter by author</DotTypography>
                            </div>
                            <div className="filter-checkbox-group">{renderAuthorAutocomplete()}</div>
                        </>
                    ) : null}
                </DotDrawer>
            </div>
        </DotClickAwayListener>
    );
};
