import { CardSearch } from '../ducks/analytics.reducer';
import { AnalyticsCardData, NativeFilterConfiguration, PlatformAuthor, DashboardView } from '../types';
import { ANALYTICS_SAMPLE_CARDS, DIGITAL_AI_AUTHOR } from '../constants';

export const calculateNumberOfFiltersApplied = (cardSearch: CardSearch | undefined) => {
    let numberOfFilters = 0;
    const { categories, authorNames } = cardSearch || {};
    if (categories && categories.length > 0) numberOfFilters += categories.length;
    if (authorNames && authorNames.length > 0) numberOfFilters += authorNames.length;
    return numberOfFilters;
};

export const removeCategoryFromFilters = (categoryName: string, filteredCategories: string[] | undefined) =>
    filteredCategories?.filter((currentCategory) => currentCategory !== categoryName);

export const removeAuthorFromFilters = (authorName: string, filteredAuthorNames: string[] | undefined) =>
    filteredAuthorNames?.filter((currentAuthorName) => currentAuthorName !== authorName);

export const convertArrayToString = (input: string[] | undefined): string | undefined => {
    if (input && input.length > 0) {
        return input.join(',');
    } else {
        return undefined;
    }
};

export const calculateTitleAndSubtitleForEmptyState = (isFavoriteTab: boolean, isAnalyticsConfigured: boolean): { subtitle: string; title: string } => {
    if (isFavoriteTab) {
        return {
            subtitle: 'To easily add dashboards to your favorites, simply click the star icon located next to each of the dashboards. 🌟',
            title: 'No favorites have been added so far',
        };
    } else {
        return isAnalyticsConfigured
            ? {
                  subtitle: 'Create and manage Analytics dashboards',
                  title: 'Analytics dashboards',
              }
            : {
                  subtitle: 'There are no available dashboards yet.',
                  title: 'No dashboards yet',
              };
    }
};

export const getFolderFilter = (dashboard: DashboardView, folderColumn: string): NativeFilterConfiguration | undefined => {
    const nativeFilters: Array<NativeFilterConfiguration> = dashboard.filter_configuration ?? [];
    return nativeFilters.find((filter) => filter.name === folderColumn);
};

export const mapAuthorName = (authors: PlatformAuthor[]): string[] => {
    return authors.map((author) => (author.full_name === ' ' ? DIGITAL_AI_AUTHOR : author.full_name));
};

export const getAuthorName = (dashboardItem: DashboardView) => {
    return dashboardItem.is_ootb_dashboard ? DIGITAL_AI_AUTHOR : dashboardItem.author_fullname;
};

export const mapSupersetToAnalyticsCards = (list: Array<DashboardView>, folderColumn: string): Array<AnalyticsCardData> => {
    return list.map((item) => mapSupersetToAnalyticsCard(item, folderColumn));
};

export const mapSupersetToAnalyticsCard = (item: DashboardView, folderColumn: string): AnalyticsCardData => {
    return {
        embeddingId: item.external_embedding_id,
        id: item.id,
        isFavorite: item.favorite,
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore wrong type on dot, don't really care
        embedConfig: item.embed_config,
        folderFilter: getFolderFilter(item, folderColumn),
        externalId: parseInt(item.external_id),
        tags: item.categories,
        title: item.name,
        serverUrl: item.server_url,
        subtitle: getAuthorName(item),
        imageSrc: item.thumbnail,
        contentText: item.description,
        originalDashboard: item,
    };
};

export const getFavoriteIcon = (isFavorite: boolean) => (isFavorite ? 'star-favorites-active' : 'star-favorites-default');
export const getFavoriteText = (isFavorite: boolean) => (isFavorite ? 'Remove from favorites' : 'Add to favorites');

export const createAnalyticsApiUrl = (cardSearch: CardSearch, intelligenceConfigurationId: string) => {
    const { itemsPerPage, page, name, authorNames, categories, isFavorite } = cardSearch;
    let url = `api/extension/intelligence/dashboards?serverId=${intelligenceConfigurationId}&page=${page}&resultsPerPage=${itemsPerPage}`;

    const andFilters: string[] = [];
    const orFilters: string[][] = [];
    let authorNameAndFilter = '';
    let hasOotbFilter = false;
    const ootbDashboardFilter = 'is_ootb_dashboard:1';

    if (isFavorite) {
        url += `&favorite=true`;
    }
    if (name) {
        andFilters.push(`name[isearch]:${encodeURIComponent(name)}`);
    }
    if (categories?.length) {
        andFilters.push(`categories[in]:${encodeURIComponent(categories.join(','))}`);
    }
    if (authorNames?.length) {
        const nonOotbAuthorNames = authorNames.filter((authorName) => authorName !== DIGITAL_AI_AUTHOR);

        if (authorNames.includes(DIGITAL_AI_AUTHOR)) {
            hasOotbFilter = true;
        }
        if (nonOotbAuthorNames.length) {
            authorNameAndFilter = `author_fullname[in]:${encodeURIComponent(nonOotbAuthorNames.join(','))}`;
        }
    }

    if (hasOotbFilter) {
        orFilters[0] = [...andFilters, ootbDashboardFilter];
        if (authorNameAndFilter) {
            orFilters[1] = [...andFilters, authorNameAndFilter];
        }
    } else {
        if (authorNameAndFilter) {
            andFilters.push(authorNameAndFilter);
        }
        if (andFilters.length) {
            orFilters[0] = [...andFilters];
        }
    }

    if (orFilters.length) {
        orFilters.forEach((orFilter) => {
            url += `&filter=${orFilter.join(';')}`;
        });
    }

    return url;
};

export const extractRegionFromUrl = (url: string): string | undefined => {
    if (!url.includes('digital.ai')) return;
    const urlSplit = new URL(url).hostname.split('.');
    if (urlSplit.length === 4) {
        return new URL(url).hostname.split('.')[1];
    }
};

export const getUpdatedCards = (updatedCard: AnalyticsCardData, cards: AnalyticsCardData[]) => {
    const index = cards.findIndex((card) => card.id === updatedCard.id);
    if (index !== -1) {
        // Create a new array with the updated object
        return [
            ...cards.slice(0, index), // All elements before the updated object
            { ...cards[index], ...updatedCard }, // Updated object
            ...cards.slice(index + 1), // All elements after the updated object
        ];
    }
    return cards;
};

export const calculateLastLoadedPage = (numberOfCards: number, itemsPerPage: number) => {
    if (numberOfCards === 0) return 0;
    return Math.ceil(numberOfCards / itemsPerPage) - 1;
};

export const getFilteredSampleCards = (cardSearch?: CardSearch): AnalyticsCardData[] => {
    let filteredCards = ANALYTICS_SAMPLE_CARDS;
    if (cardSearch) {
        const { name, authorNames, categories } = cardSearch;

        if (name) {
            filteredCards = filteredCards.filter((card) => card.title.toLowerCase().includes(name.toLowerCase()));
        }
        if (authorNames && authorNames.length > 0) {
            filteredCards = filteredCards.filter((card) => authorNames.includes(card.subtitle));
        }
        if (categories && categories.length > 0) {
            filteredCards = filteredCards.filter((card) => card.tags.some((tag) => categories.some((category) => tag.includes(category))));
        }
    }
    return filteredCards;
};
