import React, { useState } from 'react';
import { ButtonProps, CssCell, CssGrid, DotDialog, DotEmptyFilterState, DotEmptyState, DotIcon, DotTypography } from '@digital-ai/dot-components';
import { AnalyticsCard } from './analytics-card.component';
import { AnalyticsCardData } from '../types';
import { calculateCellProps } from '@xlr-ui/app/features/workflow/helper';
import { calculateTitleAndSubtitleForEmptyState } from '../helpers';
import { AnalyticsCardSkeleton } from './analytics-card.skeleton';
import { NUMBER_OF_LOADING_CARD_SKELETONS } from '../constants';
import './analytics-cards.component.less';

const analyticsDefaultEmptyStateImg = 'static/@project.version@/styles/img/astronaut_rafiki.svg';
const analyticsConfiguredEmptyStateImg = 'static/@project.version@/styles/img/data_analysis.svg';
const analyticsFavouritesEmptyStateIllustration = 'digitalai-favorite';

export interface AnalyticsCardsProps {
    cards: AnalyticsCardData[];
    folderId?: string;
    isAnalyticsConfigured: boolean;
    isFavoritePage: boolean;
    isLoading: boolean;
    isSearchOrFilterApplied: boolean;
    onFavoriteDashboard: (id: string, isCurrentFavorite: boolean) => void;
    updatingDashboardIds: string[];
}

export const AnalyticsCards = ({
    cards,
    folderId,
    isAnalyticsConfigured,
    isFavoritePage,
    isLoading,
    isSearchOrFilterApplied,
    onFavoriteDashboard,
    updatingDashboardIds,
}: AnalyticsCardsProps) => {
    const [isDialogOpened, setDialogOpened] = useState(false);
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore pendo is global variable
    const isPendoEnabled = typeof pendo !== 'undefined' && pendo.isSendingEvents();

    const handleFavoriteClick = (id: string, isCurrentFavorite: boolean) => () => {
        onFavoriteDashboard(id, isCurrentFavorite);
    };

    const handleOpenDashboardClick = (id: string) => () => {
        if (isAnalyticsConfigured) {
            const analyticsUrl = `analytics/${id}`;
            if (folderId) {
                window.location.href = `#/folders/${folderId}/${analyticsUrl}`;
            } else {
                window.location.href = `#/${analyticsUrl}`;
            }
        } else if (isPendoEnabled) {
            // pendo guide will be open
        } else {
            setDialogOpened(true);
        }
    };

    const renderEmptyState = () => {
        const emptyStateClassName = 'analytics-empty-state';
        const emptyStateTitleAndSubtitle = calculateTitleAndSubtitleForEmptyState(isFavoritePage, isAnalyticsConfigured);
        const emptyStateButtonProps: ButtonProps | undefined =
            isAnalyticsConfigured && !isFavoritePage
                ? {
                      children: 'Manage dashboards',
                      endIcon: <DotIcon iconId="open-new-tab" />,
                  }
                : undefined;
        const emptyStateImage = isAnalyticsConfigured ? analyticsConfiguredEmptyStateImg : analyticsDefaultEmptyStateImg;

        return isSearchOrFilterApplied ? (
            <DotEmptyFilterState className={emptyStateClassName} />
        ) : (
            <DotEmptyState
                buttonProps={emptyStateButtonProps}
                className={emptyStateClassName}
                illustrationId={isFavoritePage ? analyticsFavouritesEmptyStateIllustration : undefined}
                imageAltText="analytics dashboards empty state image"
                imageSrc={!isFavoritePage ? emptyStateImage : undefined}
                subtitle={emptyStateTitleAndSubtitle.subtitle}
                title={emptyStateTitleAndSubtitle.title}
            />
        );
    };

    return (
        <div className="analytics-cards">
            {cards.length > 0 ? (
                <>
                    <CssGrid className="css-grid">
                        {cards.map((currentAnalyticsCard, index) => {
                            return (
                                <CssCell {...calculateCellProps(index)} key={currentAnalyticsCard.id}>
                                    <AnalyticsCard
                                        {...currentAnalyticsCard}
                                        isAnalyticsConfigured={isAnalyticsConfigured}
                                        isFavorite={currentAnalyticsCard.isFavorite}
                                        isUpdating={updatingDashboardIds.includes(currentAnalyticsCard.id)}
                                        onFavoriteClick={handleFavoriteClick(currentAnalyticsCard.id, currentAnalyticsCard.isFavorite)}
                                        onOpenDashboardClick={handleOpenDashboardClick(currentAnalyticsCard.id)}
                                    />
                                </CssCell>
                            );
                        })}
                        {isLoading &&
                            [...Array(NUMBER_OF_LOADING_CARD_SKELETONS)].map((_value, skeletonIndex) => {
                                const props = calculateCellProps(cards.length + skeletonIndex);
                                return (
                                    <CssCell {...props} key={cards.length + skeletonIndex}>
                                        <AnalyticsCardSkeleton />
                                    </CssCell>
                                );
                            })}
                    </CssGrid>
                </>
            ) : (
                !isLoading && renderEmptyState()
            )}
            {isDialogOpened && (
                <DotDialog
                    cancelButtonVisible={false}
                    closeIconVisible={true}
                    closeOnClickAway={false}
                    onSubmit={() => setDialogOpened(false)}
                    open={isDialogOpened}
                    submitButtonProps={{ label: 'Close' }}
                    title="Content not available"
                >
                    <DotTypography>
                        You are seeing this message because your Release product setup has disabled guides and in-product analytics, or you are in an air-gapped
                        environment. Please contact your Release administrator for further assistance.
                    </DotTypography>
                </DotDialog>
            )}
        </div>
    );
};
